package ai.accurat.sdk.core;

import android.content.Context;

import androidx.annotation.NonNull;
import androidx.work.Worker;
import androidx.work.WorkerParameters;

import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;

import ai.accurat.sdk.config.Configuration;
import ai.accurat.sdk.managers.RealmManager;

public class AccuratGeofenceNotificationsWorker extends Worker {

    private static final String TAG = AccuratGeofenceNotificationsWorker.class.getSimpleName();

    public AccuratGeofenceNotificationsWorker(@NonNull Context context, @NonNull WorkerParameters workerParameters) {
        super(context, workerParameters);
    }

    @NonNull
    @Override
    public Result doWork() {
        AccuratLogger.init(getApplicationContext());
        AccuratLogger.log(AccuratLogger.WORKMANAGER, TAG + ".doWork()");
        RealmManager.init(getApplicationContext());
        // If we wouldn't use an array, we wouldn't be able to set another result value.
        final Result[] result = {Result.retry()};
        GeofenceNotificationManager.init(getApplicationContext());

        // The doWork()-method requires a synchronous return.
        // However, making network requests is usually async. We could make a sync request, but any
        // exceptions will be thrown on the main thread, not on this worker thread. To handle that
        // situation, we would need to set a timeout here, but the timeout might be shorter than the
        // network response time in random cases, which would cause us to miss the response.
        // Instead we will use a CountDownLatch to block this worker thread and wait for all
        // responses of the network thread.
        final CountDownLatch countDownLatch = new CountDownLatch(1);

        GeofenceNotificationManager.getInstance().fetchNotifications(success -> {
            result[0] = success ? Result.success() : Result.failure();
            countDownLatch.countDown();
        });

        try {
            // Await the network response.
            countDownLatch.await(Configuration.DEFAULT_TIMEOUT_MS, TimeUnit.MILLISECONDS);
        } catch (InterruptedException e) {
            AccuratLogger.log(AccuratLogger.JSON_ERROR, TAG + ".doWork(): " + e.getMessage());
        }

        AccuratLogger.log(AccuratLogger.WORKMANAGER, TAG + " - Work done, returning result[0]");
        return result[0];
    }

    @Override
    public void onStopped() {
        GeofenceNotificationManager.init(getApplicationContext());
        GeofenceNotificationManager.cancelNetworkRequests();
    }
}
