package ai.accurat.sdk.core;

import org.json.JSONException;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.List;

import ai.accurat.sdk.config.Configuration;
import ai.accurat.sdk.constants.Manufacturer;
import ai.accurat.sdk.constants.StorageKeys;

/**
 * @author Kenneth Saey
 * @Accurat
 * @since 02-07-2018 11:14.
 */
public class AccuratSettings {

    // <editor-fold desc="Fields">
    private static final String TAG = AccuratSettings.class.getSimpleName();
    private static final Manufacturer manufacturer = Manufacturer.fromDevice();
    private boolean sdkEnabled;
    private boolean geofencingEnabled = true;

    private List<OpenLocateBasedEndpoint> endpoints = new ArrayList<>();

    private long locationInterval;// seconds
    private long fastestLocationInterval;// seconds
    private long transmissionIntervalInSecs;// seconds
    private long maxWaitTime;// milliseconds
    private long smallestDisplacement;// meters
    private long locationRestartTimeout;// seconds
    private int currentGeofenceRadius;// meters

    private int maxNotificationsPerDay;
    private int maxConsentRefuseCount = Configuration.MAX_CONSENT_REFUSE_COUNT;
    private int maxPermissionRefuseCount = Configuration.MAX_PERMISSION_REFUSE_COUNT;

    private boolean loggingEnabled = Configuration.DEFAULT_LOGGING_ENABLED;
    // </editor-fold>

    // <editor-fold desc="Construction">
    public AccuratSettings() {
        endpoints = new ArrayList<>();
    }

    public AccuratSettings(
            boolean sdkEnabled,
            boolean geofencingEnabled,
            long locationInterval,
            long fastestLocationInterval,
            long transmissionIntervalInSecs,
            long maxWaitTime,
            long smallestDisplacement,
            long locationRestartTimeout,
            int currentGeofenceRadius,
            int maxNotificationsPerDay,
            int maxConsentRefuseCount,
            int maxPermissionRefuseCount
    ) {
        this();
        this.sdkEnabled = sdkEnabled;
        this.geofencingEnabled = geofencingEnabled;
        this.locationInterval = locationInterval;
        this.fastestLocationInterval = fastestLocationInterval;
        this.transmissionIntervalInSecs = transmissionIntervalInSecs;
        this.maxWaitTime = maxWaitTime;
        this.smallestDisplacement = smallestDisplacement;
        this.locationRestartTimeout = locationRestartTimeout;
        this.currentGeofenceRadius = currentGeofenceRadius;
        this.maxNotificationsPerDay = maxNotificationsPerDay;
        this.maxConsentRefuseCount = maxConsentRefuseCount;
        this.maxPermissionRefuseCount = maxPermissionRefuseCount;
    }
    // </editor-fold>

    // <editor-fold desc="Factory">
    public static AccuratSettings getDefaultSettings() {
        return new AccuratSettings(
                Configuration.DEFAULT_SDK_ENABLED,
                Configuration.DEFAULT_GEOFENCING_ENABLED,
                Configuration.DEFAULT_LOCATION_INTERVAL,
                Configuration.DEFAULT_FASTEST_LOCATION_INTERVAL,
                Configuration.DEFAULT_TRANSMISSION_INTERVAL_IN_SECS,
                Configuration.DEFAULT_MAX_WAIT_TIME,
                Configuration.DEFAULT_SMALLEST_DISPLACEMENT,
                Configuration.DEFAULT_LOCATION_RESTART_TIMEOUT,
                Configuration.DEFAULT_CURRENT_GEOFENCE_RADIUS,
                Configuration.DEFAULT_MAX_NOTIFICATIONS_PER_DAY,
                Configuration.MAX_CONSENT_REFUSE_COUNT,
                Configuration.MAX_PERMISSION_REFUSE_COUNT
        );
    }
    // </editor-fold>

    // <editor-fold desc="JSON Handling">
    public static AccuratSettings fromJson(String jsonString) throws JSONException {
        return fromJson(new JSONObject(jsonString));
    }

    public static AccuratSettings fromJson(JSONObject json) throws JSONException {
        AccuratSettings accuratSettings = new AccuratSettings();

        accuratSettings.endpoints = json.has(StorageKeys.ACCURAT_ENDPOINTS) ? OpenLocateBasedEndpoint.fromJson(json.getString(StorageKeys.ACCURAT_ENDPOINTS)) : accuratSettings.endpoints;

        accuratSettings.sdkEnabled = json.has(StorageKeys.ACCURAT_SDK_ENABLED) ? json.getBoolean(StorageKeys.ACCURAT_SDK_ENABLED) : Configuration.DEFAULT_SDK_ENABLED;
        accuratSettings.geofencingEnabled = json.has(StorageKeys.ACCURAT_GEOFENCING_ENABLED) ? json.getBoolean(StorageKeys.ACCURAT_GEOFENCING_ENABLED) : Configuration.DEFAULT_GEOFENCING_ENABLED;
        accuratSettings.locationInterval = json.has(StorageKeys.ACCURAT_LOCATION_INTERVAL) ? json.getLong(StorageKeys.ACCURAT_LOCATION_INTERVAL) : Configuration.DEFAULT_LOCATION_INTERVAL;
        accuratSettings.fastestLocationInterval = json.has(StorageKeys.ACCURAT_FASTEST_LOCATION_INTERVAL) ? json.getLong(StorageKeys.ACCURAT_FASTEST_LOCATION_INTERVAL) : Configuration.DEFAULT_FASTEST_LOCATION_INTERVAL;
        accuratSettings.transmissionIntervalInSecs = json.has(StorageKeys.ACCURAT_TRANSMISSION_INTERVAL) ? json.getLong(StorageKeys.ACCURAT_TRANSMISSION_INTERVAL) : Configuration.DEFAULT_TRANSMISSION_INTERVAL_IN_SECS;
        accuratSettings.maxWaitTime = json.has(StorageKeys.ACCURAT_MAX_WAIT_TIME) ? json.getLong(StorageKeys.ACCURAT_MAX_WAIT_TIME) : Configuration.DEFAULT_MAX_WAIT_TIME * 1000;// seconds => milliseconds
        accuratSettings.smallestDisplacement = json.has(StorageKeys.ACCURAT_SMALLEST_DISPLACEMENT) ? json.getLong(StorageKeys.ACCURAT_SMALLEST_DISPLACEMENT) : Configuration.DEFAULT_SMALLEST_DISPLACEMENT;
        accuratSettings.locationRestartTimeout = json.has(StorageKeys.ACCURAT_LOCATION_RESTART_TIMEOUT) ? json.getLong(StorageKeys.ACCURAT_LOCATION_RESTART_TIMEOUT) : Configuration.DEFAULT_LOCATION_RESTART_TIMEOUT;
        accuratSettings.currentGeofenceRadius = json.has(StorageKeys.ACCURAT_CURRENT_GEOFENCE_RADIUS) ? json.getInt(StorageKeys.ACCURAT_CURRENT_GEOFENCE_RADIUS) : Configuration.DEFAULT_CURRENT_GEOFENCE_RADIUS;
        accuratSettings.maxNotificationsPerDay = json.has(StorageKeys.ACCURAT_MAX_NOTIFICATIONS_PER_DAY) ? json.getInt(StorageKeys.ACCURAT_MAX_NOTIFICATIONS_PER_DAY) : Configuration.DEFAULT_MAX_NOTIFICATIONS_PER_DAY;
        accuratSettings.maxConsentRefuseCount = json.has(StorageKeys.ACCURAT_MAX_CONSENT_REFUSE_COUNT) ? json.getInt(StorageKeys.ACCURAT_MAX_CONSENT_REFUSE_COUNT) : Configuration.MAX_CONSENT_REFUSE_COUNT;
        accuratSettings.maxPermissionRefuseCount = json.has(StorageKeys.ACCURAT_MAX_PERMISSION_REFUSE_COUNT) ? json.getInt(StorageKeys.ACCURAT_MAX_PERMISSION_REFUSE_COUNT) : Configuration.MAX_PERMISSION_REFUSE_COUNT;

        accuratSettings.loggingEnabled = json.has(StorageKeys.ACCURAT_LOGGING_ENABLED) ? json.getBoolean(StorageKeys.ACCURAT_LOGGING_ENABLED) : accuratSettings.loggingEnabled;

        return accuratSettings;
    }

    public static AccuratSettings fromServerJson(JSONObject json) throws JSONException {
        AccuratSettings accuratSettings = getDefaultSettings();

        AccuratLogger.log(AccuratLogger.NONE, "Using " + (isSamsung() ? "Samsung" : "default") + " settings from server.");

        accuratSettings.sdkEnabled = getJsonBoolean(json, "tracking", accuratSettings.sdkEnabled);
        accuratSettings.geofencingEnabled = getJsonBoolean(json, "geofence", accuratSettings.geofencingEnabled);
        accuratSettings.locationInterval = getJsonLong(json, "location_interval", accuratSettings.locationInterval / 60) * 60;// minutes => seconds
        accuratSettings.fastestLocationInterval = getJsonLong(json, "fastest_location_interval", accuratSettings.fastestLocationInterval / 60) * 60;// minutes => seconds
        accuratSettings.transmissionIntervalInSecs = getJsonLong(json, "transmission_interval", accuratSettings.transmissionIntervalInSecs / 60) * 60;// minutes => seconds
        accuratSettings.maxWaitTime = getJsonLong(json, "maximum_wait_time", accuratSettings.maxWaitTime / 60000) * 60000;// minutes => milliseconds
        accuratSettings.smallestDisplacement = getJsonLong(json, "small_displacement", accuratSettings.smallestDisplacement);// Already in meters
        accuratSettings.locationRestartTimeout = getJsonLong(json, "location_restart_timeout", accuratSettings.locationRestartTimeout / 60) * 60;// minutes => seconds
        accuratSettings.currentGeofenceRadius = getJsonInt(json, "current_geofence_radius", accuratSettings.currentGeofenceRadius);// Already in meters
        accuratSettings.maxNotificationsPerDay = getJsonInt(json, "max_notifications_per_consumer", accuratSettings.maxNotificationsPerDay);
        accuratSettings.maxConsentRefuseCount = getJsonInt(json, "screen_gdpr_repeat", accuratSettings.maxConsentRefuseCount);
        accuratSettings.maxPermissionRefuseCount = getJsonInt(json, "screen_always_repeat", accuratSettings.maxPermissionRefuseCount);

        return accuratSettings;
    }

    public JSONObject getJson() {
        JSONObject json = new JSONObject();
        try {
            json.put(StorageKeys.ACCURAT_ENDPOINTS, OpenLocateBasedEndpoint.toJson(endpoints));

            json.put(StorageKeys.ACCURAT_SDK_ENABLED, sdkEnabled);
            json.put(StorageKeys.ACCURAT_GEOFENCING_ENABLED, geofencingEnabled);
            json.put(StorageKeys.ACCURAT_LOCATION_INTERVAL, locationInterval);
            json.put(StorageKeys.ACCURAT_FASTEST_LOCATION_INTERVAL, fastestLocationInterval);
            json.put(StorageKeys.ACCURAT_TRANSMISSION_INTERVAL, transmissionIntervalInSecs);
            json.put(StorageKeys.ACCURAT_MAX_WAIT_TIME, maxWaitTime);
            json.put(StorageKeys.ACCURAT_SMALLEST_DISPLACEMENT, smallestDisplacement);
            json.put(StorageKeys.ACCURAT_LOCATION_RESTART_TIMEOUT, locationRestartTimeout);
            json.put(StorageKeys.ACCURAT_CURRENT_GEOFENCE_RADIUS, currentGeofenceRadius);
            json.put(StorageKeys.ACCURAT_MAX_NOTIFICATIONS_PER_DAY, maxNotificationsPerDay);
            json.put(StorageKeys.ACCURAT_MAX_CONSENT_REFUSE_COUNT, maxConsentRefuseCount);
            json.put(StorageKeys.ACCURAT_MAX_PERMISSION_REFUSE_COUNT, maxPermissionRefuseCount);

            json.put(StorageKeys.ACCURAT_LOGGING_ENABLED, loggingEnabled);
        } catch (JSONException e) {
            AccuratLogger.log(AccuratLogger.JSON_ERROR, TAG + ".getJson(): " + e.getMessage());
        }

        return json;
    }

    @Override
    public String toString() {
        return getJson().toString();
    }

    // </editor-fold>

    // <editor-fold desc="Getters">

    public List<OpenLocateBasedEndpoint> getEndpoints() {
        return endpoints;
    }

    public boolean isSdkEnabled() {
        return sdkEnabled;
    }

    public boolean isGeofencingEnabled() {
        return geofencingEnabled;
    }

    public long getLocationInterval() {
        return locationInterval;
    }

    public long getFastestLocationInterval() {
        return fastestLocationInterval;
    }

    public long getTransmissionIntervalInSecs() {
        return transmissionIntervalInSecs;
    }

    public long getMaxWaitTime() {
        return maxWaitTime;
    }

    public long getSmallestDisplacement() {
        return smallestDisplacement;
    }

    public long getLocationRestartTimeout() {
        return locationRestartTimeout;
    }

    public boolean isLoggingEnabled() {
        return loggingEnabled;
    }

    public int getCurrentGeofenceRadius() {
        return currentGeofenceRadius;
    }

    public int getMaxNotificationsPerDay() {
        return maxNotificationsPerDay;
    }

    public int getMaxConsentRefuseCount() {
        AccuratLogger.log(AccuratLogger.NONE, "Using server maxConsentRefuseCount");
        return maxConsentRefuseCount;
    }

    public int getMaxPermissionRefuseCount() {
        AccuratLogger.log(AccuratLogger.NONE, "Using server maxPermissionRefuseCount");
        return maxPermissionRefuseCount;
    }
    // </editor-fold>

    public void addDefaultEndpoint() {
        OpenLocateBasedEndpoint defaultEndpoint = OpenLocateBasedEndpoint.builder(Configuration.DEFAULT_ENDPOINT_URL)
                .build();
        endpoints.add(defaultEndpoint);
    }

    // <editor-fold desc="Helpers">
    private static boolean isSamsung() {
        return Manufacturer.SAMSUNG == manufacturer;
    }

    private static boolean getJsonBoolean(JSONObject json, String name, boolean defaultValue) {
        if (isSamsung() && json.has(Manufacturer.SAMSUNG.prefix + name)) {
            return json.optBoolean(Manufacturer.SAMSUNG.prefix + name, defaultValue);
        }

        return json.optBoolean(name, defaultValue);
    }

    private static int getJsonInt(JSONObject json, String name, int defaultValue) {
        if (isSamsung() && json.has(Manufacturer.SAMSUNG.prefix + name)) {
            return json.optInt(Manufacturer.SAMSUNG.prefix + name, defaultValue);
        }

        return json.optInt(name, defaultValue);
    }

    private static long getJsonLong(JSONObject json, String name, long defaultValue) {
        if (isSamsung() && json.has(Manufacturer.SAMSUNG.prefix + name)) {
            return json.optLong(Manufacturer.SAMSUNG.prefix + name, defaultValue);
        }

        return json.optLong(name, defaultValue);
    }
    // </editor-fold>

}
