package ai.accurat.sdk.core;

import android.content.Context;

import com.google.android.gms.ads.identifier.AdvertisingIdClient;
import com.google.android.gms.common.GooglePlayServicesNotAvailableException;
import com.google.android.gms.common.GooglePlayServicesRepairableException;

import org.json.JSONException;
import org.json.JSONObject;

import java.io.IOException;

import ai.accurat.sdk.callbacks.AccuratCompletionCallback;
import ai.accurat.sdk.constants.StorageKeys;
import ai.accurat.sdk.data.models.ConsentManagerState;
import ai.accurat.sdk.managers.AccuratConfigurationManager;
import ai.accurat.sdk.managers.RealmManager;
import io.reactivex.Single;
import io.reactivex.android.schedulers.AndroidSchedulers;
import io.reactivex.disposables.CompositeDisposable;
import io.reactivex.schedulers.Schedulers;

/**
 * @author Kenneth Saey
 * @Accurat
 * @since 03-07-2018 15:09.
 */
public class AdvertisingManager {

    // <editor-fold desc="Constants">
    private static final String TAG = AdvertisingManager.class.getSimpleName();

    private static final String JSON_KEY_ID = "id";
    private static final String JSON_KEY_LIMIT_AD_TRACKING = "limit_ad_tracking";
    // </editor-fold>

    // <editor-fold desc="Fields">
    private static final CompositeDisposable compositeDisposable;
    private static MultiProcessStorage storage;

    private static AdvertisingIdClient.Info advertisingInfo;
    // </editor-fold>

    // <editor-fold desc="Initialisation">
    static {
        compositeDisposable = new CompositeDisposable();
    }

    public static void init(Context context) {
        if (!isInitialized()) {
            AccuratLogger.init(context);
            AccuratLogger.log(AccuratLogger.SDK_FLOW, "Initialising " + TAG);
            storage = MultiProcessStorage.getStorage(context, StorageKeys.ACCURAT_MULTI_PROCESS_STORAGE);
        }
    }

    private static boolean isInitialized() {
        return storage != null;
    }

    private static void checkInitialized() {
        if (!isInitialized()) {
            throw new IllegalStateException("AdvertisingManager has not yet been initialised.");
        }
    }
    // </editor-fold>

    // <editor-fold desc="Storage">
    private static void load() {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".load()");
        checkInitialized();

        String jsonString = storage.getString(StorageKeys.ACCURAT_ADVERTISING_INFO, "");
        if (jsonString == null || jsonString.isEmpty()) {
            advertisingInfo = null;
            return;
        }

        try {
            JSONObject json = new JSONObject(jsonString);

            String adId = json.getString(JSON_KEY_ID);
            boolean limitAdTracking = json.getBoolean(JSON_KEY_LIMIT_AD_TRACKING);
            AccuratLogger.log(AccuratLogger.ADVERTISING, "Ad ID = " + adId + ", ad tracking is " + (limitAdTracking ? "" : "not ") + "limited");

            advertisingInfo = new AdvertisingIdClient.Info(adId, limitAdTracking);
        } catch (JSONException e) {
            advertisingInfo = null;
            AccuratLogger.log(AccuratLogger.JSON_ERROR, "Failed to load advertising info: " + e.getMessage());
        }
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".load()");
    }

    private static void clear() {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".clear()");
        checkInitialized();
        advertisingInfo = null;
        storage.remove(StorageKeys.ACCURAT_ADVERTISING_INFO)
                .commit();
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".clear()");
    }

    private static void store(AdvertisingIdClient.Info advertisingInfo) {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".store()");
        checkInitialized();
        if (advertisingInfo == null) {
            clear();
            AccuratLogger.log(AccuratLogger.SDK_FLOW, "Removing advertising info from storage because it is NULL");
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".store()");

            return;
        }
        JSONObject json = new JSONObject();
        try {
            json.put(JSON_KEY_ID, advertisingInfo.getId());
            json.put(JSON_KEY_LIMIT_AD_TRACKING, advertisingInfo.isLimitAdTrackingEnabled());

            storage.setValue(StorageKeys.ACCURAT_ADVERTISING_INFO, json.toString())
                    .commit();
        } catch (JSONException e) {
            AccuratLogger.log(AccuratLogger.JSON_ERROR, "Failed to store advertising info: " + e.getMessage());
            e.printStackTrace();
        }
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".store()");
    }
    // </editor-fold>

    // <editor-fold desc="Public interface">
    public static void start(final Context context, final AccuratCompletionCallback onCompleted) {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".start()");
        load();

        compositeDisposable.add(Single.just(context)
                .observeOn(Schedulers.io())
                .map(ctxt -> {
                    AccuratLogger.log(AccuratLogger.SDK_FLOW, "Getting adversing info");
                    AdvertisingIdClient.Info info = AdvertisingIdClient.getAdvertisingIdInfo(ctxt);
                    if (info == null) {
                        AccuratLogger.log(AccuratLogger.ERROR, "Advertising info is null");
                        throw new NullPointerException("Advertising info object received by the AdvertisingIdClient is null.");
                    }
                    if (info.isLimitAdTrackingEnabled()) {
                        AccuratLogger.log(AccuratLogger.WARNING, "Ad tracking is limited");
                    }

                    return info;
                })
                .doOnError(throwable -> {
                    if (throwable instanceof NullPointerException
                            || throwable instanceof IOException
                            || throwable instanceof GooglePlayServicesNotAvailableException
                            || throwable instanceof GooglePlayServicesRepairableException) {
                        AccuratLogger.log(AccuratLogger.ERROR, "Failed to get ad ID: " + throwable.getMessage());
                    }
                    onCompleted.onCompleted(false);
                })
                .doOnSuccess(AdvertisingManager::store)
                .observeOn(AndroidSchedulers.mainThread())
                .subscribeOn(Schedulers.io())
                .subscribe((info, throwable) -> {
                    if (throwable != null) {
                        AccuratLogger.log(AccuratLogger.ERROR, "Failed to get ad ID: " + throwable.getMessage());
                        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".start()");

                        return;
                    }

                    if (info != null) {
                        AccuratLogger.log(AccuratLogger.SDK_FLOW, "Fetched advertising info with ad ID = " + info.getId() + " and " + (info.isLimitAdTrackingEnabled() ? "" : "un") + "limited ad tracking");
                    } else {
                        AccuratLogger.log(AccuratLogger.SDK_FLOW, "Fetched advertising info, but it is null");
                    }

                    ConsentManager.init(context);
                    RealmManager.init(context);
                    if (!hasValidId(info)) {
                        AccuratLogger.log(AccuratLogger.WARNING, "Ad ID is invalid");
                        clear();

                        if (AccuratConfigurationManager.hasGdprConsentFeature()) {
                            ConsentManager.clearGdprConsent();
                        }
                        if (AccuratConfigurationManager.hasLocationPermissionFeature()) {
                            ConsentManager.clearLocationPermission();
                        }

                        respond(onCompleted, false);
                        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".start()");

                        return;
                    }

                    String adId = info.getId();
                    if (advertisingInfo == null
                            || AdvertisingManager.isAdTrackingLimited() != info.isLimitAdTrackingEnabled()
                            || !adId.equals(AdvertisingManager.getAdId())) {
                        AccuratLogger.log(AccuratLogger.SDK_FLOW, "Ad ID is new or limit tracking has changed");
                        AccuratLogger.log(AccuratLogger.NONE, "advertisingInfo = " + advertisingInfo);
                        AccuratLogger.log(AccuratLogger.NONE, AdvertisingManager.isAdTrackingLimited() + (AdvertisingManager.isAdTrackingLimited() == info.isLimitAdTrackingEnabled() ? " == " : " != ") + info.isLimitAdTrackingEnabled());
                        AccuratLogger.log(AccuratLogger.NONE, adId + (adId.equals(AdvertisingManager.getAdId()) ? " == " : "!= ") + AdvertisingManager.getAdId());
                        advertisingInfo = info;

                        if (AccuratConfigurationManager.hasGdprConsentFeature()) {
                            ConsentManager.clearGdprConsent();
                        }
                        if (AccuratConfigurationManager.hasLocationPermissionFeature()) {
                            ConsentManager.clearLocationPermission();
                        }
                    }
                    AccuratUserManager.init(context);
                    AccuratUserManager.upload();

                    // Should never happen
                    if (advertisingInfo == null) {
                        AccuratLogger.log(AccuratLogger.ERROR, "Advertising info is NULL");
                        respond(onCompleted, false);
                        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".start()");

                        return;
                    }

                    respond(onCompleted, !advertisingInfo.isLimitAdTrackingEnabled());
                    AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".start()");
                }));
    }

    public static void startFromBackground(Context context, AccuratCompletionCallback onCompleted) {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".startFromBackground()");
        load();

        if (!hasValidId(AdvertisingManager.advertisingInfo)) {
            AccuratLogger.log(AccuratLogger.WARNING, "User has invalid ad ID");
            respond(onCompleted, false);
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".startFromBackground()");

            return;
        }

        if (AdvertisingManager.advertisingInfo != null) {
            AccuratLogger.log(AccuratLogger.SDK_FLOW, "Ad tracking is " + (AdvertisingManager.advertisingInfo.isLimitAdTrackingEnabled() ? "" : "not ") + "limited");
            respond(onCompleted, !AdvertisingManager.advertisingInfo.isLimitAdTrackingEnabled());
        } else {
            AccuratLogger.log(AccuratLogger.SDK_FLOW, "Ad tracking is unknown (AdvertisingManager.advertisingInfo is null)");
            respond(onCompleted, false);
        }
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".startFromBackground()");
    }
    // </editor-fold>

    // <editor-fold desc="Helpers">

    /**
     * Verifies whether the user has allowed us to track their location.
     * The user can disallow us from tracking their location by opting out of personalised ads (limited ad tracking).
     *
     * @return True when the ad id may be used to track the location of the user, false if not.
     */
    private static boolean hasValidId(AdvertisingIdClient.Info advertisingInfo) {
        return advertisingInfo != null
                && advertisingInfo.getId() != null
                && !advertisingInfo.getId().isEmpty()
                && !advertisingInfo.isLimitAdTrackingEnabled();
    }

    private static void respond(AccuratCompletionCallback onComplete, boolean success) {
        if (onComplete != null) {
            onComplete.onCompleted(success);
        }
    }
    // </editor-fold>

    // <editor-fold desc="Getters">
    public static String getAdId() {
        return ConsentManagerState.Companion.getAdId();
        /*if (advertisingInfo == null) {
            load();
        }

        return advertisingInfo == null ? null : advertisingInfo.getId();*/
    }

    public static boolean isAdTrackingLimited() {
        return !ConsentManagerState.Companion.load().getAdIdTrackingAllowed();
//        return advertisingInfo != null && advertisingInfo.isLimitAdTrackingEnabled();
    }

    public static AdvertisingIdClient.Info getAdvertisingInfo() {
        if (advertisingInfo == null) {
            load();
        }

        return advertisingInfo;
    }

    public static boolean hasValidAdId() {
        return ConsentManagerState.Companion.hasValidAdId();
//        return hasValidId(advertisingInfo);
    }
    // </editor-fold>
}
