package ai.accurat.sdk.core;

import org.json.JSONException;
import org.json.JSONObject;

import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;
import java.util.Locale;
import java.util.TimeZone;

import io.realm.RealmObject;

public class CampaignInteraction extends RealmObject implements JSONConvertable {

    // <editor-fold desc="Constants">
    private static final String TAG = CampaignInteraction.class.getSimpleName();
    private static final SimpleDateFormat FORMAT_TIMESTAMP = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss.SSS'Z'", Locale.UK);
    // </editor-fold>

    // <editor-fold desc="Properties">
    protected String campaign;
    protected String touchPoint;
    protected String group;
    protected String id;
    protected String start;
    protected String end;
    protected long timestamp;
    // </editor-fold>

    // <editor-fold desc="Initialisation">
    public CampaignInteraction() {
        timestamp = Calendar.getInstance().getTimeInMillis();
    }

    public CampaignInteraction(String campaign, String touchPoint, String group) {
        this(campaign, touchPoint, group, Calendar.getInstance().getTimeInMillis());
    }

    public CampaignInteraction(String campaign, String touchPoint, String group, long timestamp) {
        this.campaign = campaign;
        this.touchPoint = touchPoint;
        this.group = group;
        this.timestamp = timestamp;
    }
    // </editor-fold>

    // <editor-fold desc="Builder">
    public CampaignInteraction withCampaign(String campaign) {
        this.campaign = campaign;

        return this;
    }

    public CampaignInteraction withGroup(String group) {
        this.group = group;

        return this;
    }

    public CampaignInteraction withTouchPoint(String touchPoint) {
        this.touchPoint = touchPoint;

        return this;
    }

    /**
     * The Campaign's unique ID.
     *
     * @param id The ID
     * @return this
     */
    public CampaignInteraction withId(String id) {
        this.id = id;

        return this;
    }

    /**
     * The start date of the Campaign, formatted as yyyy-mm-dd.
     *
     * @param date The start date
     * @return this
     */
    public CampaignInteraction startsAt(String date) {
        this.start = date;

        return this;
    }

    /**
     * The end date of the Campaign, formatted as yyyy-mm-dd.
     *
     * @param date The end date
     * @return this
     */
    public CampaignInteraction endsAt(String date) {
        this.end = date;

        return this;
    }
    // </editor-fold>

    // <editor-fold desc="JSON Handling">
    @Override
    public JSONObject toServerJson() {
        FORMAT_TIMESTAMP.setTimeZone(TimeZone.getTimeZone("UTC"));
        JSONObject json = new JSONObject();
        try {
            json.put(StorageKeys.CAMPAIGN, campaign);
            json.put(StorageKeys.TOUCH_POINT, touchPoint);
            json.put(StorageKeys.GROUP, group);
            json.put(StorageKeys.ID, id);
            json.put(StorageKeys.START, start);
            json.put(StorageKeys.END, end);
            json.put(StorageKeys.TIMESTAMP, FORMAT_TIMESTAMP.format(new Date(timestamp)));
        } catch (JSONException e) {
            AccuratLogger.log(AccuratLogger.JSON_ERROR, TAG + ".getJson(): " + e.getMessage());
            e.printStackTrace();
        }

        return json;
    }

    @Override
    public JSONObject toJson() {
        return toServerJson();
    }

    // </editor-fold>

    public static class StorageKeys {
        public static final String CAMPAIGN = "campaign";
        public static final String TOUCH_POINT = "touchpoint";
        public static final String GROUP = "group";
        public static final String ID = "campaign_id";
        public static final String START = "campaign_start";
        public static final String END = "campaign_end";
        public static final String TIMESTAMP = "utc_timestamp";
    }

    public static class RealmKeys {
        public static final String _TABLE = "CampaignInteraction";

        public static final String CAMPAIGN = "campaign";
        public static final String TOUCH_POINT = "touchPoint";
        public static final String GROUP = "group";
        public static final String ID = "id";
        public static final String START = "start";
        public static final String END = "end";
        public static final String TIMESTAMP = "timestamp";
    }
}
