package ai.accurat.sdk.core;

import android.app.Activity;
import android.app.AlertDialog;
import android.content.Context;
import android.content.pm.PackageManager;
import android.os.Build;
import android.util.SparseArray;
import android.view.LayoutInflater;

import androidx.annotation.NonNull;
import androidx.core.app.ActivityCompat;

import com.android.volley.Request;
import com.android.volley.RequestQueue;
import com.android.volley.toolbox.JsonObjectRequest;
import com.android.volley.toolbox.Volley;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.concurrent.CountDownLatch;

import ai.accurat.sdk.Accurat;
import ai.accurat.sdk.R;
import ai.accurat.sdk.callbacks.AccuratCompletionCallback;
import ai.accurat.sdk.config.Configuration;
import ai.accurat.sdk.constants.AccuratEndpoints;
import ai.accurat.sdk.constants.AllConsentsState;
import ai.accurat.sdk.constants.ApiKeys;
import ai.accurat.sdk.constants.ConsentState;
import ai.accurat.sdk.constants.ConsentType;
import ai.accurat.sdk.constants.HttpMethod;
import ai.accurat.sdk.constants.StorageKeys;
import ai.accurat.sdk.data.RealmHelper;
import ai.accurat.sdk.data.models.AccuratConfiguration;
import ai.accurat.sdk.data.models.Setting;
import ai.accurat.sdk.managers.AccuratConfigurationManager;
import ai.accurat.sdk.managers.AccuratManager;
import ai.accurat.sdk.managers.RealmManager;
import ai.accurat.sdk.managers.VersionManager;
import ai.accurat.sdk.viewholders.ConsentDialogViewHolder;

/**
 * @author Kenneth Saey
 * @Accurat
 * @since 22-06-2018 09:28.
 */
public class ConsentManager {

    // <editor-fold desc="Fields">
    private static final String TAG = ConsentManager.class.getSimpleName();
    private static final ConsentManager INSTANCE = new ConsentManager();

    private static MultiProcessStorage mStorage;
    private static HashMap<ConsentType, Consent> mConsents;
    private static ConsentFlow mConsentFlow;

    private static RequestQueue requestQueue;
    // </editor-fold>

    // <editor-fold desc="Initialisation">
    public static void init(Context context) {
        if (!isInitialized()) {
            AccuratLogger.init(context);
            AccuratLogger.log(AccuratLogger.SDK_FLOW, "Initialising " + TAG);
            mStorage = MultiProcessStorage.getStorage(context, StorageKeys.ACCURAT_MULTI_PROCESS_STORAGE);
            requestQueue = Volley.newRequestQueue(context);
            AccuratImageLoader.init(context);
            AccuratSettingsManager.init(context);
        } else if (requestQueue == null) {
            requestQueue = Volley.newRequestQueue(context);
        }
    }

    private static boolean isInitialized() {
        return mStorage != null;
    }

    private static void checkInitialized() {
        if (!isInitialized()) {
            throw new IllegalStateException("ConsentManager has not yet been initialised.");
        }
    }
    // </editor-fold>

    // <editor-fold desc="Storage">
    private static void loadConsents() {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".loadConsents()");
        checkInitialized();

        String jsonString = AccuratConfigurationManager.loadConsentJson();
        if (jsonString == null || jsonString.isEmpty()) {
            AccuratLogger.log(AccuratLogger.SDK_FLOW, "No consents in storage");
            mConsents = new HashMap<>();

            return;
        }

        try {
            JSONArray json = new JSONArray(jsonString);
            AccuratLogger.log(AccuratLogger.SDK_FLOW, "Creating new HashMap for consents");
            mConsents = new HashMap<>();
            for (int i = 0; i < json.length(); i++) {
                Consent consent = Consent.fromJson(json.getString(i));
                if (consent != null) {
                    mConsents.put(consent.getType(), consent);
                }
            }
            AccuratLogger.log(AccuratLogger.SDK_FLOW, "Loaded " + mConsents.size() + " consents");
            AccuratLogger.log(AccuratLogger.STORAGE_DATA, "Consents: " + jsonString);
        } catch (JSONException e) {
            AccuratLogger.log(AccuratLogger.JSON_ERROR, "Failed to load consents: " + e.getMessage());
        }

        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".loadConsents()");
    }

    public static void clear() {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".clear()");
        checkInitialized();

        if (mConsents != null) {
            mConsents.clear();
        }

        AccuratConfigurationManager.storeConsentJson("");
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".clear()");
    }

    public static void clearGdprConsent() {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".clearGdprConsent()");
        checkInitialized();

        loadConsents();
        if (mConsents == null) {
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".clearGdprConsent()");
            return;
        }

        Consent consent = mConsents.get(ConsentType.TRACKING);
        if (consent == null) {
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".clearGdprConsent()");
            return;
        }

        mConsents.remove(ConsentType.TRACKING);

        storeConsents();
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".clearGdprConsent()");
    }

    public static void clearLocationPermission() {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".clearLocationPermission()");
        checkInitialized();

        loadConsents();
        if (mConsents == null) {
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".clearLocationPermission()");
            return;
        }

        Consent consent = mConsents.get(ConsentType.TRACKING);
        if (consent == null) {
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".clearLocationPermission()");
            return;
        }

        AccuratConfiguration config = AccuratConfigurationManager.load();
        if (config.usesGdprConsentFeature()) {
            mConsents.remove(ConsentType.TRACKING);
        } else {
            consent.setSystemPermissionState(ConsentState.UNKNOWN);
            consent.setSystemPermissionBackgroundState(ConsentState.UNKNOWN);
            consent.updatePermissionRefuseCount(0);
        }

        storeConsents();
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".clearLocationPermission()");
    }

    private static void fetchConsentModels(final Activity activity, final AccuratCompletionCallback onCompleted) {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".fetchConsentModels()");
        if (!AccuratApi.isNetworkAvailable(activity)) {
            AccuratLogger.log(AccuratLogger.NETWORK, "Network not available");
            fail(onCompleted);
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".fetchConsentModels()");

            return;
        }

        HashMap<String, Object> urlParameters = getConsentModelsFetchParams();
        JsonObjectRequest consentModelsRequest = new JsonObjectRequest(
                Request.Method.GET,
                AccuratEndpoints.GET_CONSENTS.getUrl(urlParameters),
                null,
                response -> {
                    AccuratLogger.logNetworkResponse(HttpMethod.GET, Configuration.ENDPOINT_GET_CONSENTS, response, false);
                    handleServerConsentModels(activity, response, onCompleted);
                },
                error -> {
                    AccuratLogger.logNetworkError(HttpMethod.GET, Configuration.ENDPOINT_GET_CONSENTS, error);
                    fail(onCompleted);
                }
        ) {
            @Override
            public Map<String, String> getHeaders() {
                return AccuratApi.getHeaders(mStorage, "GET", "",
                        AccuratApi.getEncodedRequestBody(""), AccuratEndpoints.GET_CONSENTS.getPath(urlParameters));
            }
        };

        consentModelsRequest.setTag(TAG)
                .setRetryPolicy(AccuratApi.defaultRetryPolicy)
                .setShouldCache(false);

        if (requestQueue == null) {
            AccuratLogger.log(AccuratLogger.WARNING, "requestQueue is NULL, re-initialising queue....");
            requestQueue = Volley.newRequestQueue(activity.getApplicationContext());
        }

        AccuratLogger.logNetworkRequest(HttpMethod.GET, Configuration.ENDPOINT_GET_CONSENTS);
        requestQueue.add(consentModelsRequest);
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".fetchConsentModels()");
    }

    private static void handleServerConsentModels(Activity activity, JSONObject response, AccuratCompletionCallback onCompleted) {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".handleServerConsentModels()");
        if (response == null || !response.has("data")) {
            AccuratLogger.log(AccuratLogger.ERROR, "Invalid response");
            clearAndFail(onCompleted);
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".handleServerConsentModels()");

            return;
        }

        try {
            JSONArray data = response.getJSONArray("data");
            if (data.length() == 0) {
                AccuratLogger.log(AccuratLogger.ERROR, "No data");
                fail(onCompleted);
                AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".handleServerConsentModels()");

                return;
            }
            SparseArray<Consent> permissions = new SparseArray<>();
            List<Consent> consents = new ArrayList<>();

            for (int i = 0; i < data.length(); i++) {
                JSONObject consentData = data.getJSONObject(i);
                boolean isSystemPermission = consentData.getBoolean("is_system_permission");
                Consent consent = Consent.fromServerJson(consentData, isSystemPermission);

                if (consent == null) {
                    continue;
                }

                if (isSystemPermission) {
                    if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M
                            && consent.getServerPermissionType() != null && activity.checkSelfPermission(consent.getServerPermissionType()) != PackageManager.PERMISSION_GRANTED) {
                        consent.setSystemPermissionState(consent.getSystemPermissionRefuseCount() == 0 ? ConsentState.UNKNOWN : ConsentState.REFUSED);
                    }
                    permissions.put(consent.getServerId(), consent);
                } else {
                    consents.add(consent);
                }
            }

            for (Consent consent : consents) {
                int serverPermissionId = consent.getServerPermissionId();
                if (serverPermissionId <= 0 || permissions.indexOfKey(serverPermissionId) < 0) {
                    continue;
                }

                consent.mergePermission(permissions.get(serverPermissionId));
            }

            mConsents = toMap(consents);
            if (mConsents.isEmpty()) {
                AccuratLogger.log(AccuratLogger.ERROR, "No consents after parsing");
                fail(onCompleted);
                AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".handleServerConsentModels()");

                return;
            }
            storeConsents();

            fetchUserConsents(activity, onCompleted);

        } catch (JSONException e) {
            AccuratLogger.log(AccuratLogger.JSON_ERROR, "Failed to parse consents: " + e.getMessage());
            e.printStackTrace();
            fail(onCompleted);
        }
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".handleServerConsentModels()");
    }

    private static void fetchUserConsents(final Activity activity, final AccuratCompletionCallback onCompleted) {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".fetchUserConsents()");
        if (!AccuratApi.isNetworkAvailable(activity)) {
            AccuratLogger.log(AccuratLogger.NETWORK, "Network not available");
            clearAndFail(onCompleted);
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".fetchUserConsents()");

            return;
        }

        HashMap<String, Object> urlParameters = getUserConsentsFetchParams();
        JsonObjectRequest userConsentRequest = new JsonObjectRequest(
                Request.Method.GET,
                AccuratEndpoints.GET_CONSENT.getUrl(urlParameters),
                null,
                response -> {
                    AccuratLogger.logNetworkResponse(HttpMethod.GET, Configuration.ENDPOINT_GET_CONSENT, response, false);
                    handleServerUserConsents(activity, response, onCompleted);
                },
                error -> {
                    AccuratLogger.logNetworkError(HttpMethod.GET, Configuration.ENDPOINT_GET_CONSENT, error);
                    clearAndFail(onCompleted);
                }
        ) {
            @Override
            public Map<String, String> getHeaders() {
                return AccuratApi.getHeaders(mStorage, "GET", "",
                        AccuratApi.getEncodedRequestBody(""), AccuratEndpoints.GET_CONSENT.getPath(urlParameters));
            }
        };

        userConsentRequest.setTag(TAG)
                .setRetryPolicy(AccuratApi.defaultRetryPolicy)
                .setShouldCache(false);

        if (requestQueue == null) {
            AccuratLogger.log(AccuratLogger.WARNING, "requestQueue is NULL, re-initialising queue....");
            requestQueue = Volley.newRequestQueue(activity.getApplicationContext());
        }

        AccuratLogger.logNetworkRequest(HttpMethod.GET, Configuration.ENDPOINT_GET_CONSENT);
        requestQueue.add(userConsentRequest);
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".fetchUserConsents()");
    }

    private static void handleServerUserConsents(Activity activity, JSONObject response, AccuratCompletionCallback onCompleted) {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".handleServerUserConsents()");
        if (response == null || !response.has("data")) {
            AccuratLogger.log(AccuratLogger.ERROR, "Invalid response");
            clearAndFail(onCompleted);
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".handleServerUserConsents()");

            return;
        }

        try {
            JSONArray data = response.getJSONArray("data");

            for (int i = 0; i < data.length(); i++) {
                JSONObject consentData = data.getJSONObject(i);
                handleServerUserConsent(consentData);
            }

            if (mConsents.isEmpty()) {
                AccuratLogger.log(AccuratLogger.ERROR, "No consents after parsing");
                fail(onCompleted);
                AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".handleServerUserConsents()");

                return;
            }
            storeConsents();
            startAfterFetch(activity, onCompleted);
        } catch (JSONException e) {
            AccuratLogger.log(AccuratLogger.JSON_ERROR, "Failed to parse consents: " + e.getMessage());
            e.printStackTrace();
            clearAndFail(onCompleted);
        }
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".handleServerUserConsents()");
    }

    private static void handleServerUserConsent(JSONObject consentData) {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".handleServerUserConsent()");
        if (consentData == null || !consentData.has("type") || !consentData.has("state")) {
            AccuratLogger.log(AccuratLogger.ERROR, "Null or missing type or state");
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".handleServerUserConsent()");

            return;
        }

        try {
            String type = consentData.getString("type");
            if (type == null || type.isEmpty()) {
                AccuratLogger.log(AccuratLogger.ERROR, "Invalid type");
                AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".handleServerUserConsent()");

                return;
            }

            boolean approved = consentData.getInt("state") == ConsentState.ACCEPTED.ordinal();

            for (Consent consent : mConsents.values()) {
                if (type.equals(consent.getServerType())) {
                    if (consentData.has("refuse_count")) {
                        consent.updateRefuseCount(consentData.getInt("refuse_count"));
                    }
                    if (consent.getState() != ConsentState.REFUSED && approved) {
                        consent.approve();

                        return;
                    } else {
                        consent.refuse();

                        return;
                    }
                }

                if (type.equals(consent.getServerPermissionType())) {
                    if (consentData.has("refuse_count")) {
                        consent.updatePermissionRefuseCount(consentData.getInt("refuse_count"));
                    }
                    if (consent.getState() != ConsentState.REFUSED && approved) {
                        consent.approvePermission();

                        return;
                    } else {
                        consent.refusePermission();

                        return;
                    }
                }

                if (type.equals(consent.getServerBackgroundPermissionType())) {
                    if (consent.getState() != ConsentState.REFUSED && approved) {
                        consent.approveBackgroundPermission();

                        return;
                    } else {
                        consent.refuseBackgroundPermission();

                        return;
                    }
                }
            }
            // TODO: 14/01/2019 what if the consent is not stored prior to this call?
        } catch (JSONException e) {
            AccuratLogger.log(AccuratLogger.JSON_ERROR, "Failed to parse user consent: " + e.getMessage());
            e.printStackTrace();
        }
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".handleServerUserConsent()");
    }

    private static HashMap<String, Object> getConsentModelsFetchParams() {
        HashMap<String, Object> params = new HashMap<>();
        params.put(ApiKeys.Url.LANGUAGE_KEY, getUserLanguage());

        return params;
    }

    private static String getUserLanguage() {
        AccuratUser user = AccuratUserManager.getUser();
        if (user == null || user.language == null) {
            return Locale.getDefault().toString();
        }

        return user.language.getCode();
    }

    private static HashMap<String, Object> getUserConsentsFetchParams() {
        HashMap<String, Object> params = new HashMap<>();
        params.put(ApiKeys.Url.AD_ID, AdvertisingManager.getAdId());

        return params;
    }

    private static void storeConsents() {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".storeConsents()");
        checkInitialized();
        if (mConsents == null || mConsents.isEmpty()) {
            AccuratLogger.log(AccuratLogger.NONE, "Nothing to store");

            return;
        }

        String serializedConsents = serialize(mConsents);

        AccuratConfigurationManager.storeConsentJson(serializedConsents);
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".storeConsents()");
    }

    private static String serialize(HashMap<ConsentType, Consent> consents) {
        if (consents == null) {
            return "";
        }

        JSONArray json = new JSONArray();
        for (Consent consent : consents.values()) {
            json.put(consent.getJson().toString());
        }

        return json.toString();
    }
    // </editor-fold>

    private static void startAfterFetch(final Activity activity, final AccuratCompletionCallback onCompleted) {
        loadConsents();

        switch (getAllConsentsStateForAsking(activity)) {
            case HAS_NONE:
            case ALL_APPROVED_OR_REFUSED:
                fail(onCompleted);
                break;
            case ALL_APPROVED:
                succeed(onCompleted);
                break;
            case HAS_UNKNOWN:
                askAllConsents(activity, approvedConsents -> {
                    // Todo - Send updated consents to server
                    uploadConsents(activity.getApplicationContext(), success -> {
                        switch (getAllConsentsStateForStarting(activity)) {
                            case ALL_APPROVED:
                                succeed(onCompleted);
                                break;
                            default:
                                fail(onCompleted);
                        }
                    });
                });
                break;
        }
    }

    // <editor-fold desc="Public interface">
    public static boolean needsUserInteraction(Context context) {
        loadConsents();
        AllConsentsState state = getAllConsentsStateForAsking(context);

        return state != AllConsentsState.ALL_APPROVED && state != AllConsentsState.ALL_APPROVED_OR_REFUSED;
    }

    public static void start(final Activity activity, final AccuratCompletionCallback onCompleted) {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".start()");
        loadConsents();

        AccuratLogger.log(AccuratLogger.SDK_FLOW, "Getting all consent states...");
        switch (getAllConsentsStateForAsking(activity)) {
            case HAS_NONE:
                AccuratLogger.log(AccuratLogger.NONE, "HAS_NONE");
                AccuratLogger.log(AccuratLogger.SDK_FLOW, "Fetching consent models...");
                fetchConsentModels(activity, onCompleted);
                break;
            case ALL_APPROVED_OR_REFUSED:
                AccuratLogger.log(AccuratLogger.NONE, "ALL_APPROVED_OR_REFUSED");
                // Upload any changes to the server
                AccuratLogger.log(AccuratLogger.SDK_FLOW, "Uploading changes to server");
                uploadConsents(activity.getApplicationContext(), success -> {
                    // Ignore the result
                });
                fail(onCompleted);
                break;
            case ALL_APPROVED:
                AccuratLogger.log(AccuratLogger.NONE, "ALL_APPROVED");
                AccuratLogger.log(AccuratLogger.SDK_FLOW, "Uploading changes to server");
                // Upload any changes to the server
                uploadConsents(activity.getApplicationContext(), success -> {
                    // Ignore the result
                });
                succeed(onCompleted);
                break;
            case HAS_UNKNOWN:
                AccuratLogger.log(AccuratLogger.NONE, "HAS_UNKNOWN");
                AccuratLogger.log(AccuratLogger.SDK_FLOW, "Asking all consents");
                askAllConsents(activity, approvedConsents -> {
                    // Todo - Send updated consents to server
                    uploadConsents(activity.getApplicationContext(), success -> {
                        switch (getAllConsentsStateForStarting(activity)) {
                            case ALL_APPROVED:
                                succeed(onCompleted);
                                break;
                            default:
                                fail(onCompleted);
                        }
                    });
                });
                break;
        }
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".start()");
    }

    public static void startFromBackground(Context context, AccuratCompletionCallback onCompleted) {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".startFromBackground()");
        loadConsents();

        AccuratLogger.log(AccuratLogger.SDK_FLOW, "Getting all consent states...");
        switch (getAllConsentsStateForStarting(context)) {
            case ALL_APPROVED:
                AccuratLogger.log(AccuratLogger.SDK_FLOW, "ALL_APPROVED");
                succeed(onCompleted);
                break;
            case HAS_NONE:
            case ALL_APPROVED_OR_REFUSED:
            case HAS_UNKNOWN:
                AccuratLogger.log(AccuratLogger.SDK_FLOW, "Not ALL_APPROVED");
                fail(onCompleted);
                break;
        }
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".startFromBackground()");
    }

    public static ConsentType[] getUnapprovedConsentTypes() {
        if (mConsents == null) {
            loadConsents();
        }

        ConsentType[] allConsentTypes = ConsentType.all();
        List<ConsentType> unapprovedConsentTypes = new ArrayList<>();
        for (ConsentType consentType : allConsentTypes) {
            if (!mConsents.containsKey(consentType) || mConsents.get(consentType).getState() != ConsentState.ACCEPTED) {
                unapprovedConsentTypes.add(consentType);
            }
        }

        AccuratLogger.log(AccuratLogger.NONE, "Unapproved ConsentTypes: " + unapprovedConsentTypes);

        return unapprovedConsentTypes.toArray(new ConsentType[0]);
    }

    public static boolean hasUnapprovedConsents() {
        return getUnapprovedConsentTypes().length > 0;
    }

    public static ConsentType[] getMissingConsentTypes() {
        if (mConsents == null || mConsents.isEmpty()) {
            loadConsents();
        }

        ConsentType[] allConsentTypes = ConsentType.all();
        List<ConsentType> missingConsentTypes = new ArrayList<>();
        for (ConsentType consentType : allConsentTypes) {
            if (!mConsents.containsKey(consentType)) {
                missingConsentTypes.add(consentType);
            }
        }
        AccuratLogger.log(AccuratLogger.NONE, "Missing ConsentTypes: " + missingConsentTypes);

        return missingConsentTypes.toArray(new ConsentType[0]);
    }

    public static boolean hasMissingConsents() {
        return getMissingConsentTypes().length > 0;
    }

    public static void askAllConsents(Activity activity, ConsentResponse onConsentAnswered) {
        askConsents(activity, ConsentType.all(), onConsentAnswered);
    }

    public static void askConsents(Activity activity, ConsentType[] consentTypes, ConsentResponse onConsentAnswered) {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".askConsents()");
        RealmManager.init(activity);
        if (activity == null || consentTypes == null || consentTypes.length == 0) {
            AccuratLogger.log(AccuratLogger.NONE, "Activity is NULL or no consent types, nothing to do");
            if (onConsentAnswered != null) {
                onConsentAnswered.onConsentsAnswered(new ConsentType[0]);
            }
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".askConsents()");

            return;
        }

        if (!isValidConsentRequestTiming()) {
            AccuratLogger.log(AccuratLogger.NONE, "Don't start, already asked recently");
            if (onConsentAnswered != null) {
                onConsentAnswered.onConsentsAnswered(new ConsentType[0]);
            }
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".askConsents()");

            return;
        }

        if (AdvertisingManager.hasValidAdId()) {
            // Iterate over the consentTypes
            loadConsents();
            mConsentFlow = new ConsentFlow(activity, consentTypes, onConsentAnswered);
            askConsent(mConsentFlow);
        } else {
            AccuratLogger.log(AccuratLogger.NONE, "Don't start, no valid ad ID");
            if (onConsentAnswered != null) {
                onConsentAnswered.onConsentsAnswered(new ConsentType[0]);
            }
        }
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".askConsents()");
    }

    /**
     * Verifies whether we can ask for any type of consent (location permission / GDPR approval)
     *
     * @return True when the last request chain has ended more than {@link Configuration#MIN_CONSENT_REQUEST_HIATUS_IN_MILLISECONDS} ago, false when more recent.
     */
    private static boolean isValidConsentRequestTiming() {
        long lastConsentRequestTimestamp = RealmHelper.loadLongSetting(Setting.Keys.State.LAST_CONSENT_REQUEST_TIMESTAMP, 0);

        return Accurat.isForceConsents() || lastConsentRequestTimestamp < (System.currentTimeMillis() - Configuration.MIN_CONSENT_REQUEST_HIATUS_IN_MILLISECONDS);
    }

    public static void resetConsents(Context context, ConsentType[] consentTypes) {
        if (consentTypes == null || consentTypes.length == 0) {
            return;
        }

        loadConsents();
        for (ConsentType consentType : consentTypes) {
            if (mConsents.containsKey(consentType)) {
                resetConsent(mConsents.get(consentType));
            }
        }
        storeConsents();
        uploadConsents(context, success -> {
            // Ignore
        });
    }

    public static void resetAllConsents(Context context) {
        resetConsents(context, ConsentType.all());
    }

    public static boolean onRequestPermissionResult(int requestCode, String[] permissions, int[] grantResults, Context context) {
        return AccuratManager.INSTANCE.onRequestPermissionResult(requestCode, permissions, grantResults, context);
        /*switch (requestCode) {
            case RequestCodes.LOCATION_PERMISSIONS:
                boolean coarseLocationGranted = ActivityCompat.checkSelfPermission(context, Manifest.permission.ACCESS_COARSE_LOCATION) == PackageManager.PERMISSION_GRANTED;
                boolean fineLocationGranted = ActivityCompat.checkSelfPermission(context, Manifest.permission.ACCESS_FINE_LOCATION) == PackageManager.PERMISSION_GRANTED;
                boolean foregroundGranted = coarseLocationGranted || fineLocationGranted;
                boolean backgroundGranted = Build.VERSION.SDK_INT < Build.VERSION_CODES.Q || ActivityCompat.checkSelfPermission(context, Manifest.permission.ACCESS_BACKGROUND_LOCATION) == PackageManager.PERMISSION_GRANTED;

                onTrackingPermissionsResult(foregroundGranted, backgroundGranted);

                return true;
            default:
                return false;
        }*/
    }

    public static void refuseTrackingPermission(Context context, @NonNull final AccuratCompletionCallback onCompleted) {
        loadConsents();

        String adId = AdvertisingManager.getAdId();
        if (adId == null || adId.isEmpty()) {
            onCompleted.onCompleted(false);

            return;
        }

        if (mConsents == null || mConsents.isEmpty() || !mConsents.containsKey(ConsentType.TRACKING)) {
            onCompleted.onCompleted(false);

            return;
        }

        Consent trackingConsent = mConsents.get(ConsentType.TRACKING);
        trackingConsent.refusePermission();
        storeConsents();

        JSONObject uploadData = getUploadDataForPermission(context, trackingConsent, adId);
        if (uploadData == null) {
            onCompleted.onCompleted(false);

            return;
        }

        uploadConsent(context, uploadData, onCompleted);
    }
    // </editor-fold>

    // <editor-fold desc="Networking">
    private static void uploadConsents(Context context, @NonNull final AccuratCompletionCallback onCompleted) {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".uploadConsents()");
        RealmManager.init(context);
        if (!AccuratApi.isNetworkAvailable(context)) {
            AccuratLogger.log(AccuratLogger.NETWORK, "Network not available");
            onCompleted.onCompleted(false);
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".uploadConsents()");

            return;
        }

        if (mConsents == null || mConsents.isEmpty()) {
            AccuratLogger.log(AccuratLogger.NONE, "No consents to upload");
            onCompleted.onCompleted(true);
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".uploadConsents()");

            return;
        }

        if (!AdvertisingManager.hasValidAdId()) {
            AccuratLogger.log(AccuratLogger.WARNING, "No valid ad ID");
            onCompleted.onCompleted(false);
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".uploadConsents()");

            return;
        }

        // CountDownLatch is used to manage async upload of multiple consents and permissions.
        // Each consent and permission had to be uploaded individually at the time of implementation
        // as no webservice to upload them in batches existed.
        final String adId = AdvertisingManager.getAdId();
        final CountDownLatch countDownLatch = new CountDownLatch(getUploadCount());
        final CountDownLatch successLatch = new CountDownLatch(getUploadCount());
        AccuratLogger.log(AccuratLogger.SDK_FLOW, "Starting upload...");
        for (Consent consent : mConsents.values()) {
            JSONObject consentData = getUploadDataForConsent(consent, adId);
            if (consentData != null) {
                AccuratLogger.log(AccuratLogger.NONE, "CU 1. uploading consent");
                uploadConsent(context, consentData, success -> {
                    AccuratLogger.log(AccuratLogger.NONE, "CU 1. => " + success);
                    countDownLatch.countDown();
                    if (success) {
                        successLatch.countDown();
                    }

                    if (countDownLatch.getCount() == 0) {
                        onCompleted.onCompleted(successLatch.getCount() == 0);
                    }
                });
            } else {
                AccuratLogger.log(AccuratLogger.NONE, "CU 1. No consent to upload");
                countDownLatch.countDown();
            }

            JSONObject permissionData = getUploadDataForPermission(context, consent, adId);
            if (consentData != null && permissionData != null) {
                AccuratLogger.log(AccuratLogger.NONE, "CU 2. uploading permission");
                uploadConsent(context, permissionData, success -> {
                    AccuratLogger.log(AccuratLogger.NONE, "CU 2. => " + success);
                    countDownLatch.countDown();
                    if (success) {
                        successLatch.countDown();
                    }

                    if (countDownLatch.getCount() == 0) {
                        onCompleted.onCompleted(successLatch.getCount() == 0);
                    }
                });
            } else {
                AccuratLogger.log(AccuratLogger.NONE, "CU 2. No permission to upload");
                countDownLatch.countDown();
            }

            JSONObject backgroundPermissionData = getUploadDataForBackgroundPermission(context, consent, adId);
            if (consentData != null && backgroundPermissionData != null) {
                AccuratLogger.log(AccuratLogger.NONE, "CU 3. uploading background permission");
                uploadConsent(context, backgroundPermissionData, success -> {
                    AccuratLogger.log(AccuratLogger.NONE, "CU 3. => " + success);
                    countDownLatch.countDown();
                    if (success) {
                        successLatch.countDown();
                    }

                    if (countDownLatch.getCount() == 0) {
                        onCompleted.onCompleted(successLatch.getCount() == 0);
                    }
                });
            } else {
                AccuratLogger.log(AccuratLogger.NONE, "CU 3. No background permission to upload");
                countDownLatch.countDown();
            }
        }
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".uploadConsents()");
    }

    private static void uploadConsent(final Context context, @NonNull JSONObject data, @NonNull final AccuratCompletionCallback onCompleted) {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".uploadConsent()");
        JsonObjectRequest uploadConsentRequest = new JsonObjectRequest(
                Request.Method.POST,
                AccuratEndpoints.POST_CONSENT.getUrl(),
                data,
                response -> {
                    AccuratLogger.logNetworkResponse(HttpMethod.POST, Configuration.ENDPOINT_POST_CONSENT, response, false);
                    onCompleted.onCompleted(true);
                },
                error -> {
                    AccuratLogger.logNetworkError(HttpMethod.POST, Configuration.ENDPOINT_POST_CONSENT, error);
                    onCompleted.onCompleted(false);
                }
        ) {
            @Override
            public Map<String, String> getHeaders() {
                return AccuratApi.getHeaders(mStorage, "POST", getBodyContentType(),
                        AccuratApi.getEncodedRequestBody(data.toString()), AccuratEndpoints.POST_CONSENT.getPath());
            }
        };

        uploadConsentRequest.setTag(TAG)
                .setRetryPolicy(AccuratApi.defaultRetryPolicy)
                .setShouldCache(false);

        if (requestQueue == null) {
            AccuratLogger.log(AccuratLogger.WARNING, "requestQueue is NULL, re-initialising queue....");
            requestQueue = Volley.newRequestQueue(context);
        }

        AccuratLogger.logNetworkRequest(HttpMethod.POST, Configuration.ENDPOINT_POST_CONSENT, data, false);
        requestQueue.add(uploadConsentRequest);
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".uploadConsent()");
    }
    // </editor-fold>

    // <editor-fold desc="Consent">
    private static void askConsent(ConsentFlow consentFlow) {
        // If we've iterated over all Consents, then call the ConsentResponse.
        if (!consentFlow.hasMore()) {
            finishConsentFlow(consentFlow);

            return;
        }

        // Else, do the next iteration
        final ConsentType consentType = consentFlow.getConsentType();
        // If we don't know the Consent, go to the next one.
        if (!mConsents.containsKey(consentType)) {
            consentFlow.setConsentState(consentType, ConsentState.UNKNOWN);
            askConsent(consentFlow.getNext());

            return;
        }

        // If we do know the Consent, handle it.
        Consent consent = mConsents.get(consentType);
        if (consent.getState() != ConsentState.ACCEPTED) {
            AccuratLogger.log(AccuratLogger.NONE, "Consent refuse count = " + consent.getRefuseCount());
            AccuratLogger.log(AccuratLogger.NONE, "Max consent refuse count = " + consent.getMaxConsentRefuseCount());
            if (Accurat.isForceConsents() || (AccuratConfigurationManager.hasGdprConsentFeature() && consent.getRefuseCount() < consent.getMaxConsentRefuseCount())) {
                // Show the Consent dialog
                showConsentDialog(consentFlow, consent);
            } else {
                finishConsentFlow(consentFlow);
            }

            return;
        }

        // Consent has already been accepted, check the associated permissions.
        if (shouldAskPermissions(consentFlow, consent)) {
            showPermissionDialog(consentFlow, consent);

            return;
        }

        askConsent(consentFlow.getNext());
    }

    /**
     * Stores the consents, records the time for future reference (hiatus) and returns the consent result.
     *
     * @param consentFlow Consent flow to finish
     */
    private static void finishConsentFlow(ConsentFlow consentFlow) {
        storeConsents();
        consentFlow.respond();
        // Remember when we've requested permissions and consents last
        RealmHelper.storeSetting(Setting.Keys.State.LAST_CONSENT_REQUEST_TIMESTAMP, System.currentTimeMillis());
    }

    private static void resetConsent(Consent consent) {
        if (consent == null) {
            return;
        }

        consent.reset();
    }

    public static ConsentState getConsentState(ConsentType consentType) {
        checkInitialized();

        loadConsents();

        if (mConsents == null || !mConsents.containsKey(consentType)) {
            return ConsentState.UNKNOWN;
        }

        return mConsents.get(consentType).getState();
    }

    public static void updateConsent(Activity activity, ConsentType consentType, ConsentState consentState, AccuratCompletionCallback onCompleted) {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".updateConsent(" + consentType + ", " + consentState + ")");
        checkInitialized();

        loadConsents();

        if (mConsents == null) {
            // Unknown consent type
            AccuratLogger.log(AccuratLogger.WARNING, "No consents yet, creating container");
            mConsents = new HashMap<>();

            return;
        }

        if (mConsents.isEmpty()) {
            AccuratLogger.log(AccuratLogger.WARNING, "No consents in storage, fetching them from server");
            fetchConsentModels(activity, success -> finalizeUpdateConsent(consentType, consentState, onCompleted));

            return;
        }

        finalizeUpdateConsent(consentType, consentState, onCompleted);
    }

    private static void finalizeUpdateConsent(ConsentType consentType, ConsentState consentState, AccuratCompletionCallback onCompleted) {
        if (mConsents == null) {
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".updateConsent()");
            fail(onCompleted);

            return;
        }

        Consent consent = mConsents.get(consentType);
        if (consent == null) {
            AccuratLogger.log(AccuratLogger.WARNING, "No " + consentType + " consent yet, creating object");
            consent = new Consent();
            consent.setType(consentType);

            mConsents.put(consentType, consent);
        }

        if (consent.getState() != consentState) {
            AccuratLogger.log(AccuratLogger.SDK_FLOW, "Setting consentState to " + consentState);
            consent.setState(consentState);
            storeConsents();
        } else {
            AccuratLogger.log(AccuratLogger.NONE, "consentState already is " + consentState);
        }

        respond(onCompleted, true);
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".updateConsent()");
    }

    public static void updatePermission(Activity activity, ConsentType consentType, ConsentState consentState, AccuratCompletionCallback onCompleted) {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".updatePermission(" + consentType + ", " + consentState + ")");
        checkInitialized();

        loadConsents();

        if (mConsents == null) {
            // Unknown consent type
            AccuratLogger.log(AccuratLogger.WARNING, "No consents yet, creating container");
            mConsents = new HashMap<>();

            return;
        }

        if (mConsents.isEmpty()) {
            AccuratLogger.log(AccuratLogger.WARNING, "No consents in storage, fetching them from server");
            fetchConsentModels(activity, success -> finalizeUpdatePermission(consentType, consentState, onCompleted));

            return;
        }

        finalizeUpdatePermission(consentType, consentState, onCompleted);
    }

    private static void finalizeUpdatePermission(ConsentType consentType, ConsentState consentState, AccuratCompletionCallback onCompleted) {
        if (mConsents == null) {
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".updatePermission()");
            fail(onCompleted);

            return;
        }

        Consent consent = mConsents.get(consentType);
        if (consent == null) {
            AccuratLogger.log(AccuratLogger.WARNING, "No " + consentType + " consent yet, creating object");
            consent = new Consent();
            consent.setType(consentType);

            mConsents.put(consentType, consent);
        }

        if (consent.getSystemPermissionState() != consentState) {
            AccuratLogger.log(AccuratLogger.SDK_FLOW, "Setting permissionState to " + consentState);
            consent.setSystemPermissionState(consentState);
            storeConsents();
        } else {
            AccuratLogger.log(AccuratLogger.NONE, "permissionState already is " + consentState);
        }

        if (consent.getSystemPermissionBackgroundState() != consentState) {
            AccuratLogger.log(AccuratLogger.SDK_FLOW, "Setting permissionBackgroundState to " + consentState);
            consent.setSystemPermissionBackgroundState(consentState);
            storeConsents();
        } else {
            AccuratLogger.log(AccuratLogger.NONE, "permissionBackgroundState already is " + consentState);
        }

        respond(onCompleted, true);
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".updatePermission()");
    }
    // </editor-fold>

    // <editor-fold desc="Dialogs">
    private static void showConsentDialog(final ConsentFlow consentFlow, @NonNull final Consent consent) {
        LayoutInflater inflater = consentFlow.activity.getLayoutInflater();

        // Build the Consent dialog and ViewHolder
        ConsentDialogViewHolder viewHolder = new ConsentDialogViewHolder(inflater.inflate(R.layout.dialog_consent, null));
        viewHolder.setConsent(consent);

        AlertDialog.Builder builder = new AlertDialog.Builder(consentFlow.activity)
                .setView(viewHolder.rootView)
                .setCancelable(false)
                .setOnCancelListener(dialog -> {
                    mConsents.get(consent.getType()).increaseRefuseCount();
                    // The Consent was ignored
                    consentFlow.setConsentState(consent.getType(), ConsentState.UNKNOWN);
                    askConsent(consentFlow.getNext());
                });

        final AlertDialog consentDialog = builder.create();
        consentDialog.setCanceledOnTouchOutside(false);
        viewHolder.setPositiveButtonHandler(v -> {
            consentDialog.dismiss();

            // The Consent was approved
            mConsents.get(consent.getType()).approve();
            consentFlow.setConsentState(consent.getType(), ConsentState.ACCEPTED);
            storeConsents();
            if (shouldAskPermissions(consentFlow, consent)) {
                showPermissionDialog(consentFlow, consent);
            } else {
                askConsent(consentFlow.getNext());
            }
        }).setNegativeButtonHandler(v -> {
            consentDialog.dismiss();

            // The Consent was refused
            mConsents.get(consent.getType()).refuse();
            mConsents.get(consent.getType()).increaseRefuseCount();
            consentFlow.setConsentState(consent.getType(), ConsentState.REFUSED);
            finishConsentFlow(consentFlow);
        });

        if (!consentFlow.activity.isFinishing()) {
            // Show the Consent dialog
            consentDialog.show();
            viewHolder.enableDialogLinks();
        }
    }

    private static boolean shouldAskPermissions(ConsentFlow consentFlow, Consent consent) {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".shouldAskPermissions()");
        boolean foregroundPermissionsGranted = consent.getType().hasPermissions(consentFlow.activity);
        boolean backgroundPermissionsGranted = consent.getType().hasBackgroundPermission(consentFlow.activity);
        if (foregroundPermissionsGranted && backgroundPermissionsGranted) {
            AccuratLogger.log(AccuratLogger.SDK_FLOW, "-- No, app already has permissions");
            consent.updatePermissionRefuseCount(0);

            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".shouldAskPermissions()");
            return false;
        }

        AccuratLogger.log(AccuratLogger.NONE, "isForceConsents() = " + Accurat.isForceConsents());
        AccuratLogger.log(AccuratLogger.NONE, "System permission refuse count = " + consent.getSystemPermissionRefuseCount());
        AccuratLogger.log(AccuratLogger.NONE, "Max system permission refuse count = " + consent.getMaxPermissionRefuseCount());
        AccuratLogger.log(AccuratLogger.NONE, "Foreground permissions granted?: " + foregroundPermissionsGranted);
        AccuratLogger.log(AccuratLogger.NONE, "Background permissions granted?: " + backgroundPermissionsGranted);
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".shouldAskPermissions()");
        return Accurat.isForceConsents() || (AccuratConfigurationManager.hasLocationPermissionFeature() && consent.getSystemPermissionRefuseCount() < consent.getMaxPermissionRefuseCount());
    }

    private static void showPermissionDialog(final ConsentFlow consentFlow, final Consent consent) {
        LayoutInflater inflater = consentFlow.activity.getLayoutInflater();

        // Build the Permission dialog and ViewHolder
        ConsentDialogViewHolder viewHolder = new ConsentDialogViewHolder(inflater.inflate(R.layout.dialog_consent, null));
        viewHolder.setPermission(consent);

        AlertDialog.Builder builder = new AlertDialog.Builder(consentFlow.activity)
                .setView(viewHolder.rootView)
                .setCancelable(false);

        final AlertDialog permissionDialog = builder.create();
        permissionDialog.setCanceledOnTouchOutside(false);
        viewHolder.setPositiveButtonHandler(v -> {
            permissionDialog.dismiss();

            // The permission may be requested
            askPermissions(consentFlow, consent);
        });

        if (!consentFlow.activity.isFinishing()) {
            // Show the Permission dialog
            permissionDialog.show();
            viewHolder.enableDialogLinks();
        }
    }
    // </editor-fold>

    // <editor-fold desc="Permission">
    private static void askPermissions(ConsentFlow consentFlow, @NonNull final Consent consent) {
        switch (consent.getType()) {
            case TRACKING:
                ActivityCompat.requestPermissions(consentFlow.activity, consent.getType().getPermissions(), consent.getType().getRequestCode());
                break;
            default:
                askConsent(consentFlow.getNext());
        }
    }

    private static void onTrackingPermissionsResult(boolean granted, boolean backgroundGranted) {
        if (mConsentFlow == null) {
            return;
        }

        if (!mConsents.containsKey(ConsentType.TRACKING)) {
            askConsent(mConsentFlow.getNext());

            return;
        }

        Consent consent = mConsents.get(ConsentType.TRACKING);
        if (consent == null) {
            askConsent(mConsentFlow.getNext());

            return;
        }

        if (granted) {
            consent.approvePermission();
            if (VersionManager.isBelowAndroid10()) {
                consent.approveBackgroundPermission();
            } else if (backgroundGranted) {
                consent.approveBackgroundPermission();
            } else {
                consent.refuseBackgroundPermission();
            }
        } else {
            consent.refusePermission();
            consent.refuseBackgroundPermission();
        }

        VersionManager.whenBelow10(() -> {
            if (!granted) {
                consent.increasePermissionRefuseCount();
            }
        }, () -> {
            if (!backgroundGranted) {
                consent.increasePermissionRefuseCount();

            }
        });
        storeConsents();
        askConsent(mConsentFlow.getNext());

    }
    // </editor-fold>

    // <editor-fold desc="Helpers">
    private static AllConsentsState getAllConsentsStateForStarting(Context context) {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".getAllConsentsStateForStarting()");
        if (mConsents == null || mConsents.isEmpty()) {
            AccuratLogger.log(AccuratLogger.NONE, mConsents == null ? "Consents are null" : "Consents are empty");
            return AllConsentsState.HAS_NONE;
        }

        boolean allApproved = true;
        for (Consent consent : mConsents.values()) {
            try {
                AccuratLogger.log(AccuratLogger.NONE, "Checking consent: " + consent.getJson().toString());
            } catch (NullPointerException e) {
                AccuratLogger.log(AccuratLogger.ERROR, e.getMessage());
            }
            AccuratLogger.log(AccuratLogger.NONE, "Checking system permission");
            ConsentState systemPermissionState = consent.getSystemPermissionState(context);
            ConsentState systemPermissionBackgroundState = consent.getSystemPermissionBackgroundState(context);
            AccuratLogger.log(AccuratLogger.NONE, "systemPermissionState = " + systemPermissionState);
            AccuratLogger.log(AccuratLogger.NONE, "systemPermissionBackgroundState = " + systemPermissionBackgroundState);

            ConsentState combinedState = ConsentState.UNKNOWN;
            if (systemPermissionState == ConsentState.ACCEPTED || systemPermissionBackgroundState == ConsentState.ACCEPTED) {
                combinedState = ConsentState.ACCEPTED;
            } else if (systemPermissionState == ConsentState.REFUSED || systemPermissionBackgroundState == ConsentState.REFUSED) {
                combinedState = ConsentState.REFUSED;
            }
            AccuratLogger.log(AccuratLogger.NONE, "combinedState = " + combinedState);

            switch (combinedState) {
                case UNKNOWN:
                    AccuratLogger.log(AccuratLogger.NONE, "System permission is UNKNOWN");
                    return AllConsentsState.HAS_UNKNOWN;
                case REFUSED:
                    AccuratLogger.log(AccuratLogger.NONE, "System permission is REFUSED");
                    AccuratLogger.log(AccuratLogger.NONE, "isForceConsents() = " + Accurat.isForceConsents());
                    AccuratLogger.log(AccuratLogger.NONE, "System permission refuse count = " + consent.getSystemPermissionRefuseCount());
                    AccuratLogger.log(AccuratLogger.NONE, "Max system permission refuse count = " + consent.getMaxPermissionRefuseCount());
                    if (!Accurat.isForceConsents() && consent.getSystemPermissionRefuseCount() >= consent.getMaxPermissionRefuseCount()) {
                        AccuratLogger.log(AccuratLogger.NONE, "Setting allApproved to false");
                        allApproved = false;
                    } else {
                        AccuratLogger.log(AccuratLogger.NONE, "Returning HAS_UNKNOWN");
                        return AllConsentsState.HAS_UNKNOWN;
                    }
                    break;
                case ACCEPTED:
                    AccuratLogger.log(AccuratLogger.NONE, "System permission is ACCEPTED");
                    VersionManager.whenBelow10(() -> consent.updatePermissionRefuseCount(0), () -> {
                        if (systemPermissionBackgroundState == ConsentState.ACCEPTED) {
                            consent.updatePermissionRefuseCount(0);
                        }
                    });
                    break;
            }

            AccuratLogger.log(AccuratLogger.NONE, "Checking consent");
            switch (consent.getState()) {
                case UNKNOWN:
                    AccuratLogger.log(AccuratLogger.NONE, "Consent is UNKNOWN");
                    return AllConsentsState.HAS_UNKNOWN;
                case REFUSED:
                    AccuratLogger.log(AccuratLogger.NONE, "Consent is REFUSED");
                    if (!Accurat.isForceConsents() && consent.getRefuseCount() >= consent.getMaxConsentRefuseCount()) {
                        AccuratLogger.log(AccuratLogger.NONE, "Setting allApproved to false");
                        allApproved = false;
                    } else {
                        AccuratLogger.log(AccuratLogger.NONE, "Returning HAS_UNKNOWN");
                        return AllConsentsState.HAS_UNKNOWN;
                    }
                    break;
                case ACCEPTED:
                    AccuratLogger.log(AccuratLogger.NONE, "Consent is ACCEPTED");
                    consent.updateRefuseCount(0);
            }
        }
        AccuratLogger.log(AccuratLogger.NONE, "Returning " + (allApproved ? "ALL_APPROVED" : "ALL_APPROVED_OR_REFUSED"));

        return allApproved ? AllConsentsState.ALL_APPROVED : AllConsentsState.ALL_APPROVED_OR_REFUSED;
    }

    private static AllConsentsState getAllConsentsStateForAsking(Context context) {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".getAllConsentsStateForAsking()");
        if (mConsents == null || mConsents.isEmpty()) {
            AccuratLogger.log(AccuratLogger.NONE, mConsents == null ? "Consents are null" : "Consents are empty");
            return AllConsentsState.HAS_NONE;
        }

        boolean allApproved = true;
        for (Consent consent : mConsents.values()) {
            try {
                AccuratLogger.log(AccuratLogger.NONE, "Checking consent: " + consent.getJson().toString());
            } catch (NullPointerException e) {
                AccuratLogger.log(AccuratLogger.ERROR, e.getMessage());
            }
            AccuratLogger.log(AccuratLogger.NONE, "Checking system permission");
            ConsentState systemPermissionState = consent.getSystemPermissionState(context);
            ConsentState systemPermissionBackgroundState = consent.getSystemPermissionBackgroundState(context);
            AccuratLogger.log(AccuratLogger.NONE, "systemPermissionState = " + systemPermissionState);
            AccuratLogger.log(AccuratLogger.NONE, "systemPermissionBackgroundState = " + systemPermissionBackgroundState);

            ConsentState combinedState = ConsentState.UNKNOWN;
            if (systemPermissionState == ConsentState.ACCEPTED && systemPermissionBackgroundState == ConsentState.ACCEPTED) {
                combinedState = ConsentState.ACCEPTED;
            } else if (systemPermissionState == ConsentState.REFUSED && systemPermissionBackgroundState == ConsentState.REFUSED) {
                combinedState = ConsentState.REFUSED;
            }
            AccuratLogger.log(AccuratLogger.NONE, "combinedState = " + combinedState);

            switch (combinedState) {
                case UNKNOWN:
                    AccuratLogger.log(AccuratLogger.NONE, "System permission is UNKNOWN");
                    return AllConsentsState.HAS_UNKNOWN;
                case REFUSED:
                    AccuratLogger.log(AccuratLogger.NONE, "System permission is REFUSED");
                    AccuratLogger.log(AccuratLogger.NONE, "isForceConsents() = " + Accurat.isForceConsents());
                    AccuratLogger.log(AccuratLogger.NONE, "System permission refuse count = " + consent.getSystemPermissionRefuseCount());
                    AccuratLogger.log(AccuratLogger.NONE, "Max system permission refuse count = " + consent.getMaxPermissionRefuseCount());
                    if (!Accurat.isForceConsents() && consent.getSystemPermissionRefuseCount() >= consent.getMaxPermissionRefuseCount()) {
                        AccuratLogger.log(AccuratLogger.NONE, "Setting allApproved to false");
                        allApproved = false;
                    } else {
                        AccuratLogger.log(AccuratLogger.NONE, "Returning HAS_UNKNOWN");
                        return AllConsentsState.HAS_UNKNOWN;
                    }
                    break;
                case ACCEPTED:
                    AccuratLogger.log(AccuratLogger.NONE, "System permission is ACCEPTED");
                    consent.updatePermissionRefuseCount(0);
                    break;
            }

            AccuratLogger.log(AccuratLogger.NONE, "Checking consent");
            switch (consent.getState()) {
                case UNKNOWN:
                    AccuratLogger.log(AccuratLogger.NONE, "Consent is UNKNOWN");
                    return AllConsentsState.HAS_UNKNOWN;
                case REFUSED:
                    AccuratLogger.log(AccuratLogger.NONE, "Consent is REFUSED");
                    if (!Accurat.isForceConsents() && consent.getRefuseCount() >= consent.getMaxConsentRefuseCount()) {
                        AccuratLogger.log(AccuratLogger.NONE, "Setting allApproved to false");
                        allApproved = false;
                    } else {
                        AccuratLogger.log(AccuratLogger.NONE, "Returning HAS_UNKNOWN");
                        return AllConsentsState.HAS_UNKNOWN;
                    }
                    break;
                case ACCEPTED:
                    AccuratLogger.log(AccuratLogger.NONE, "Consent is ACCEPTED");
                    consent.updateRefuseCount(0);
            }
        }
        AccuratLogger.log(AccuratLogger.NONE, "Returning " + (allApproved ? "ALL_APPROVED" : "ALL_APPROVED_OR_REFUSED"));

        return allApproved ? AllConsentsState.ALL_APPROVED : AllConsentsState.ALL_APPROVED_OR_REFUSED;
    }

    private static void respond(AccuratCompletionCallback onCompleted, boolean success) {
        if (onCompleted != null) {
            onCompleted.onCompleted(success);
        }
    }

    private static void succeed(AccuratCompletionCallback onCompleted) {
        storeConsents();
        ConsentVerifier.planConsentVerification();
        respond(onCompleted, true);
    }

    private static void fail(AccuratCompletionCallback onCompleted) {
        respond(onCompleted, false);
    }

    private static void clearAndFail(AccuratCompletionCallback onCompleted) {
        if (AccuratConfigurationManager.hasGdprConsentFeature()) {
            clearGdprConsent();
        }
        if (AccuratConfigurationManager.hasLocationPermissionFeature()) {
            clearLocationPermission();
        }
        respond(onCompleted, false);
    }

    private static HashMap<ConsentType, Consent> toMap(List<Consent> consents) {
        HashMap<ConsentType, Consent> map = new HashMap<>();
        for (Consent consent : consents) {
            map.put(consent.getType(), consent);
        }

        return map;
    }

    private static int getUploadCount() {
        if (mConsents == null || mConsents.isEmpty()) {
            return 0;
        }

        int count = 0;
        for (Consent consent : mConsents.values()) {
            String serverType = consent.getServerType();
            if (serverType != null && !serverType.isEmpty()) {
                count++;
            }

            String serverPermissionType = consent.getServerPermissionType();
            if (serverPermissionType != null && !serverPermissionType.isEmpty()) {
                count++;
            }

            String serverBackgroundPermissionType = consent.getServerBackgroundPermissionType();
            if (serverBackgroundPermissionType != null && !serverBackgroundPermissionType.isEmpty()) {
                count++;
            }
        }

        return count;
    }

    private static JSONObject getUploadDataForConsent(Consent consent, String adId) {
        if (consent == null) {
            return null;
        }

        JSONObject json = new JSONObject();

        try {
            json.put("ad_id", adId);
            json.put("type", consent.getServerType() == null ? "gdpr" : consent.getServerType());
            json.put("state", consent.getState() == ConsentState.ACCEPTED ? 1 : 0);
            json.put("version", Configuration.API_CONSENT_VERSION);
            json.put("refuse_count", consent.getRefuseCount());
        } catch (JSONException e) {
            e.printStackTrace();

            return null;
        }

        return json;
    }

    private static JSONObject getUploadDataForPermission(Context context, Consent consent, String adId) {
        if (consent == null) {
            return null;
        }

        JSONObject json = new JSONObject();

        try {
            json.put("ad_id", adId);
//            json.put("type", consent.getServerPermissionType() == null ? "inapp_permission" : consent.getServerPermissionType());
            json.put("type", "inapp_permission");
            json.put("state", consent.getSystemPermissionState(context) == ConsentState.ACCEPTED ? 1 : 0);
            json.put("version", Configuration.API_CONSENT_VERSION);
            json.put("refuse_count", consent.getSystemPermissionRefuseCount());
        } catch (JSONException e) {
            e.printStackTrace();

            return null;
        }

        return json;
    }

    private static JSONObject getUploadDataForBackgroundPermission(Context context, Consent consent, String adId) {
        if (consent == null) {
            return null;
        }

        JSONObject json = new JSONObject();

        try {
            json.put("ad_id", adId);
            json.put("type", consent.getServerBackgroundPermissionType() == null ? "always_permission" : consent.getServerBackgroundPermissionType());
            json.put("state", consent.getSystemPermissionBackgroundState(context) == ConsentState.ACCEPTED ? 1 : 0);
            json.put("version", Configuration.API_CONSENT_VERSION);
            json.put("refuse_count", consent.getSystemPermissionRefuseCount());
        } catch (JSONException e) {
            e.printStackTrace();

            return null;
        }

        return json;
    }

    private static int getMaxConsentRefuseCount() {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".getMaxConsentRefuseCount()");
        AccuratSettings settings = AccuratSettingsManager.getSettings();
        if (settings == null) {
            AccuratLogger.log(AccuratLogger.SDK_FLOW, "Settings not yet fetched from server, using Configuration default (" + Configuration.MAX_CONSENT_REFUSE_COUNT + ")");
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".getMaxConsentRefuseCount()");
            return Configuration.MAX_CONSENT_REFUSE_COUNT;
        }

        AccuratLogger.log(AccuratLogger.SDK_FLOW, "Setting from server available (" + settings.getMaxConsentRefuseCount() + ")");
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".getMaxConsentRefuseCount()");
        return settings.getMaxConsentRefuseCount();
    }

    private static int getMaxPermissionRefuseCount() {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".getMaxPermissionRefuseCount()");
        AccuratSettings settings = AccuratSettingsManager.getSettings();
        if (settings == null) {
            AccuratLogger.log(AccuratLogger.SDK_FLOW, "Settings not yet fetched from server, using Configuration default (" + Configuration.MAX_PERMISSION_REFUSE_COUNT + ")");
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".getMaxPermissionRefuseCount()");
            return Configuration.MAX_PERMISSION_REFUSE_COUNT;
        }

        AccuratLogger.log(AccuratLogger.SDK_FLOW, "Setting from server available (" + settings.getMaxPermissionRefuseCount() + ")");
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".getMaxPermissionRefuseCount()");
        return settings.getMaxPermissionRefuseCount();
    }
    // </editor-fold>
}
