package ai.accurat.sdk.core;

import android.content.Context;
import android.location.Location;

import androidx.annotation.NonNull;

import java.util.ArrayList;
import java.util.List;

import ai.accurat.sdk.constants.StorageKeys;
import ai.accurat.sdk.data.RealmHelper;
import ai.accurat.sdk.data.enums.GeofenceType;
import ai.accurat.sdk.data.models.DatabaseGeofence;
import ai.accurat.sdk.data.models.TriggeredGeofence;
import ai.accurat.sdk.managers.AccuratConfigurationManager;

public class CustomGeofenceManager {

    // <editor-fold desc="Properties">
    private static final String TAG = CustomGeofenceManager.class.getSimpleName();

    private static MultiProcessStorage storage;
    // </editor-fold>

    // <editor-fold desc="Initialisation">
    public static void init(Context context) {
        if (!isInitialized()) {
            AccuratLogger.init(context);
            AccuratLogger.log(AccuratLogger.SDK_FLOW, "Initialising " + TAG);
            CustomGeofenceStorageManager.init(context);
            CustomGeofenceSyncManager.init(context);
            CustomGeofenceSelectionManager.init(context);
            GeofenceNotificationManager.init(context);
            CampaignManager.init(context);
            storage = MultiProcessStorage.getStorage(context, StorageKeys.ACCURAT_MULTI_PROCESS_STORAGE);
        }
    }

    private static boolean isInitialized() {
        return storage != null;
    }

    private static void checkInitialized() {
        if (isInitialized()) {
            return;
        }

        throw new IllegalStateException(TAG + " has not yet been initialised.");
    }

    // </editor-fold>

    // <editor-fold desc="Public interface">
    static void onLocationChanged(Context context, Location location) {
        checkInitialized();
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".onLocationChanged()");

        // Step 1 - Check the usability of the location
        AccuratLogger.log(AccuratLogger.SDK_FLOW, "Step 1 - Checking location usability");
        if (!isUsable(location)) {
            return;
        }

        // Step 2 - Update the monitored geofences
        AccuratLogger.log(AccuratLogger.SDK_FLOW, "Step 2 - Update the geofence selection, if necessary");
        AccuratGeofence metaGeofence = CustomGeofenceStorageManager.getMetaGeofence();
        if (metaGeofence == null || !metaGeofence.contains(location)) {
            if (metaGeofence == null) {
                AccuratLogger.log(AccuratLogger.NONE, "The meta geofence is null");
            } else {
                AccuratLogger.log(AccuratLogger.NONE, "The current location is outside the meta geofence");
            }
            AccuratLogger.log(AccuratLogger.NONE, "Going to update the geofence selection");
            CustomGeofenceSelectionManager.update(location, success -> {
                findGeofencesAndTrigger(context, location);
            });

            AccuratLogger.log(AccuratLogger.SDK_FLOW, "Updating the geofence selection is asynchronous, so onLocationChanged() stops here.");
            return;
        } else {
            AccuratLogger.log(AccuratLogger.NONE, "The current location is inside the meta geofence, no need to update the selection");
        }

        findGeofencesAndTrigger(context, location);
    }

    public static void stop() {
        CustomGeofenceSyncManager.stop();
    }
    // </editor-fold>

    // <editor-fold desc="SDK Flow">
    private static void findGeofencesAndTrigger(Context context, Location location) {
        AccuratDatabase database = new AccuratDatabase(DatabaseHelper.getInstance(context));
        boolean isTrackingTriggeredGeofences = AccuratConfigurationManager.load().isTrackingTriggeredGeofences();
        if (isTrackingTriggeredGeofences) {
            AccuratLogger.log(AccuratLogger.SDK_FLOW, "TRACK_TRIGGERED_GEOFENCES is enabled, SDK won't post notifications");
        } else {
            AccuratLogger.log(AccuratLogger.SDK_FLOW, "TRACK_TRIGGERED_GEOFENCES is disabled, SDK will post notifications");
        }

        // Step 2.5 - Find geofences which were exited
        AccuratLogger.log(AccuratLogger.SDK_FLOW, "Step 2.5 - Find geofences which were exited");
        List<DatabaseGeofence> exitedGeofences = DatabaseGeofence.Companion.findExitedGeofences(location.getLongitude(), location.getLatitude());
        AccuratLogger.log(AccuratLogger.GEOFENCE, "Exiting " + exitedGeofences.size() + " geofences");
        for (DatabaseGeofence exitedGeofence : exitedGeofences) {
            AccuratLogger.log(
                    AccuratLogger.GEOFENCE,
                    "Exited geofence " + exitedGeofence.getId()
                            + "([" + exitedGeofence.getLatitude() + " | " + exitedGeofence.getLongitude()
                            + "] and radius " + exitedGeofence.getRadius() + ")"
            );
            RealmHelper.deleteFromRealm(DatabaseGeofence.class, DatabaseGeofence.RealmColumns.ID, exitedGeofence.getId());
            AccuratGeofence exitedAccuratGeofence = database.getGeofenceById(exitedGeofence.getId());
            if (exitedAccuratGeofence != null && GeofenceType.EXIT.matches(exitedAccuratGeofence.getType())) {
                if (isTrackingTriggeredGeofences) {
                    AccuratLogger.log(AccuratLogger.SDK_FLOW, "[EXIT] Storing triggered geofence " + exitedGeofence.getId());
                    TriggeredGeofence.Companion.storeTriggeredGeofence(
                            exitedAccuratGeofence,
                            GeofenceType.EXIT,
                            location.getTime()
                    );
                } else {
                    AccuratLogger.log(AccuratLogger.SDK_FLOW, "[EXIT] Posting a notification for " + exitedGeofence.getId());
                    postNotification(context, exitedAccuratGeofence);
                }
            }
        }

        // Step 3 - Find geofences around the location
        AccuratLogger.log(AccuratLogger.SDK_FLOW, "Step 3 - Find triggering geofences around the current location");
        List<AccuratGeofence> triggeringGeofences = findTriggeringGeofences(location);
        if (!triggeringGeofences.isEmpty()) {
            for (AccuratGeofence geofence : triggeringGeofences) {
                AccuratLogger.log(AccuratLogger.GEOFENCE, "Triggering geofence " + geofence.getLogString());
                AccuratLogger.log(AccuratLogger.GEOFENCE, "-- with data " + geofence.getData());
            }

            List<AccuratGeofence> entryGeofences = CollectionUtilsKt.filter(triggeringGeofences, geofence -> GeofenceType.ENTER.matches(geofence.getType()));
            CollectionUtilsKt.forEach(entryGeofences, geofence -> {
                if (isTrackingTriggeredGeofences) {
                    AccuratLogger.log(AccuratLogger.SDK_FLOW, "[ENTER] Storing triggered geofence " + geofence.getId());
                    TriggeredGeofence.Companion.storeTriggeredGeofence(
                            geofence,
                            GeofenceType.ENTER,
                            location.getTime()
                    );
                } else {
                    AccuratLogger.log(AccuratLogger.SDK_FLOW, "[ENTER] Posting a notification for " + geofence.getId());
                    postNotification(context, geofence);
                }

                return null;
            });

            List<AccuratGeofence> otherGeofences = CollectionUtilsKt.filter(triggeringGeofences, geofence -> !GeofenceType.ENTER.matches(geofence.getType()));
            DatabaseGeofence.Companion.storeAccuratGeofences(otherGeofences, location, context);
        } else {
            AccuratLogger.log(AccuratLogger.NONE, "No geofences triggered");
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".onLocationChanged()");

            return;
        }

        // Step 4 - Trigger notifications
        /*AccuratLogger.log(AccuratLogger.SDK_FLOW, "Step 4 - Find and show notifications for the triggered geofences");
        for (AccuratGeofence geofence : triggeringGeofences) {
            GeofencesManager.showNotificationForGeofence(context, geofence, success -> {
                if (success) {
                    AccuratLogger.log(AccuratLogger.NOTIFICATION, "Successfully posted notification for geofence with ID " + geofence.getId());
                } else {
                    AccuratLogger.log(AccuratLogger.WARNING, "Failed to post notification for geofence with ID " + geofence.getId());
                }
            });
        }*/

        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".onLocationChanged()");
    }

    private static void postNotification(Context context, AccuratGeofence geofence) {
        GeofencesManager.showNotificationForGeofence(context, geofence, success -> {
            if (success) {
                AccuratLogger.log(AccuratLogger.NOTIFICATION, "Successfully posted notification for geofence with ID " + geofence.getId());
            } else {
                AccuratLogger.log(AccuratLogger.WARNING, "Failed to post notification for geofence with ID " + geofence.getId());
            }
        });
    }
    // </editor-fold>

    // <editor-fold desc="Validation">
    private static boolean isUsable(Location location) {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".isUsable()");
        if (location == null) {
            // Can't use something which doesn't exist
            AccuratLogger.log(AccuratLogger.NONE, "Location is null");
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".isUsable() = false");

            return false;
        }

        // Step 1 - Check the horizontal accuracy of the location
        AccuratLogger.log(AccuratLogger.SDK_FLOW, "Step 1 - Checking location accuracy");
        if (location.hasAccuracy() && location.getAccuracy() > Constants.CUSTOM_GEOFENCE_DEFAULT_HORIZONTAL_ACCURACY_BOUNDARY) {
            // Accuracy is too low, completely ignore this location
            AccuratLogger.log(AccuratLogger.NONE, "Accuracy is too low (" + location.getAccuracy() + " > " + Constants.CUSTOM_GEOFENCE_DEFAULT_HORIZONTAL_ACCURACY_BOUNDARY + ")");
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".isUsable() = false");

            return false;
        } else {
            // The location is accurat enough to be used in future isUsable()-calculations (or accuracy doesn't exist), so store it
            if (!location.hasAccuracy()) {
                AccuratLogger.log(AccuratLogger.NONE, "Location has no accuracy, skipping step");
            } else {
                AccuratLogger.log(AccuratLogger.NONE, "Accuracy is suitable, continuing (" + location.getAccuracy() + " <= " + Constants.CUSTOM_GEOFENCE_DEFAULT_HORIZONTAL_ACCURACY_BOUNDARY + ")");
            }
        }

        GeofenceLocation previousLocation = GeofenceLocation.fromJson(storage.getString(StorageKeys.ACCURAT_CUSTOM_GEOFENCES_LAST_LOCATION, null));

        // Step 2 - Checking elapsed time since previous location
        AccuratLogger.log(AccuratLogger.SDK_FLOW, "Step 2 - Checking elapsed time since previous location");
        // Only do checks with the previous location if one exists
        if (previousLocation != null && previousLocation.timestamp + Constants.CUSTOM_GEOFENCE_DEFAULT_GEOFENCE_THROTTLE * 1000 >= location.getTime()) {
            // We've already processed a more recent location, or a location less than 5 seconds ago, so we should completely ignore this location
            AccuratLogger.log(AccuratLogger.NONE, "We've already processed a more recent location, or one less than " + Constants.CUSTOM_GEOFENCE_DEFAULT_GEOFENCE_THROTTLE + " seconds ago");
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".isUsable() = false");

            return false;
        } else {
            AccuratLogger.log(AccuratLogger.NONE, "The elapsed time is acceptable");
        }

        AccuratLogger.log(AccuratLogger.NONE, "The current location is usable as a previous location in a future geofence check, storing it");
        storeLastLocation(location);
        AccuratLogger.log(AccuratLogger.NONE, "Stored location as new last location");

        // Step 3 - Check the displacement from the previous geofence location
        AccuratLogger.log(AccuratLogger.SDK_FLOW, "Step 3 - Check the displacement from the previous geofence location");
        if (previousLocation != null) {
            double displacement = distance(location, previousLocation);
            if (previousLocation.horizontalAccuracy <= Constants.CUSTOM_GEOFENCE_DEFAULT_HORIZONTAL_ACCURACY_BOUNDARY && displacement <= Constants.CUSTOM_GEOFENCE_DEFAULT_DISPLACEMENT_BOUNDARY) {
                // Displacement is too small, don't trigger a geofence, but save it as the last known location
                AccuratLogger.log(AccuratLogger.NONE, "Displacement is too small (" + displacement + " <= " + Constants.CUSTOM_GEOFENCE_DEFAULT_DISPLACEMENT_BOUNDARY + ")");
                AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".isUsable() = false");
                // Should not store the last location here, or all displacements might be too small

                return false;
            } else {
                AccuratLogger.log(AccuratLogger.NONE, "Displacement is large enough (" + displacement + " > " + Constants.CUSTOM_GEOFENCE_DEFAULT_DISPLACEMENT_BOUNDARY + ")");
            }
        } else {
            AccuratLogger.log(AccuratLogger.NONE, "Previous location is null, can't execute displacement check");
        }

        // Disabled as requested by https://accurat.atlassian.net/browse/AC-7610 - Eenvoudigere Entry-exit notificaties
        /*// Step 4 - Check the speed
        AccuratLogger.log(AccuratLogger.SDK_FLOW, "Step 4 - Check the speed");
        AccuratLogger.log(AccuratLogger.NONE, "Info (not yet used in calculation): Speed from location is: " + location.getSpeed());
        if (location.hasSpeed() && location.getSpeed() > 0) {
            AccuratLogger.log(AccuratLogger.NONE, "Using speed from location");
            // Use the location's speed value
            float speed = location.getSpeed();
            if (speed > Constants.CUSTOM_GEOFENCE_DEFAULT_SPEED_BOUNDARY) {
                // Speed is too high, don't trigger a geofence, but save it as the last known location
                AccuratLogger.log(AccuratLogger.NONE, "Speed (from location) is too high (" + speed + " > " + Constants.CUSTOM_GEOFENCE_DEFAULT_SPEED_BOUNDARY + ")");
                AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".isUsable() = false");

                return false;
            } else {
                AccuratLogger.log(AccuratLogger.NONE, "Speed (from location) is usable (0 < " + speed + " <= " + Constants.CUSTOM_GEOFENCE_DEFAULT_SPEED_BOUNDARY + ")");
            }
        } else if (previousLocation != null) {
            AccuratLogger.log(AccuratLogger.NONE, "Using calculated speed");
            double speed = speed(location, previousLocation);
            if (speed > Constants.CUSTOM_GEOFENCE_DEFAULT_SPEED_BOUNDARY) {
                // Own calculated speed is too high, don't trigger a geofence, but save it as the last known location
                AccuratLogger.log(AccuratLogger.NONE, "Speed (calculated) is too high (" + speed + " > " + Constants.CUSTOM_GEOFENCE_DEFAULT_SPEED_BOUNDARY + ")");
                AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".isUsable() = false");

                return false;
            } else {
                AccuratLogger.log(AccuratLogger.NONE, "Speed (calculated) is usable (" + speed + " <= " + Constants.CUSTOM_GEOFENCE_DEFAULT_SPEED_BOUNDARY + ")");
            }
        } else {
            AccuratLogger.log(AccuratLogger.NONE, "Location's speed is unavailable and previous location is null, skipping step");
        }*/

        AccuratLogger.log(AccuratLogger.NONE, "Location is usable");
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".isUsable() = true");

        return true;
    }
    // </editor-fold>

    // <editor-fold desc="Geofences">
    private static List<AccuratGeofence> findTriggeringGeofences(Location location) {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".findTriggeringGeofences()");
        List<AccuratGeofence> geofences = CustomGeofenceStorageManager.getMonitoredGeofences();
        if (geofences == null || geofences.isEmpty()) {
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".findTriggeringGeofences()");
            return new ArrayList<>();
        }

        List<AccuratGeofence> triggers = new ArrayList<>();
        for (AccuratGeofence geofence : geofences) {
            if (geofence.contains(location)) {
                triggers.add(geofence);
            }
        }

        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".findTriggeringGeofences()");
        return triggers;
    }
    // </editor-fold>

    // <editor-fold desc="Helpers">

    /**
     * The distance between locations, in m.
     *
     * @param location         The current location
     * @param geofenceLocation The previous location
     * @return Distance, in m
     */
    private static double distance(@NonNull Location location, @NonNull GeofenceLocation geofenceLocation) {
        return LocationUtils.distance(location.getLongitude(), location.getLatitude(), geofenceLocation.longitude, geofenceLocation.latitude);
    }

    /**
     * The speed between locations, in m/s.
     *
     * @param location         The current location
     * @param geofenceLocation the previous location
     * @return Speed, in m/s
     */
    private static double speed(@NonNull Location location, @NonNull GeofenceLocation geofenceLocation) {
        return distance(location, geofenceLocation) / (location.getTime() - geofenceLocation.timestamp) * 1000;
    }

    private static void storeLastLocation(Location location) {
        storage.setValue(StorageKeys.ACCURAT_CUSTOM_GEOFENCES_LAST_LOCATION, new GeofenceLocation(location).toString())
                .commit();
    }
    // </editor-fold>
}
