package ai.accurat.sdk.core;

import android.content.Context;
import android.location.Location;
import android.os.AsyncTask;
import android.util.Pair;

import java.util.ArrayList;
import java.util.List;

import ai.accurat.sdk.callbacks.AccuratCompletionCallback;
import ai.accurat.sdk.config.Configuration;
import ai.accurat.sdk.constants.StorageKeys;

public class CustomGeofenceSelectionManager {

    // <editor-fold desc="Properties">
    private static final String TAG = CustomGeofenceSelectionManager.class.getSimpleName();

    private static MultiProcessStorage storage;
    // </editor-fold>

    // <editor-fold desc="Initialisation">
    public static void init(Context context) {
        if (!isInitialized()) {
            AccuratLogger.init(context);
            AccuratLogger.log(AccuratLogger.SDK_FLOW, "Initialising " + TAG);
            CustomGeofenceStorageManager.init(context);
            storage = MultiProcessStorage.getStorage(context, StorageKeys.ACCURAT_MULTI_PROCESS_STORAGE);
        }
    }

    private static boolean isInitialized() {
        return storage != null;
    }

    private static void checkInitialized() {
        if (isInitialized()) {
            return;
        }

        throw new IllegalStateException(TAG + " has not yet been initialised.");
    }
    // </editor-fold>

    // <editor-fold desc="Public interface">
    public static void update(Location location, AccuratCompletionCallback callback) {
        checkInitialized();
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".update()");
        try {
            new CalculateSelectionTask().execute(new Pair<>(location, callback));
        } catch (Exception e) {
            AccuratLogger.log(AccuratLogger.ERROR, "Failed to start CalculateSelectionTask: " + e.getMessage());
        }
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".update()");
    }
    // </editor-fold>

    // <editor-fold desc="Helpers">
    private static AccuratGeofence calculateMetaGeofence(List<AccuratGeofence> geofences) {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".calculateMetaGeofence()");
        if (geofences == null || geofences.isEmpty()) {
            return null;
        }

        AccuratGeofence firstGeofence = geofences.get(0);
        double minLatitude = firstGeofence.getLatitude();
        double maxLatitude = firstGeofence.getLatitude();
        double minLongitude = firstGeofence.getLongitude();
        double maxLongitude = firstGeofence.getLongitude();

        for (int i = 1; i < geofences.size(); i++) {
            AccuratGeofence geofence = geofences.get(i);
            // Update latitudes
            double latitude = geofence.getLatitude();
            if (latitude < minLatitude) {
                minLatitude = latitude;
            } else if (latitude > maxLatitude) {
                maxLatitude = latitude;
            }

            // Update longitudes
            double longitude = geofence.getLongitude();
            if (longitude < minLongitude) {
                minLongitude = longitude;
            } else if (longitude > maxLongitude) {
                maxLongitude = longitude;
            }
        }

        double centerLatitude = (minLatitude + maxLatitude) / 2;
        double centerLongitude = (minLongitude + maxLongitude) / 2;
        double widthRadius = LocationUtils.distance(centerLongitude, centerLatitude, minLongitude, centerLatitude);
        double heightRadius = LocationUtils.distance(centerLongitude, centerLatitude, centerLongitude, minLatitude);

        double rectangleRadius = Math.sqrt(Math.pow(widthRadius, 2) + Math.pow(heightRadius, 2));

        AccuratGeofence metaGeofence = new AccuratGeofence("meta_geofence", centerLatitude, centerLongitude, null, null, (int) rectangleRadius);
        AccuratLogger.log(AccuratLogger.GEOFENCE, "Calculated new meta geofence: " + metaGeofence.getLogString());

        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".calculateMetaGeofence()");
        return metaGeofence;
    }

    private static SearchBox calculateStartingSearchBox(Location location) {
        if (location == null) {
            return null;
        }

        double radiusAtLatitude = Configuration.EARTH_EQUATORIAL_RADIUS_METERS * Math.abs(Math.cos(location.getLatitude()));
        double angularWidth = Configuration.DEFAULT_GEOFENCE_SEARCHBOX_WIDTH_METER / (2 * Math.PI * radiusAtLatitude);
        double angularHeight = Configuration.DEFAULT_GEOFENCE_SEARCHBOX_HEIGHT_METER / (2 * Math.PI * Configuration.EARTH_POLE_RADIUS_METERS);

        return new SearchBox(
                location.getLatitude() - angularHeight / 2,
                location.getLatitude() + angularHeight / 2,
                location.getLongitude() - angularWidth / 2,
                location.getLongitude() + angularWidth / 2
        );
    }

    private static Pair<List<AccuratGeofence>, Integer> selectClosestGeofences(Location location) {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".selectClosestGeofences()");
        SearchBox searchBox = calculateStartingSearchBox(location);
        int iterationCount = 1;
        AccuratLogger.log(AccuratLogger.NONE, "Iteration " + iterationCount + ", searchBox " + searchBox);
        List<AccuratGeofence> selectedGeofences = CustomGeofenceStorageManager.getGeofencesInSearchBox(searchBox);
        AccuratLogger.log(AccuratLogger.NONE, "Selected " + (selectedGeofences == null ? 0 : selectedGeofences.size()) + " geofences");
        while(selectedGeofences != null && selectedGeofences.size() > Configuration.MAX_MONITORED_GEOFENCE_COUNT) {
            iterationCount++;
            searchBox.halve();
            AccuratLogger.log(AccuratLogger.NONE, "Iteration " + iterationCount + ", searchBox " + searchBox);
            if(searchBox.isSmallerThan(Configuration.MIN_GEOFENCE_SEARCHBOX_WIDTH_METER, Configuration.MIN_GEOFENCE_SEARCHBOX_HEIGHT_METER)) {
                AccuratLogger.log(AccuratLogger.NONE, "Search box has become to small");
                // The search box has reached it's minimum size, return the first MAX_MONITORED_GEOFENCE_COUNT geofence
                // (with the size being so small, it doesn't really matter which ones)
                selectedGeofences = selectedGeofences.subList(0, Configuration.MAX_MONITORED_GEOFENCE_COUNT);
                AccuratLogger.log(AccuratLogger.NONE, "Selected " + selectedGeofences.size() + " geofences");

                break;
            }
            selectedGeofences = CustomGeofenceStorageManager.getGeofencesInSearchBox(searchBox);
            AccuratLogger.log(AccuratLogger.NONE, "Selected " + (selectedGeofences == null ? 0 : selectedGeofences.size()) + " geofences");
        }

        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".selectClosestGeofences()");
        return new Pair<>(selectedGeofences == null ? new ArrayList<>() : selectedGeofences, iterationCount);
    }
    // </editor-fold>

    private static class CalculateSelectionTask extends AsyncTask<Pair<Location, AccuratCompletionCallback>, Void, Void> {

        public static final String TAG = CalculateSelectionTask.class.getSimpleName();

        @SafeVarargs
        @Override
        protected final Void doInBackground(Pair<Location, AccuratCompletionCallback>... pairs) {
            AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".doInBackground()");
            if(pairs == null || pairs.length == 0 || pairs[pairs.length - 1].first == null) {
                AccuratLogger.log(AccuratLogger.WARNING, "Can't calculate geofence selection without location");
                AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".doInBackground()");
                return null;
            }
            Location location = pairs[pairs.length - 1].first;
            long startTime = System.currentTimeMillis();
            int iterationCount = 0;
            long geofenceCount = 0;
            AccuratGeofence metaGeofence = null;

            long allGeofencesCount = CustomGeofenceStorageManager.getStoredGeofenceCount();
            if (allGeofencesCount == 0) {
                // No geofences in storage, clear the meta and monitored geofences
                AccuratLogger.log(AccuratLogger.SDK_FLOW, "No geofences in storage, selecting none");
                CustomGeofenceStorageManager.storeMetaGeofence(null);
                CustomGeofenceStorageManager.storeMonitoredGeofences(null);
            } else if (allGeofencesCount <= Configuration.MAX_MONITORED_GEOFENCE_COUNT) {
                // A small amount of geofences in storage, build the meta geofence and return all as new monitored geofences
                AccuratLogger.log(AccuratLogger.SDK_FLOW, "A small amount of geofences in storage, selecting all");
                List<AccuratGeofence> allGeofences = CustomGeofenceStorageManager.getGeofences();
                geofenceCount = allGeofencesCount;
                metaGeofence = calculateMetaGeofence(allGeofences);
                CustomGeofenceStorageManager.storeMetaGeofence(metaGeofence);
                CustomGeofenceStorageManager.storeMonitoredGeofences(allGeofences);
            } else {
                // Too much geofences in storage, find a good selection
                AccuratLogger.log(AccuratLogger.SDK_FLOW, "Too much geofences in storage, calculating selection");
                Pair<List<AccuratGeofence>, Integer> geofenceSelection = selectClosestGeofences(location);
                geofenceCount = geofenceSelection.first.size();
                iterationCount = geofenceSelection.second;
                metaGeofence = calculateMetaGeofence(geofenceSelection.first);
                CustomGeofenceStorageManager.storeMetaGeofence(metaGeofence);
                CustomGeofenceStorageManager.storeMonitoredGeofences(geofenceSelection.first);
            }

            long endTime = System.currentTimeMillis();
            AccuratLogger.log(
                    AccuratLogger.GEOFENCE,
                    "Calculated selection with " + geofenceCount + " geofences in "
                            + ((endTime - startTime) / 1000) + "ms and "
                            + iterationCount + " iterations"
                            + (
                            metaGeofence == null
                                    ? ""
                                    : " inside meta geofence " + metaGeofence.getLogString()
                    )
            );

            AccuratCompletionCallback callback = pairs[pairs.length - 1].second;
            if(callback != null) {
                callback.onCompleted(true);
            }

            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".doInBackground()");
            return null;
        }
    }
}
