package ai.accurat.sdk.core;

import android.content.Context;
import androidx.annotation.NonNull;
import android.text.TextUtils;

import org.json.JSONArray;
import org.json.JSONException;

import java.util.ArrayList;
import java.util.List;

import ai.accurat.sdk.constants.StorageKeys;

/**
 * A class responsible for storing and accessing geofences in the database and local storage.
 */
public class CustomGeofenceStorageManager {

    // <editor-fold desc="Properties">
    private static final String TAG = CustomGeofenceStorageManager.class.getSimpleName();

    private static AccuratDatabase database;
    private static MultiProcessStorage storage;

    private static List<AccuratGeofence> monitoredGeofences;// A list of the closest geofences
    // </editor-fold>

    // <editor-fold desc="Initialisation">
    public static void init(Context context) {
        if (!isInitialized()) {
            AccuratLogger.init(context);
            AccuratLogger.log(AccuratLogger.SDK_FLOW, "Initialising " + TAG);
            database = new AccuratDatabase(DatabaseHelper.getInstance(context));
            storage = MultiProcessStorage.getStorage(context, StorageKeys.ACCURAT_MULTI_PROCESS_STORAGE);
        }
    }

    private static boolean isInitialized() {
        return database != null && storage != null;
    }

    private static void checkInitialized() {
        if (isInitialized()) {
            return;
        }

        throw new IllegalStateException(TAG + " has not yet been initialised.");
    }
    // </editor-fold>

    // <editor-fold desc="Public interface">
    public static List<AccuratGeofence> getGeofences() {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".getGeofence()");
        checkInitialized();
        try {
            List<AccuratGeofence> geofences = database.getGeofences();
            AccuratLogger.log(AccuratLogger.DATABASE, "Retrieved " + geofences.size() + " geofences from the database");
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".getGeofence()");
            return geofences;
        } catch (Exception e) {
            AccuratLogger.log(AccuratLogger.ERROR, "Could not get geofences from the database: " + e.getMessage());
        }

        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".getGeofence()");
        return null;
    }

    public static List<AccuratGeofence> getGeofencesInSearchBox(SearchBox searchBox) {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".getGeofencesInSearchBox()");
        checkInitialized();
        try {
            List<AccuratGeofence> geofences = database.getGeofencesWithinBox(searchBox);
            AccuratLogger.log(AccuratLogger.DATABASE, "Retrieved " + geofences.size() + " geofences from the database for search box " + searchBox.toString());
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".getGeofencesInSearchBox()");

            return geofences;
        } catch (Exception e) {
            AccuratLogger.log(AccuratLogger.ERROR, "Could not get geofences from the database: " + e.getMessage());
        }

        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".getGeofencesInSearchBox()");
        return null;
    }

    public static void storeServerGeofences(List<AccuratGeofence> geofences) {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".storeServerGeofences()");
        checkInitialized();

        // Check if there's anything to store
        if (geofences == null || geofences.isEmpty()) {
            AccuratLogger.log(AccuratLogger.WARNING, "No geofences received to store, keeping old geofences");
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".storeServerGeofences()");
            return;
        }

        // Clear the old geofences from the database
        if (!clearServerGeofences()) {
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".storeServerGeofences()");
            return;
        }

        // Add the new geofences to the database
        if (!addGeofences(geofences)) {
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".storeServerGeofences()");
            return;
        }

        // Log the database size
        logStoreGeofenceCount();
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".storeServerGeofences()");
    }

    public static boolean clearServerGeofences() {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".clearServerGeofences()");
        checkInitialized();

        try {
            AccuratLogger.log(AccuratLogger.SDK_FLOW, "Clearing old geofences from the database");
            long removeCount = database.removeGeofences();
            AccuratLogger.log(AccuratLogger.DATABASE, "Removed " + removeCount + " old geofences from the database");

            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".clearServerGeofences()");
            return true;
        } catch (Exception e) {
            AccuratLogger.log(AccuratLogger.ERROR, "Could not clear geofences from the database: " + e.getMessage());
        }

        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".clearServerGeofences()");
        return false;
    }

    public static long getStoredGeofenceCount() {
        try {
            return database.getNumberOfGeofences();
        } catch (Exception e) {
            AccuratLogger.log(AccuratLogger.ERROR, "Could not get stored geofence count");
        }

        return 0;
    }

    public static void storeMonitoredGeofences(List<AccuratGeofence> geofences) {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".storeMonitoredGeofences()");
        checkInitialized();

        // Check if there's anything to store
        if(geofences == null) {
            AccuratLogger.log(AccuratLogger.WARNING, "No geofences received to store, keeping old geofences");
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".storeMonitoredGeofences()");
            return;
        }

        List<String> geofenceStrings = new ArrayList<>(geofences.size());
        for (AccuratGeofence geofence : geofences) {
            geofenceStrings.add(geofence.toString());
        }

        // Save the geofences to storage
        storage.setValue(StorageKeys.ACCURAT_CUSTOM_MONITORED_GEOFENCES, JSON.toJsonStringArray(geofenceStrings).toString())
        .commit();
        AccuratLogger.log(AccuratLogger.STORAGE, "Stored " + geofences.size() + " monitored geofences");
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".storeMonitoredGeofences()");
    }

    public static List<AccuratGeofence> getMonitoredGeofences() {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".getMonitoredGeofences()");
        checkInitialized();
        if (monitoredGeofences == null) {
            loadMonitoredGeofences();
        }

        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".getMonitoredGeofences()");
        return monitoredGeofences;
    }

    public static void storeMetaGeofence(AccuratGeofence geofence) {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".storeMetaGeofence()");
        if(geofence == null) {
            AccuratLogger.log(AccuratLogger.WARNING, "Meta geofence is null, removing it from storage");
            storage.remove(StorageKeys.ACCURAT_CUSTOM_META_GEOFENCE)
                    .commit();
        } else {
            storage.setValue(StorageKeys.ACCURAT_CUSTOM_META_GEOFENCE, geofence.toString())
            .commit();
            AccuratLogger.log(AccuratLogger.SDK_FLOW, "Stored meta geofence:  " + geofence.getLogString());
        }
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".storeMetaGeofence()");
    }

    public static AccuratGeofence getMetaGeofence() {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".getMetaGeofence()");
        try {
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".getMetaGeofence()");
            return AccuratGeofence.fromJson(storage.getString(StorageKeys.ACCURAT_CUSTOM_META_GEOFENCE, null));
        } catch (JSONException e) {
            AccuratLogger.log(AccuratLogger.JSON_ERROR, "Failed to load meta geofence from storage: " + e.getMessage());
        }

        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".getMetaGeofence()");
        return null;
    }
    // </editor-fold>

    // <editor-fold desc="Storage">
    private static void loadMonitoredGeofences() {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".loadMonitoredGeofences()");
        String jsonString = storage.getString(StorageKeys.ACCURAT_CUSTOM_MONITORED_GEOFENCES, null);

        // Check if there's anything to load
        if(TextUtils.isEmpty(jsonString)) {
            AccuratLogger.log(AccuratLogger.WARNING, "No monitored geofences in storage");
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".loadMonitoredGeofences()");
            monitoredGeofences = null;

            return;
        }

        try {
            JSONArray json = new JSONArray(jsonString);
            List<AccuratGeofence> geofences = new ArrayList<>(json.length());
            for (int i = 0; i < json.length(); i++) {
                geofences.add(AccuratGeofence.fromJson(json.getString(i)));
            }
            AccuratLogger.log(AccuratLogger.STORAGE, "Loaded " + geofences.size() + " monitored geofences from storage");
            monitoredGeofences = geofences;
        } catch (JSONException e) {
            AccuratLogger.log(AccuratLogger.JSON_ERROR, e.getMessage());
        }

        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".loadMonitoredGeofences()");
    }
    // </editor-fold>

    // <editor-fold desc="Database">
    private static boolean addGeofences(@NonNull List<AccuratGeofence> geofences) {
        try {
            AccuratLogger.log(AccuratLogger.SDK_FLOW, "Storing server geofences in the database");
            database.addGeofences(geofences);
            AccuratLogger.log(AccuratLogger.DATABASE, "Stored " + geofences.size() + " geofences from the server in the database");

            return true;
        } catch (Exception e) {
            AccuratLogger.log(AccuratLogger.ERROR, "Could not add " + geofences.size() + " server geofences to database: " + e.getMessage());
        }

        return false;
    }

    private static void logStoreGeofenceCount() {
        checkInitialized();
        try {
            long geofenceCount = database.getNumberOfGeofences();
            AccuratLogger.log(AccuratLogger.DATABASE, "Database contains " + geofenceCount + " geofences");
        } catch (Exception e) {
            AccuratLogger.log(AccuratLogger.ERROR, "Could not get stored geofence count");
        }
    }
    // </editor-fold>
}
