package ai.accurat.sdk.core;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.List;

/**
 * A utility class for JSON (de)serialisation.
 *
 * @author Kenneth Saey
 * @since 19-02-2019 14:04.
 */
public abstract class JSON implements JSONConvertable {

    // <editor-fold desc="Constants">
    public static final String VALUE_NULL_STRING = "null";
    // </editor-fold>

    // <editor-fold desc="Abstract functions">
    public abstract void update(JSONObject json);

    public abstract JSONObject toJson();
    // </editor-fold>

    public JSONObject toServerJson() {
        return toJson();
    }

    @Override
    public String toString() {
        return toJson().toString();
    }

    // <editor-fold desc="Checks">
    public static boolean check(JSONObject json, String key) {
        return key == null || (json.has(key) && !json.isNull(key));
    }
    // </editor-fold>

    // <editor-fold desc="Load single primitive values">
    public static String load(JSONObject json, String key) {
        return load(json, key, (String) null);
    }


    public static int load(JSONObject json, String key, int defaultValue) {
        if (check(json, key)) {
            try {
                return json.getInt(key);
            } catch (JSONException e) {
                e.printStackTrace();
            }
        }
        return defaultValue;
    }

    public static long load(JSONObject json, String key, long defaultValue) {
        if (check(json, key)) {
            try {
                return json.getLong(key);
            } catch (JSONException e) {
                e.printStackTrace();
            }
        }
        return defaultValue;
    }

    public static float load(JSONObject json, String key, float defaultValue) {
        if (check(json, key)) {
            try {
                return (float) json.getDouble(key);
            } catch (JSONException e) {
                e.printStackTrace();
            }
        }
        return defaultValue;
    }

    public static double load(JSONObject json, String key, double defaultValue) {
        if (check(json, key)) {
            try {
                return json.getDouble(key);
            } catch (JSONException e) {
                e.printStackTrace();
            }
        }
        return defaultValue;
    }

    public static Double load(JSONObject json, String key, Double defaultValue) {
        if (key != null && json.has(key)) {
            if (json.isNull(key)) {
                return null;
            }
            try {
                return json.getDouble(key);
            } catch (JSONException e) {
                e.printStackTrace();
            }
        }
        return defaultValue;
    }

    public static boolean load(JSONObject json, String key, boolean defaultValue) {
        if (check(json, key)) {
            try {
                return json.getBoolean(key);
            } catch (JSONException e) {
                e.printStackTrace();
            }
        }
        return defaultValue;
    }

    public static String load(JSONObject json, String key, String defaultValue) {
        if (check(json, key)) {
            try {
                return json.getString(key);
            } catch (JSONException e) {
                e.printStackTrace();
            }
        }
        return defaultValue;
    }

    public static <E extends Enum<E>> E load(JSONObject json, String key, E defaultValue, Class<E> cls) {
        if (check(json, key)) {
            try {
                return E.valueOf(cls, json.getString(key));
            } catch (JSONException e) {
                e.printStackTrace();
            }
        }

        return defaultValue;
    }

    public static JSONObject load(JSONObject json, String key, JSONObject defaultValue) {
        if (check(json, key)) {
            try {
                return json.getJSONObject(key);
            } catch (JSONException e) {
                e.printStackTrace();
            }
        }
        return defaultValue;
    }

    public static JSONArray load(JSONObject json, String key, JSONArray defaultValue) {
        if (check(json, key)) {
            try {
                return json.getJSONArray(key);
            } catch (JSONException e) {
                e.printStackTrace();
            }
        }
        return defaultValue;
    }
    // </editor-fold>

    // <editor-fold desc="Load single JSON values">
    public static <E extends JSON> E load(JSONObject json, String key, E defaultValue, Class<E> cls) {
        if (check(json, key)) {
            try {
                E instance = cls.newInstance();
                instance.update(key == null ? json : json.getJSONObject(key));
                return instance;
            } catch (JSONException | InstantiationException | IllegalAccessException e) {
                e.printStackTrace();
            }
        }
        return defaultValue;
    }
    // </editor-fold>

    // <editor-fold desc="Load array of primitive values">
    public static ArrayList<String> loadStringList(JSONObject json, String key, ArrayList<String> defaultValue) {
        if (check(json, key)) {
            try {
                JSONArray valuesArray = json.getJSONArray(key);
                ArrayList<String> values = new ArrayList<>();
                for (int i = 0; i < valuesArray.length(); i++) {
                    values.add(valuesArray.getString(i));
                }
                return values;
            } catch (JSONException e) {
                e.printStackTrace();
            }
        }
        return defaultValue;
    }

    public static ArrayList<Integer> loadIntegerList(JSONObject json, String key, ArrayList<Integer> defaultValues) {
        if (check(json, key)) {
            try {
                JSONArray valuesArray = json.getJSONArray(key);
                ArrayList<Integer> values = new ArrayList<>();
                for (int i = 0; i < valuesArray.length(); i++) {
                    values.add(valuesArray.getInt(i));
                }
                return values;
            } catch (JSONException e) {
                e.printStackTrace();
            }
        }
        return defaultValues;
    }
    // </editor-fold>

    // <editor-fold desc="Load array of JSON values">
    public static <E extends JSON> ArrayList<E> loadObjectList(JSONObject json, String key, ArrayList<E> defaultValue, Class<E> cls) {
        if (check(json, key)) {
            try {
                JSONArray valuesArray = json.getJSONArray(key);
                ArrayList<E> values = new ArrayList<>();
                for (int i = 0; i < valuesArray.length(); i++) {
                    E instance = cls.newInstance();
                    instance.update(valuesArray.getJSONObject(i));
                    values.add(instance);
                }
                return values;
            } catch (JSONException | InstantiationException | IllegalAccessException e) {
                e.printStackTrace();
            }
        }
        return defaultValue;
    }

    public static <E extends JSON> ArrayList<E> loadObjectList(JSONArray json, ArrayList<E> defaultValue, Class<E> cls) {
        try {
            ArrayList<E> values = new ArrayList<>();
            for (int i = 0; i < json.length(); i++) {
                E instance = cls.newInstance();
                instance.update(json.getJSONObject(i));
                values.add(instance);
            }
            return values;
        } catch (JSONException | InstantiationException | IllegalAccessException e) {
            e.printStackTrace();
        }
        return defaultValue;
    }
    // </editor-fold>

    // <editor-fold desc="Jsonify an array of primitive values">
    public static JSONArray toJsonStringArray(List<String> values) {
        JSONArray json = new JSONArray();
        if (values != null && !values.isEmpty()) {
            for (String value : values) {
                json.put(value);
            }
        }
        return json;
    }

    public static JSONArray toJsonIntegerArray(List<Integer> values) {
        JSONArray json = new JSONArray();
        if (values != null && !values.isEmpty()) {
            for (Integer value : values) {
                json.put(value);
            }
        }
        return json;
    }
    // </editor-fold>

    // <editor-fold desc="Jsonify an array of JSON values">
    public static <E extends JSONConvertable> JSONArray toJsonObjectArray(List<E> values) {
        JSONArray json = new JSONArray();
        if (values != null && !values.isEmpty()) {
            for (E value : values) {
                json.put(value.toJson());
            }
        }
        return json;
    }

    public static <E extends JSONConvertable> JSONArray toServerJsonObjectArray(List<E> values) {
        JSONArray json = new JSONArray();
        if (values != null && !values.isEmpty()) {
            for (E value : values) {
                json.put(value.toServerJson());
            }
        }
        return json;
    }
    // </editor-fold>

    // <editor-fold desc="Jsonify a JSON value">
    public static <E extends JSON> JSONObject toJson(E object) {
        if (object == null) {
            return null;
        } else {
            return object.toJson();
        }
    }
    // </editor-fold>
}