package ai.accurat.sdk.data;

import ai.accurat.sdk.core.AccuratLogger;
import ai.accurat.sdk.data.models.Setting;
import ai.accurat.sdk.managers.RealmManager;
import io.realm.RealmObject;

public class RealmHelper {

    public static final String TAG = RealmHelper.class.getSimpleName();
    public static final long NO_ID = Long.MIN_VALUE;

    // <editor-fold desc="Setting helpers">
    public static void storeSetting(Setting setting) {
        if (setting == null) {
            return;
        }

        RealmManager.executeTransaction(realm -> {
            realm.copyToRealmOrUpdate(setting);
        });
    }

    public static void storeSetting(String key, String value) {
        storeSetting(new Setting(key, value));
    }

    public static void storeSetting(String key, boolean value) {
        storeSetting(new Setting(key, value));
    }

    public static void storeSetting(String key, int value) {
        storeSetting(new Setting(key, value));
    }

    public static void storeSetting(String key, long value) {
        storeSetting(new Setting(key, value));
    }

    public static Setting loadSetting(String key) {
        return RealmManager.getAccuratInstance()
                .where(Setting.class)
                .equalTo(Setting.RealmColumns.KEY, key)
                .findFirst();
    }

    public static String loadStringSetting(String key, String defaultValue) {
        Setting setting = loadSetting(key);

        return setting == null ? defaultValue : setting.stringValue;
    }

    public static String loadStringSetting(String key) {
        return loadStringSetting(key, null);
    }

    public static boolean loadBooleanSetting(String key, boolean defaultValue) {
        Setting setting = loadSetting(key);
        if (!key.equals(Setting.Keys.Config.HAS_DEBUG_LOGS_ENABLED)) {
            AccuratLogger.log(AccuratLogger.NONE, TAG + "loadBooleanSetting(" + key + ") = " + (setting == null ? defaultValue + " (defaultValue)" : setting.booleanValue));
        }

        return setting == null ? defaultValue : setting.booleanValue;
    }

    public static boolean loadBooleanSetting(String key) {
        return loadBooleanSetting(key, false);
    }

    public static int loadIntSetting(String key, int defaultValue) {
        Setting setting = loadSetting(key);

        return setting == null ? defaultValue : setting.intValue;
    }

    public static int loadIntSetting(String key) {
        return loadIntSetting(key, 0);
    }

    public static long loadLongSetting(String key, long defaultValue) {
        Setting setting = loadSetting(key);

        return setting == null ? defaultValue : setting.longValue;
    }

    public static long loadLongSetting(String key) {
        return loadLongSetting(key, 0);
    }
    // </editor-fold>

    // <editor-fold desc="General helpers">
    public static <E extends RealmObject> void deleteFromRealm(Class<E> clazz, String idField, String id) {
        RealmManager.executeTransaction(realm -> {
            RealmObject model = realm.where(clazz)
                    .equalTo(idField, id)
                    .findFirst();
            if (model != null) {
                model.deleteFromRealm();
            } else {
                AccuratLogger.log(AccuratLogger.WARNING, "deleteFromRealm: " + clazz.getSimpleName() + " with id " + id + " not found");
            }
        });
    }
    // </editor-fold>
}
