package ai.accurat.sdk.data.models;

import androidx.annotation.NonNull;

import java.util.ArrayList;
import java.util.List;

import ai.accurat.sdk.core.AccuratLogger;
import io.realm.RealmDictionary;
import io.realm.RealmObject;
import io.realm.annotations.PrimaryKey;

public class AccuratConfiguration extends RealmObject {

    // <editor-fold desc="Fields">
    @PrimaryKey
    public String username;
    public String password;
    public String notificationTargetPackage;
    public String notificationTargetClass;
    public boolean gdprConsent;
    public boolean locationPermission;
    public String appVersion;
    public boolean hasDebugLogsEnabled = false;
    public int appVersionCode;
    public RealmDictionary<Boolean> featureSwitches = new RealmDictionary<>();
    // </editor-fold>

    // <editor-fold desc="Initialisation">
    public AccuratConfiguration() {
    }
    // </editor-fold>

    // <editor-fold desc="Fluent setters">
    public AccuratConfiguration withCredentials(String username, String password) {
        this.username = username;
        this.password = password;

        return this;
    }

    public AccuratConfiguration withNotificationTargetPackage(String packageName) {
        this.notificationTargetPackage = packageName;

        return this;
    }

    public AccuratConfiguration withNotificationTargetClass(String className) {
        this.notificationTargetClass = className;

        return this;
    }

    public AccuratConfiguration withFeatures(AccuratFeature... features) {
        for (AccuratFeature feature : features) {
            if (feature == null) {
                continue;
            }

            switch (feature) {
                case GDPR_CONSENT:
                    gdprConsent = true;
                    break;
                case LOCATION_PERMISSION:
                    locationPermission = true;
                    break;
            }
        }

        return this;
    }

    public AccuratConfiguration withAppVersion(String appVersion) {
        this.appVersion = appVersion;

        return this;
    }

    public AccuratConfiguration withDebugLogs(boolean enabled) {
        this.hasDebugLogsEnabled = enabled;

        return this;
    }

    public AccuratConfiguration enableDebugLogs() {
        return withDebugLogs(true);
    }

    public AccuratConfiguration disableDebugLogs() {
        return withDebugLogs(false);
    }

    public AccuratConfiguration withAppVersionCode(int appVersionCode) {
        this.appVersionCode = appVersionCode;

        return this;
    }

    public AccuratConfiguration withFeatureSwitches(AccuratFeatureSwitch... switches) {
        featureSwitches = new RealmDictionary<>();
        for (AccuratFeatureSwitch featureSwitch : switches) {
            featureSwitches.put(featureSwitch.name(), true);
        }

        AccuratLogger.log(AccuratLogger.NONE, "[AccuratConfiguration] Feature switches: " + featureSwitches.keySet());

        return this;
    }
    // </editor-fold>

    // <editor-fold desc="Getters">
    public String getUsername() {
        return username;
    }

    public String getPassword() {
        return password;
    }

    public String getNotificationTargetPackage() {
        return notificationTargetPackage;
    }

    public String getNotificationTargetClass() {
        return notificationTargetClass;
    }

    public boolean usesGdprConsentFeature() {
        return gdprConsent;
    }

    public boolean usesLocationPermissionFeature() {
        return locationPermission;
    }

    public AccuratFeature[] getFeatures() {
        List<AccuratFeature> features = new ArrayList<>();
        if (gdprConsent) {
            features.add(AccuratFeature.GDPR_CONSENT);
        }
        if (locationPermission) {
            features.add(AccuratFeature.LOCATION_PERMISSION);
        }

        return features.toArray(new AccuratFeature[0]);
    }

    public String getAppVersion() {
        return appVersion;
    }

    public boolean hasDebugLogsEnabled() {
        return hasDebugLogsEnabled;
    }

    public int getAppVersionCode() {
        return appVersionCode;
    }

    public RealmDictionary<Boolean> getFeatureSwitches() {
        return featureSwitches;
    }

    public boolean isFeatureSwitchEnabled(AccuratFeatureSwitch featureSwitch) {
        Boolean value = featureSwitches.get(featureSwitch.name());

        return value != null && value;
    }

    public boolean isTrackingTriggeredGeofences() {
        return isFeatureSwitchEnabled(AccuratFeatureSwitch.TRACK_TRIGGERED_GEOFENCES);
    }
    // </editor-fold>

    // <editor-fold desc="Helpers">
    private void clearFeatures() {
        gdprConsent = false;
        locationPermission = false;
    }

    public boolean hasUsername() {
        return username != null;
    }

    public boolean hasValidUsername() {
        return hasUsername() && !username.trim().isEmpty();
    }

    public boolean hasPassword() {
        return password != null;
    }

    public boolean hasValidPassword() {
        return hasPassword() && !password.trim().isEmpty();
    }
    // </editor-fold>

    public static final class RealmColumns {
        public static final String _TABLE = "AccuratConfiguration";

        public static final String USERNAME = "username";
        public static final String PASSWORD = "password";
        public static final String NOTIFICATION_TARGET_PACKAGE = "notificationTargetPackage";
        public static final String NOTIFICATION_TARGET_CLASS = "notificationTargetClass";
        public static final String GDPR_CONSENT = "gdprConsent";
        public static final String LOCATION_PERMISSION = "locationPermission";
        public static final String APP_VERSION = "appVersion";
        public static final String HAS_DEBUG_LOGS_ENABLED = "hasDebugLogsEnabled";
        public static final String APP_VERSION_CODE = "appVersionCode";
        public static final String FEATURE_SWITCHES = "featureSwitches";
    }

    @NonNull
    @Override
    public String toString() {
        return "{GDPR: " + gdprConsent + ", Location: " + locationPermission + ", FeatureSwitches: " + featureSwitches.keySet() + "}";
    }
}
