package ai.accurat.sdk.data.models

import ai.accurat.sdk.core.AccuratGeofence
import ai.accurat.sdk.core.AccuratLogger
import ai.accurat.sdk.core.LocationUtils
import ai.accurat.sdk.data.all
import ai.accurat.sdk.data.enums.GeofenceType
import ai.accurat.sdk.data.save
import ai.accurat.sdk.managers.RealmManager
import ai.accurat.sdk.workers.GeofenceDwellWorker
import android.content.Context
import android.location.Location
import io.realm.RealmObject
import io.realm.annotations.PrimaryKey
import io.realm.kotlin.where

/**
 * @author Kenneth
 * @since 2021-05-25
 */
open class DatabaseGeofence : RealmObject() {

    // <editor-fold desc="Fields">
    @PrimaryKey
    var id: String = ""
    var longitude: Double = 0.0
    var latitude: Double = 0.0
    var radius: Int = 0
    var enteredAt: Long = System.currentTimeMillis()
    // </editor-fold>

    // <editor-fold desc="Helpers">
    fun contains(longitude: Double, latitude: Double): Boolean {
        val distance = LocationUtils.distance(
            this.longitude,
            this.latitude,
            longitude,
            latitude
        )

        return distance <= radius
    }
    // </editor-fold>

    companion object {
        private fun fromAccuratGeofence(
            geofence: AccuratGeofence,
            location: Location
        ): DatabaseGeofence =
            DatabaseGeofence().apply {
                id = geofence.id
                longitude = geofence.longitude
                latitude = geofence.latitude
                radius = geofence.radius
                enteredAt = location.time
            }

        private fun all(): List<DatabaseGeofence> = DatabaseGeofence().all()

        fun storeAccuratGeofences(
            geofences: List<AccuratGeofence>,
            location: Location,
            context: Context
        ) {
            AccuratLogger.log(
                AccuratLogger.METHOD_START,
                "DatabaseGeofence.storeAccuratGeofences()"
            )
            geofences
                .filter {
                    AccuratLogger.log(
                        AccuratLogger.GEOFENCE,
                        "Checking if geofence '${it.id}' exists in database: ${exists(it.id)}"
                    )
                    !exists(it.id)
                }
                .map {
                    it to fromAccuratGeofence(it, location)
                }
                .forEach {
                    AccuratLogger.log(
                        AccuratLogger.GEOFENCE,
                        "Saving geofence ${it.second.id} in database"
                    )
                    it.second.save()
                    if (GeofenceType.EXIT.matches(it.first.type)) {
                        AccuratLogger.log(
                            AccuratLogger.SDK_FLOW,
                            "${it.second.id} is an EXIT-geofence, not launching GeofenceDwellWorker"
                        )
                        // Don't launch a DwellWorker if the geofence should only be triggered on EXIT
                    } else {
                        AccuratLogger.log(
                            AccuratLogger.SDK_FLOW,
                            "${it.second.id} is a DWELL-geofence, launching GeofenceDwellWorker"
                        )
                        GeofenceDwellWorker.launchRequest(context, it.second.id, it.first.dwellTime * 60L)
                    }
                }
            AccuratLogger.log(AccuratLogger.METHOD_END, "DatabaseGeofence.storeAccuratGeofences()")
        }

        fun findExitedGeofences(
            longitude: Double,
            latitude: Double
        ): List<DatabaseGeofence> {
            val enteredGeofences = all()
            AccuratLogger.log(
                AccuratLogger.GEOFENCE,
                "Currently dwelling in ${enteredGeofences.size} geofences: ${enteredGeofences.joinToString { it.id }}"
            )

            return enteredGeofences
                .filter {
                    !it.contains(longitude, latitude)
                }
        }

        fun exists(geofenceId: String): Boolean {
            val realm = RealmManager.getAccuratInstance()
            val exists = realm.where<DatabaseGeofence>()
                .equalTo(RealmColumns.ID, geofenceId)
                .count() > 0
            realm.close()

            return exists
        }
    }

    object RealmColumns {
        @JvmField
        val _TABLE: String = "DatabaseGeofence"

        const val ID: String = "id"
        const val LONGITUDE: String = "longitude"
        const val LATITUDE: String = "latitude"
        const val RADIUS: String = "radius"
        const val ENTERED_AT: String = "enteredAt"
    }
}