package ai.accurat.sdk.data.models

import ai.accurat.sdk.config.Configuration
import ai.accurat.sdk.constants.ApiKeys
import ai.accurat.sdk.data.enums.ConsentType
import ai.accurat.sdk.data.toInt
import io.realm.RealmObject
import io.realm.annotations.PrimaryKey
import org.json.JSONArray
import org.json.JSONException
import org.json.JSONObject

/**
 * @author Kenneth
 * @since 2020-11-10
 */
open class UserConsent : RealmObject() {

    // <editor-fold desc="Fields">
    @PrimaryKey
    private var typeString: String = ConsentType.UNKNOWN.name
    var type: ConsentType
        get() = ConsentType.values().first { it.name == typeString }
        set(value) {
            typeString = value.name
        }
    var accepted: Boolean = false
    var refuseCount: Int = 0
    var version: Long = 0
    // </editor-fold>

    fun getUploadData(adId: String): JSONObject? = try {
        JSONObject().apply {
            put(ServerKeys.AD_ID, adId)
            put(ServerKeys.ACCEPTED, accepted.toInt())
            put(ServerKeys.REFUSE_COUNT, refuseCount)
            put(ServerKeys.VERSION, version.toString())
            put(ServerKeys.TYPE, type.value)
        }
    } catch (e: JSONException) {
        e.printStackTrace()

        null
    }

    override fun toString(): String {
        return "{type: $type, accepted: $accepted, refuseCount: $refuseCount, version: $version}"
    }

    object RealmColumns {
        @JvmField
        val _TABLE: String = "UserConsent"

        const val TYPE_STRING: String = "typeString"
        const val ACCEPTED: String = "accepted"
        const val REFUSE_COUNT: String = "refuseCount"
        const val VERSION: String = "version"
    }

    object ServerKeys {
        const val AD_ID: String = "ad_id"
        const val TYPE: String = "type"
        const val ACCEPTED: String = "state"
        const val REFUSE_COUNT: String = "refuse_count"
        const val VERSION: String = "version"
    }

    companion object {
        fun fromServerJson(json: JSONObject): UserConsent = UserConsent().apply {
            type = ConsentType.fromValue(json.optString(ServerKeys.TYPE, type.value))
            accepted = json.optInt(ServerKeys.ACCEPTED) == 1
            refuseCount = json.optInt(ServerKeys.REFUSE_COUNT, refuseCount)
            version = json.optLong(ServerKeys.VERSION, version)
        }

        fun create(consentType: ConsentType, accepted: Boolean = false): UserConsent {
            return UserConsent().apply {
                type = consentType
                this.accepted = accepted
                version = Configuration.API_CONSENT_VERSION.toLong()
            }
        }
    }
}

fun List<UserConsent>.find(type: ConsentType): UserConsent? = find {
    it.type == type
}

fun List<UserConsent>.getUploadData(adId: String?): JSONObject? {
    if (adId == null) {
        return null
    }

    return try {
        val data = JSONArray().apply {
            this@getUploadData.map {
                it.getUploadData(adId)
            }.forEach { json ->
                put(json)
            }
        }
        JSONObject().apply {
            put(ApiKeys.PostConsents.Request.DATA, data)
        }
    } catch (e: JSONException) {
        e.printStackTrace()

        null
    }
}