package ai.accurat.sdk.workers

import ai.accurat.sdk.config.Configuration
import ai.accurat.sdk.core.AccuratDatabase
import ai.accurat.sdk.core.AccuratGeofence
import ai.accurat.sdk.core.AccuratLogger
import ai.accurat.sdk.core.DatabaseHelper
import ai.accurat.sdk.core.GeofencesManager
import ai.accurat.sdk.data.enums.GeofenceType
import ai.accurat.sdk.data.models.DatabaseGeofence
import ai.accurat.sdk.data.models.TriggeredGeofence
import ai.accurat.sdk.managers.AccuratConfigurationManager
import android.content.Context
import androidx.work.Data
import androidx.work.OneTimeWorkRequest
import androidx.work.WorkManager
import androidx.work.WorkRequest
import androidx.work.Worker
import androidx.work.WorkerParameters
import java.util.concurrent.TimeUnit

/**
 * @author Kenneth
 * @since 2021-06-01
 */
class GeofenceDwellWorker(context: Context, workerParameters: WorkerParameters) :
    Worker(context, workerParameters) {

    override fun doWork(): Result {
        val now = System.currentTimeMillis()
        AccuratLogger.init(applicationContext)
        val isTrackingTriggeredGeofences = AccuratConfigurationManager.load().isTrackingTriggeredGeofences
        AccuratLogger.log(
            AccuratLogger.SDK_FLOW,
            if (isTrackingTriggeredGeofences) {
                "TRACK_TRIGGERED_GEOFENCES is enabled, SDK won't post notifications"
            } else {
                "TRACK_TRIGGERED_GEOFENCES is disabled, SDK will post notifications"
            }
        )
        val geofenceId = getGeofenceId()
        AccuratLogger.log(AccuratLogger.WORKMANAGER, "GeofenceDwellWorker.doWork($geofenceId)")
        if (geofenceId == null || !DatabaseGeofence.exists(geofenceId)) {
            AccuratLogger.log(AccuratLogger.GEOFENCE, "No longer dwelling in $geofenceId")
            return Result.success()
        }

        AccuratLogger.log(AccuratLogger.GEOFENCE, "Dwelling in $geofenceId")
        val geofence = getAccuratGeofence(geofenceId)
        if (geofence != null) {
            AccuratLogger.log(AccuratLogger.NONE, "Geofence found: $geofence")
            if (isTrackingTriggeredGeofences) {
                AccuratLogger.log(AccuratLogger.SDK_FLOW, "[DWELL] Storing triggered geofence " + geofence.id)
                TriggeredGeofence.Companion.storeTriggeredGeofence(
                    geofence,
                    GeofenceType.DWELL,
                    now,
                )
            } else {
                checkNotification(geofence)
            }
        } else {
            AccuratLogger.log(
                AccuratLogger.WARNING,
                "Geofence with ID $geofenceId not found in database"
            )
        }

        return Result.success()
    }

    private fun getGeofenceId(): String? {
        return inputData.getString(DATA_GEOFENCE_ID)
    }

    private fun getAccuratGeofence(geofenceId: String): AccuratGeofence? {
        AccuratLogger.log(AccuratLogger.METHOD_START, "GeofenceDwellWorker.getAccuratGeofence($geofenceId)")
        val database = AccuratDatabase(DatabaseHelper.getInstance(applicationContext))
        val geofence = database.getGeofenceById(geofenceId)

        AccuratLogger.log(AccuratLogger.METHOD_END, "GeofenceDwellWorker.getAccuratGeofence($geofenceId)")
        return geofence
    }

    private fun checkNotification(geofence: AccuratGeofence) {
        AccuratLogger.log(AccuratLogger.METHOD_START, "GeofenceDwellManager.checkNotification(${geofence.id})")
        GeofencesManager.init(applicationContext)
        GeofencesManager.showNotificationForGeofence(
            applicationContext, geofence
        ) { success: Boolean ->
            if (success) {
                AccuratLogger.log(
                    AccuratLogger.NOTIFICATION,
                    "Successfully posted notification for geofence with ID " + geofence.id
                )
            } else {
                AccuratLogger.log(
                    AccuratLogger.WARNING,
                    "Failed to post notification for geofence with ID " + geofence.id
                )
            }
        }
        AccuratLogger.log(AccuratLogger.METHOD_END, "GeofenceDwellManager.checkNotification(${geofence.id})")
    }

    companion object {

        const val DATA_GEOFENCE_ID = "DATA_GEOFENCE_ID"

        fun launchRequest(
            context: Context,
            geofenceId: String,
            dwellTime: Long = Configuration.DEFAULT_DWELL_SECONDS
        ) {
            AccuratLogger.log(
                AccuratLogger.WORKMANAGER,
                "Creating WorkRequest for geofence $geofenceId"
            )
            val actualDwellTime = if (dwellTime == 0L) {
                Configuration.DEFAULT_DWELL_SECONDS
            } else {
                dwellTime
            }

            val workRequest: WorkRequest =
                OneTimeWorkRequest.Builder(GeofenceDwellWorker::class.java)
                    .setInitialDelay(actualDwellTime, TimeUnit.SECONDS)
                    .setInputData(
                        Data.Builder()
                            .putString(DATA_GEOFENCE_ID, geofenceId)
                            .build()
                    )
                    .build()
            WorkManager.getInstance(context)
                .enqueue(workRequest)
        }
    }
}