package ai.accurat.sdk;

import android.Manifest;
import android.app.Activity;
import android.content.Context;
import android.content.pm.PackageManager;
import android.location.Location;
import android.text.TextUtils;
import android.util.Log;

import androidx.annotation.DrawableRes;
import androidx.core.app.ActivityCompat;

import com.google.android.gms.common.util.CollectionUtils;
import com.google.android.gms.location.LocationRequest;
import com.google.android.gms.location.LocationServices;
import com.google.android.gms.maps.model.LatLng;

import java.util.ArrayList;
import java.util.List;
import java.util.Locale;

import ai.accurat.sdk.callbacks.AccuratCompletionCallback;
import ai.accurat.sdk.callbacks.AccuratProcessCallback;
import ai.accurat.sdk.callbacks.AccuratResultCallback;
import ai.accurat.sdk.constants.AccuratLanguage;
import ai.accurat.sdk.constants.ConsentState;
import ai.accurat.sdk.constants.ConsentType;
import ai.accurat.sdk.constants.StorageKeys;
import ai.accurat.sdk.core.AccuratGeofence;
import ai.accurat.sdk.core.AccuratLocationManager;
import ai.accurat.sdk.core.AccuratLogger;
import ai.accurat.sdk.core.AccuratSettingsManager;
import ai.accurat.sdk.core.AccuratUserManager;
import ai.accurat.sdk.core.AdvertisingManager;
import ai.accurat.sdk.core.CampaignInteraction;
import ai.accurat.sdk.core.CampaignManager;
import ai.accurat.sdk.core.ConsentManager;
import ai.accurat.sdk.core.GdprRightsManager;
import ai.accurat.sdk.core.GeofenceNotificationManager;
import ai.accurat.sdk.core.GeofencesManager;
import ai.accurat.sdk.core.LocationContext;
import ai.accurat.sdk.core.LocationInterface;
import ai.accurat.sdk.core.Meta;
import ai.accurat.sdk.core.MetaCallback;
import ai.accurat.sdk.core.MultiProcessStorage;
import ai.accurat.sdk.data.RealmHelper;
import ai.accurat.sdk.data.exceptions.UninitialisedException;
import ai.accurat.sdk.data.models.AccuratConfiguration;
import ai.accurat.sdk.data.models.AccuratFeatureSwitch;
import ai.accurat.sdk.data.models.ConsentManagerState;
import ai.accurat.sdk.data.models.Setting;
import ai.accurat.sdk.data.models.TriggeredGeofence;
import ai.accurat.sdk.managers.AccuratConfigurationManager;
import ai.accurat.sdk.managers.AccuratManager;
import ai.accurat.sdk.managers.RealmManager;

/**
 * @Accurat
 */
public class Accurat {

    // <editor-fold desc="Fields">
    /**
     * Key to retrieve notification ID from intent launched when a geofence notification has been clicked.
     */
    public static final String GEOFENCE_NOTIFICATION_ID = GeofencesManager.EXTRA_NOTIFICATION_ID;
    public static final String GEOFENCE_NOTIFICATION_DATA = GeofencesManager.EXTRA_NOTIFICATION_DATA;
    private static final String TAG = Accurat.class.getSimpleName();
    private static Accurat INSTANCE;
    static boolean FORCE_CONSENTS = false;

    private MultiProcessStorage storage;
    private AccuratProcessCallback processCallback;
    public static AccuratLocationUpdate onLocationUpdateCallback;

    // </editor-fold>

    // <editor-fold desc="Construction">
    private Accurat(Context context) {
        AccuratLogger.log(AccuratLogger.METHOD_START, "Creating Accurat instance");
        RealmManager.init(context);
        AdvertisingManager.init(context);
//        ConsentManager.init(context);
        AccuratSettingsManager.init(context);
        GeofenceNotificationManager.init(context);
        AccuratUserManager.init(context);
        CampaignManager.init(context);
        GdprRightsManager.init(context);
        storage = MultiProcessStorage.getStorage(context, StorageKeys.ACCURAT_MULTI_PROCESS_STORAGE);
        AccuratLogger.log(AccuratLogger.METHOD_END, "Creating Accurat instance");
    }
    // </editor-fold>

    // <editor-fold desc="The public interface">

    // <editor-fold desc="Creation">

    /**
     * Initialize the Accurat SDK.
     *
     * @param context       Used for accessing local storage and network.
     * @param configuration An AccuratConfiguration-object containing at least credentials
     */
    public static void initialize(Context context, AccuratConfiguration configuration) {
        if (INSTANCE == null) {
            // Initialise logging
            AccuratLogger.init(context);
            AccuratLogger.log(AccuratLogger.SDK_FLOW, "App was opened by user");

            // Validate the configuration
            if (!AccuratConfigurationManager.isValid(configuration)) {
                AccuratLogger.log(AccuratLogger.WARNING, "The configuration is invalid, initialisation failed. Accurat will not run.");

                return;
            }

            if (configuration.getAppVersion() == null) {
                AccuratLogger.log(AccuratLogger.WARNING, "App version not set. Version specific kill switch not available.");
                Log.w("Accurat SDK", "App version not set. Version specific kill switch not available.");
                Log.i("Accurat SDK", "Use AccuratConfiguration.withAppVersion(String) to set your app version.");
            }

            // Log info about feature switches
            for (AccuratFeatureSwitch featureSwitch : AccuratFeatureSwitch.values()) {
                AccuratLogger.log(AccuratLogger.NONE, featureSwitch.name() + " feature switch is " + (configuration.isFeatureSwitchEnabled(featureSwitch) ? "enabled" : "disabled"));
            }

            // Create a new instance
            INSTANCE = new Accurat(context);

            // Set the notification target package
            configuration.withNotificationTargetPackage(context.getApplicationContext().getPackageName());
            // Store the configuration
            AccuratConfigurationManager.store(configuration);
            AccuratManager.INSTANCE.init(context);
            CampaignManager.upload(null);
        }
    }

    public static void initialize(Context context) {
        if (INSTANCE == null) {
            // Initialise logging
            AccuratLogger.init(context);
            AccuratLogger.log(AccuratLogger.SDK_FLOW, "App was opened after boot or by service");

            // Create a new instance
            INSTANCE = new Accurat(context);

            // Validate the configuration (should be in storage)
            AccuratConfiguration config = AccuratConfigurationManager.load();
            if (!AccuratConfigurationManager.isValid(config)) {
                AccuratLogger.log(AccuratLogger.WARNING, "The configuration is invalid, initialisation failed. Accurat will not run.");

                return;
            }
            CampaignManager.upload(null);
        }
    }

    private static boolean isInitialized() {
        return INSTANCE != null;
    }

    private static void checkInitialized() {
        if (!isInitialized()) {
            throw new IllegalStateException("Accurat has not yet been initialised.");
        }
    }

    /**
     * For debugging purposes only.
     * This returns the last known location detected by the SDK.
     *
     * @return The last known location of the user.
     */
    public static LocationInterface getLastLocation() {
        return AccuratLocationManager.getInstance().getLastLocation();
    }

    /**
     * For debugging purposes only.
     * This returns an array of coordinates which can be used to highlight the search area on a map.
     *
     * @return Coordinates from north-west to south-west.
     */
    public static LatLng[] getSearchBox() {
        return GeofencesManager.getSearchBox();
    }

    /**
     * For debugging purposes only.
     * This returns all the geofences that are currently being tracked.
     *
     * @return List of all geofences currently being monitored.
     */
    public static List<AccuratGeofence> getCurrentGeofences() {
        return CollectionUtils.isEmpty(GeofencesManager.getCurrentGeofences()) ? new ArrayList<>(0) : GeofencesManager.getCurrentGeofences();
    }

    /**
     * Determines whether this geofence is the META-geofence surrounding all other geofences
     *
     * @param accuratGeofence Geofence to verify the identity of
     * @return true when this geofence is the META-geofence
     */
    public static boolean isMetaFence(AccuratGeofence accuratGeofence) {
        return GeofencesManager.isMetaFence(accuratGeofence);
    }

    /**
     * Determines whether this geofence is a META-geofence used to monitor the user's current location.
     *
     * @param accuratGeofence Geofence to verify the identity of
     * @return true when this geofence is a 'current location' META-geofence
     */
    public static boolean isCurrentLocationFence(AccuratGeofence accuratGeofence) {
        return GeofencesManager.isCurrentLocationMetaFence(accuratGeofence);
    }

    /**
     * Handle an app's onResume-method to get additional location information.
     *
     * @param context The current Context
     */
    public static void onResume(Context context) {
        AccuratLogger.init(context);
        AccuratLogger.log(AccuratLogger.METHOD_START, "Accurat.onResume()");
        if (!ConsentManagerState.Companion.canTrack()) {
            AccuratLogger.log(AccuratLogger.SDK_FLOW, "Don't fetch last location because canTrack() is false");
            AccuratLogger.log(AccuratLogger.METHOD_END, "Accurat.onResume()");

            return;
        }

        if (
                ActivityCompat.checkSelfPermission(context, Manifest.permission.ACCESS_FINE_LOCATION) != PackageManager.PERMISSION_GRANTED
                        && ActivityCompat.checkSelfPermission(context, Manifest.permission.ACCESS_COARSE_LOCATION) != PackageManager.PERMISSION_GRANTED
        ) {
            AccuratLogger.log(AccuratLogger.NONE, "Don't fetch last location because fine and coarse location aren't granted");
            AccuratLogger.log(AccuratLogger.METHOD_END, "Accurat.onResume()");

            return;
        }

        LocationServices.getFusedLocationProviderClient(context)
                .getCurrentLocation(LocationRequest.PRIORITY_HIGH_ACCURACY, null)
                .addOnSuccessListener(location -> {
                    if (location == null) {
                        AccuratLogger.log(AccuratLogger.NONE, "Last location is null");
                        AccuratLogger.log(AccuratLogger.METHOD_END, "Accurat.onResume()");

                        return;
                    }
                    AccuratLogger.log(AccuratLogger.SDK_FLOW, "Received last location");

                    if (onLocationUpdateCallback != null) {
                        List<Location> locations = new ArrayList<>();
                        locations.add(location);
                        onLocationUpdateCallback.onLocationUpdated(locations);
                    }

                    AccuratLocationManager.getInstance().onLocationChanged(context, location, LocationContext.getLocationContext(), true);
                    AccuratLogger.log(AccuratLogger.METHOD_END, "Accurat.onResume()");
                });
    }
    // </editor-fold>

    // <editor-fold desc="Tracking">
    public static void startAfterReboot(Context context) {
        FORCE_CONSENTS = false;
        AccuratLogger.init(context);
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".startAfterReboot()");
        Accurat.initialize(context);
        try {
            checkInitialized();
        } catch (IllegalStateException e) {
            AccuratLogger.log(AccuratLogger.ERROR, e.getMessage());
            return;
        }
        AccuratLogger.log(AccuratLogger.STORAGE, "Dumping MultiProcessStorage");
        AccuratLogger.log(AccuratLogger.STORAGE_DATA, INSTANCE.storage.dump());
        if (!ConsentManagerState.Companion.isStartRequested()) {
            AccuratLogger.log(AccuratLogger.NONE, "No start requested");
            return;
        }

        if (!AccuratManager.INSTANCE.canStartFromBackground(context)) {
            AccuratLogger.log(AccuratLogger.WARNING, "Can't restart Accurat from background, user interaction required");
            return;
        }

        INSTANCE.checkSettingsFromBackground(context);
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".startAfterReboot()");
    }

    public static void startTracking(Activity activity, AccuratProcessCallback processCallback) {
        AccuratLogger.init(activity.getApplicationContext());
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".startTracking()");
        FORCE_CONSENTS = false;
        internalStartTracking(activity, processCallback);
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".startTracking()");
    }

    public static void startTracking(Activity activity) {
        startTracking(activity, null);
    }

    public static void forceStartTracking(Activity activity, AccuratProcessCallback processCallback) {
        AccuratLogger.init(activity.getApplicationContext());
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".forceStartTracking()");
        FORCE_CONSENTS = true;
        internalStartTracking(activity, processCallback);
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".forceStartTracking()");
    }

    public static void forceStartTracking(Activity activity) {
        forceStartTracking(activity, null);
    }

    private static void internalStartTracking(Activity activity, AccuratProcessCallback processCallback) {
        try {
            checkInitialized();
        } catch (IllegalStateException e) {
            AccuratLogger.log(AccuratLogger.ERROR, e.getMessage());
            return;
        }

        INSTANCE.processCallback = processCallback;
        AccuratManager.INSTANCE.forceAskConsents(FORCE_CONSENTS);
        try {
            AccuratManager.INSTANCE.startTracking(activity, started -> {
                if (started) {
                    INSTANCE.checkSettings(activity);
                } else if (processCallback != null) {
                    processCallback.onProcessed(false);
                }
            });
        } catch (UninitialisedException ue) {
            AccuratLogger.log(AccuratLogger.ERROR, "UninitialisedException: " + ue.getMessage());
            processCallback.onProcessed(false);
        }
//        INSTANCE.processCallback = processCallback
//        INSTANCE.storeStartRequested();
//
//        INSTANCE.checkAdId(activity);
    }

    /*private void checkAdId(final Activity activity) {
        AccuratLogger.init(activity.getApplicationContext());
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".checkAdId()");
        AdvertisingManager.start(activity, success -> {
            if (!success) {
                if (processCallback != null) {
                    processCallback.onProcessed(false);
                }

                return;
            }

            checkConsents(activity);
        });
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".checkAdId()");
    }*/

    private void checkAdIdFromBackground(final Context context) {
        AccuratLogger.init(context);
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".checkAdIdFromBackground()");
        AdvertisingManager.startFromBackground(context, success -> {
            if (!success) {
                if (processCallback != null) {
                    processCallback.onProcessed(false);
                }

                return;
            }

            checkConsentsFromBackground(context);
        });
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".checkAdIdFromBackground()");
    }

    /*private void checkConsents(final Activity activity) {
        AccuratLogger.init(activity.getApplicationContext());
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".checkConsents()");
        ConsentManager.start(activity, success -> {
            if (!success) {
                if (processCallback != null) {
                    processCallback.onProcessed(false);
                }

                return;
            }

            checkSettings(activity);
        });
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".checkConsents()");
    }*/

    private void checkConsentsFromBackground(final Context context) {
        AccuratLogger.init(context);
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".checkConsentsFromBackground()");
        ConsentManager.startFromBackground(context, success -> {
            if (!success) {
                if (processCallback != null) {
                    processCallback.onProcessed(false);
                }

                return;
            }

            checkSettingsFromBackground(context);
        });
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".checkConsentsFromBackground()");
    }

    private void checkSettings(final Activity activity) {
        AccuratLogger.init(activity.getApplicationContext());
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".checkSettings()");
        RealmManager.init(activity);
        AccuratSettingsManager.start(success -> {
            if (!success) {
                if (processCallback != null) {
                    processCallback.onProcessed(false);
                }

                return;
            }

            GeofenceNotificationManager.start();
            startLocationManager(activity);
        });
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".checkSettings()");
    }

    private void checkSettingsFromBackground(final Context context) {
        AccuratLogger.init(context);
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".checkSettingsFromBackground()");
        AccuratSettingsManager.start(success -> {
            if (!success) {
                if (processCallback != null) {
                    processCallback.onProcessed(false);
                }

                return;
            }

            startLocationManagerFromBackground(context);
        });
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".checkSettingsFromBackground()");
    }

    private void startLocationManager(Activity activity) {
        AccuratLogger.init(activity.getApplicationContext());
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".startLocationManager()");
        boolean initialised = initLocationManager(activity.getApplicationContext());
        if (!initialised) {
            AccuratLogger.log(AccuratLogger.WARNING, "AccuratLocationManager could not be initialised");
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".startLocationManager()");
            return;
        }

        // Start location manager
        AccuratLocationManager.getInstance().start(activity.getApplicationContext(), processCallback);
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".startLocationManager()");
    }

    private void startLocationManagerFromBackground(Context context) {
        AccuratLogger.init(context);
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".startLocationManagerFromBackground");
        boolean initialised = initLocationManager(context);
        if (!initialised) {
            AccuratLogger.log(AccuratLogger.WARNING, "AccuratLocationManager could not be initialised");
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".startLocationManagerFromBackground");
            return;
        }

        AccuratLocationManager.getInstance().start(context.getApplicationContext(), processCallback);
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".startLocationManagerFromBackground");
    }

    private boolean initLocationManager(Context context) {
        AccuratLocationManager.getInstance().init(context);

        return true;
    }

    /**
     * Stop Accurat tracking.
     * <p>
     * This method can be safely called without effect if Accurat isn't tracking.
     *
     * @param context Required to stop certain background operations.
     */
    public static void stopTracking(Context context) {
        AccuratLogger.init(context);
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".stopTracking()");
        if (isTrackingEnabled()) {
            AccuratLogger.log(AccuratLogger.NONE, "Tracking was enabled, stopping...");
            INSTANCE.storeStopped();
            AccuratManager.INSTANCE.stopTracking();
            AccuratLocationManager.getInstance().stop(context);
            AccuratLogger.log(AccuratLogger.NONE, "Stopped");
        } else {
            AccuratLogger.log(AccuratLogger.NONE, "Tracking was not enabled, no need to stop");
        }
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".stopTracking()");
    }

    /**
     * Check if Accurat is tracking.
     *
     * @return True if Accurat is tracking, false otherwise.
     */
    public static boolean isTrackingEnabled() {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".isTrackingEnabled()");
        try {
            boolean isRunning = AccuratLocationManager.getInstance().isRunning();
            AccuratLogger.log(AccuratLogger.NONE, "Accurat is " + (isRunning ? "" : "NOT ") + "tracking");
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".isTrackingEnabled()");

            return isRunning;
        } catch (IllegalStateException e) {
            AccuratLogger.log(AccuratLogger.ERROR, "IllegalStateException: " + e.getMessage());
            AccuratLogger.log(AccuratLogger.WARNING, "Could not check tracking status, LocationManager is not initialised");
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".isTrackingEnabled()");

            return false;
        }
    }

    public static boolean isTrackingEnabled(Context context) {
        AccuratLogger.init(context);
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".isTrackingEnabled(context)");
        try {
            boolean isRunning = AccuratLocationManager.getInstance().isRunning();
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".isTrackingEnabled(context)");

            return isRunning;
        } catch (IllegalStateException e) {
            AccuratLogger.log(AccuratLogger.ERROR, "IllegalStateException: " + e.getMessage());
            if (INSTANCE != null) {
                AccuratLogger.log(AccuratLogger.WARNING, "Accurat.INSTANCE != null");
                AccuratLogger.log(AccuratLogger.NONE, "Initialising LocationManager");
                INSTANCE.initLocationManager(context);
                boolean isTrackingEnabled = isTrackingEnabled();
                AccuratLogger.log(AccuratLogger.NONE, "Accurat is " + (isTrackingEnabled ? "" : "NOT ") + "tracking");

                return isTrackingEnabled;
            }
            AccuratLogger.log(AccuratLogger.WARNING, "Could not check tracking status, LocationManager is not initialised");
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".isTrackingEnabled()");

            return false;
        }

    }

    public static boolean isTrackingStarted(Context context) {
        RealmManager.init(context);

        return ConsentManagerState.Companion.isStartRequested();
    }
    // </editor-fold>

    // <editor-fold desc="Logging">

    /**
     * Returns the logs if logging is enabled.
     *
     * @return The logs as a String
     */
    public static String getLog() {
        try {
            return AccuratLogger.getDebugLog();
        } catch (IllegalStateException e) {
            e.printStackTrace();
        }

        return "<Could not get debug log>";
    }

    /**
     * Clear the logs if logging is enabled.
     */
    public static void clearLog() {
        try {
            AccuratLogger.removeDebugLog();
        } catch (IllegalStateException e) {
            e.printStackTrace();
        }
    }
    // </editor-fold>

    // <editor-fold desc="Consent">

    public static boolean isForceConsents() {
        AccuratLogger.log(AccuratLogger.SDK_FLOW, TAG + ".isForceConsents() = " + FORCE_CONSENTS);
        return FORCE_CONSENTS;
    }

    /**
     * Manually start the consent flow.
     *
     * @param activity   The current Activity, used to show consent dialogs.
     * @param onComplete Indicates the consent flow has finished.
     */
    public static void askConsents(Activity activity, final Runnable onComplete) {
        AccuratLogger.init(activity.getApplicationContext());
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".askConsents()");
        AccuratLogger.log(AccuratLogger.NONE, "Start consent flow manually");
        AccuratManager.INSTANCE.askConsents(onComplete);
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".askConsents()");
    }

    /**
     * Get the state of a consent.
     *
     * @param activity    The current Activity, used to read consents from storage.
     * @param consentType The type of the consent to get the state for.
     * @return The ConsentState of the consentType.
     */
    public static ConsentState getConsentState(Activity activity, ConsentType consentType) {
        RealmManager.init(activity);

        return ConsentManagerState.Companion.getConsentState(ai.accurat.sdk.data.enums.ConsentType.GDPR);
    }

    /**
     * Update the state of a consent.
     *
     * @param activity     The current Activity, used to store consents.
     * @param consentType  The type of the consent to update.
     * @param consentState The new state for the consent.
     */
    public static void updateConsent(Activity activity, ConsentType consentType, ConsentState consentState, AccuratCompletionCallback onCompleted) {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".updateConsent()");
        RealmManager.init(activity);
        ai.accurat.sdk.data.enums.ConsentType newConsentType;
        switch (consentType) {
            case LOCATION:
                newConsentType = ai.accurat.sdk.data.enums.ConsentType.IN_APP_LOCATION;
                break;
            case TRACKING:
            case GDPR:
            default:
                newConsentType = ai.accurat.sdk.data.enums.ConsentType.GDPR;
                break;
        }
        AccuratManager.INSTANCE.init(activity);
        AccuratManager.INSTANCE.setAndPushConsentsToServer(newConsentType, consentState == ConsentState.ACCEPTED, null);
        onCompleted.onCompleted(true);
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".updateConsent()");
    }

    /**
     * Update the state of a permission.
     * Warning: If the consentState value doesn't match the actual system permission's value,
     * the working of the SDK is not guaranteed.
     *
     * @param activity     The current Activity, used to store consents.
     * @param consentType  The type of the permission to update.
     * @param consentState The new state for the permission.
     */
    public static void updatePermission(Activity activity, ConsentType consentType, ConsentState consentState, AccuratCompletionCallback onCompleted) {
        RealmManager.init(activity);
        ConsentManagerState.Companion.setAppConsent(ai.accurat.sdk.data.enums.ConsentType.IN_APP_LOCATION, consentState == ConsentState.ACCEPTED);
        onCompleted.onCompleted(true);
    }

    /**
     * Reset all consents.
     *
     * @param activity The current Activity, used to read and store consents.
     */
    public static void resetConsents(Activity activity) {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".resetConsents()");
        RealmManager.init(activity);
        Accurat.stopTracking(activity.getApplicationContext());
        ConsentManagerState.Companion.reset();
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".resetConsents()");
    }
    // </editor-fold>

    // <editor-fold desc="Language">

    /**
     * Set the language used for consents and geofence notifications.
     *
     * @param languageCode Language code representing a language, f.e. 'en'
     *                     See {@link Locale#getLanguage()} for reference.
     */
    public static void setLanguage(String languageCode) {
        try {
            checkInitialized();
        } catch (IllegalStateException e) {
            AccuratLogger.log(AccuratLogger.ERROR, e.getMessage());
            return;
        }

        if (TextUtils.isEmpty(languageCode)) {
            languageCode = Locale.getDefault().getLanguage();
        }

        AccuratLanguage language;
        try {
            language = AccuratLanguage.fromLanguage(languageCode.toLowerCase());
        } catch (IllegalArgumentException e) {
            // English is the fallback language
            language = AccuratLanguage.EN;
        }
        setLanguage(language);
    }

    /**
     * Set the language used for consents and geofence notifications
     *
     * @param language Supported AccuratLanguage
     */
    public static void setLanguage(AccuratLanguage language) {
        try {
            checkInitialized();
        } catch (IllegalStateException e) {
            AccuratLogger.log(AccuratLogger.ERROR, e.getMessage());
            return;
        }

        AccuratUserManager.setLanguage(language);
    }
    // </editor-fold>

    /**
     * Set a custom icon for geofence notifications
     *
     * @param drawableId Drawable resource ID to be used for geofence notifications
     */
    public static void setNotificationIcon(@DrawableRes int drawableId) {
        try {
            checkInitialized();
        } catch (IllegalStateException e) {
            AccuratLogger.log(AccuratLogger.ERROR, e.getMessage());
            return;
        }

        INSTANCE.storage.setValue(StorageKeys.CLIENT_NOTIFICATION_DRAWABLE_ID, drawableId)
                .commit();
    }

    //<editor-fold desc="Analytics">

    /**
     * Add interaction for consumer, based on given campaign and touchpoint.
     * If campaign and/or touchpoint does not exist, they will be created.
     * <br />
     * <br />
     * Deprecated: Use Accurat.interact(CampaignInteraction, AccuratCompletionCallback) instead.
     *
     * @param group      The group which contains the active campaign
     * @param campaign   The campaign representing the presented advertising messages
     * @param touchpoint The way the consumer interacted with the business
     * @param callback   Indicates whether the interaction has saved correctly.
     */
    @Deprecated
    public static void interact(String group, String campaign, String touchpoint, AccuratCompletionCallback callback) {
        interact(
                new CampaignInteraction()
                        .withGroup(group)
                        .withCampaign(campaign)
                        .withTouchPoint(touchpoint),
                callback
        );
    }

    /**
     * Add an interaction for a consumer, based on a given CampaignInteraction.
     * If campaign and/or touch point do not exist, they will be created.
     * <br />
     * <br />
     * A CampaignInteraction-object can be created using chainable builder methods:
     * <pre>{@code
     * new CampaignInteraction()
     *   .withCampaign(campaign)
     *   .withGroup(group)
     *   .withTouchPoint(touchPoint)
     *   .withId(id)
     *   .startsAt(start)
     *   .endsAt(end)
     * }</pre>
     * @param interaction The CampaignInteraction
     * @param callback Indicates whether the interaction has saved correctly.
     */
    public static void interact(CampaignInteraction interaction, AccuratCompletionCallback callback) {
        try {
            checkInitialized();
        } catch (IllegalStateException e) {
            AccuratLogger.log(AccuratLogger.ERROR, e.getMessage());
            return;
        }

        CampaignManager.add(interaction, callback);
    }

    /**
     * Fetch segments the current consumer belongs to.
     * If the consumer does not exist, an empty list is returned.
     *
     * @param callback Contains the segments the current consumer belongs to.
     */
    public static void getSegments(AccuratResultCallback<String[]> callback) {
        try {
            checkInitialized();
        } catch (IllegalStateException e) {
            AccuratLogger.log(AccuratLogger.ERROR, e.getMessage());
            return;
        }

        AccuratUserManager.getSegments(callback);
    }
    // </editor-fold>

    // <editor-fold desc="Location updates">
    public static void setOnLocationUpdateCallback(AccuratLocationUpdate callback) {
        onLocationUpdateCallback = callback;
    }

    public static void removeOnLocationUpdateCallback() {
        onLocationUpdateCallback = null;
    }
    // </editor-fold>

    // <editor-fold desc="GDPR Rights">
    public static void invokeRight(String right, String info) {
        GdprRightsManager.invokeRight(right, info);
    }
    // </editor-fold>

    // <editor-fold desc="Geofencing">
    /**
     * Returns a list of all stored geofences triggers if AccuratFeatureSwitch.TRACK_TRIGGERED_GEOFENCES is enabled, or an empty list otherwise.
     * All returned geofence triggers are then cleared from storage.
     *
     * @return A list of all stored geofences triggers
     */
    public static ArrayList<TriggeredGeofence> getTriggeredGeofences() {
        if (!AccuratConfigurationManager.load().isTrackingTriggeredGeofences()) {
            return new ArrayList<>();
        }

        return new ArrayList<>(
                TriggeredGeofence
                        .Companion
                        .getAll(true)
        );
    }
    // </editor-fold>
    // </editor-fold>

    // <editor-fold desc="Storage">
    private void storeStartRequested() {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".storeStartRequested()");
        RealmHelper.storeSetting(Setting.Keys.State.START_REQUESTED, true);
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".storeStartRequested()");
    }

    private void storeStopped() {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".storeStopped()");
        RealmHelper.storeSetting(Setting.Keys.State.START_REQUESTED, false);
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".storeStopped()");
    }

    private boolean isStartRequested() {
        return RealmHelper.loadBooleanSetting(Setting.Keys.State.START_REQUESTED, false);
    }

    // </editor-fold>

    // <editor-fold desc="Meta">
    public static void getMeta(MetaCallback callback) {
        try {
            checkInitialized();
        } catch (IllegalStateException e) {
            AccuratLogger.log(AccuratLogger.ERROR, e.getMessage());
            return;
        }

        AccuratUserManager.getMeta(callback);
    }

    public static void setMeta(String key, String value) {
        try {
            checkInitialized();
        } catch (IllegalStateException e) {
            AccuratLogger.log(AccuratLogger.ERROR, e.getMessage());
            return;
        }

        AccuratUserManager.setMeta(new Meta(key, value));
    }

    public static void setMeta(Meta... metas) {
        try {
            checkInitialized();
        } catch (IllegalStateException e) {
            AccuratLogger.log(AccuratLogger.ERROR, e.getMessage());
            return;
        }

        AccuratUserManager.setMeta(metas);
    }
    // </editor-fold>

    public interface AccuratLocationUpdate {
        void onLocationUpdated(List<Location> locations);
    }
}
