package ai.accurat.sdk.core;

import android.content.Context;
import androidx.annotation.NonNull;
import androidx.work.Constraints;
import androidx.work.NetworkType;
import androidx.work.OneTimeWorkRequest;
import androidx.work.PeriodicWorkRequest;
import androidx.work.WorkManager;

import com.android.volley.Request;
import com.android.volley.RequestQueue;
import com.android.volley.toolbox.JsonObjectRequest;
import com.android.volley.toolbox.Volley;

import org.json.JSONException;
import org.json.JSONObject;

import java.util.HashMap;
import java.util.Map;
import java.util.Random;
import java.util.concurrent.TimeUnit;

import ai.accurat.sdk.callbacks.AccuratCompletionCallback;
import ai.accurat.sdk.config.Configuration;
import ai.accurat.sdk.constants.AccuratEndpoints;
import ai.accurat.sdk.constants.ApiKeys;
import ai.accurat.sdk.constants.HttpMethod;
import ai.accurat.sdk.constants.StorageKeys;
import ai.accurat.sdk.managers.AccuratConfigurationManager;

/**
 * @author Kenneth Saey
 * @Accurat
 * @since 02-07-2018 11:09.
 */
public class AccuratSettingsManager {

    // <editor-fold desc="Fields">
    private static final String TAG = AccuratSettingsManager.class.getSimpleName();
    private static final AccuratSettingsManager INSTANCE = new AccuratSettingsManager();

    private static MultiProcessStorage storage;
    private static AccuratSettings settings;
    private static RequestQueue requestQueue;
    // </editor-fold>

    // <editor-fold desc="Initialisation">
    public static void init(@NonNull Context context) {
        if (!isInitialized()) {
            AccuratLogger.init(context);
            AccuratLogger.log(AccuratLogger.SDK_FLOW, "Initialising " + TAG);
            storage = MultiProcessStorage.getStorage(context, StorageKeys.ACCURAT_MULTI_PROCESS_STORAGE);
            requestQueue = Volley.newRequestQueue(context);
        } else if (requestQueue == null) {
            requestQueue = Volley.newRequestQueue(context);
        }
    }

    private static boolean isInitialized() {
        return storage != null;
    }

    private static void checkInitialized() {
        if (!isInitialized()) {
            throw new IllegalStateException("AccuratSettingsManager has not yet been initialised.");
        }
    }
    // </editor-fold>

    // <editor-fold desc="Storage">
    private static void loadSettings() {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".loadSettings()");
        checkInitialized();

        String jsonString = storage.getString(StorageKeys.ACCURAT_SETTINGS, "");
        if (jsonString == null || jsonString.isEmpty()) {
            AccuratLogger.log(AccuratLogger.WARNING, "No settings in local storage");
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".loadSettings()");

            return;
        }

        try {
            settings = AccuratSettings.fromJson(jsonString);
            AccuratLogger.log(AccuratLogger.SETTINGS_DATA, settings.toString());
        } catch (JSONException e) {
            AccuratLogger.log(AccuratLogger.JSON_ERROR, "Failed to parse settings: " + e.getMessage());
            e.printStackTrace();
        }
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".loadSettings()");
    }

    private static void storeSettings(@NonNull AccuratSettings settings) {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".storeSettings()");
        checkInitialized();

        AccuratLogger.log(AccuratLogger.SETTINGS_DATA, settings.toString());
        storage.setValue(StorageKeys.ACCURAT_SETTINGS, settings.toString())
                .commit();
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".storeSettings()");
    }
    // </editor-fold>

    // <editor-fold desc="Package private interface">
    public static void fetchSettings(final AccuratCompletionCallback onCompleted) {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".fetchSettings()");

        String appVersion = AccuratConfigurationManager.getAppVersion();
        HashMap<String, Object> urlParameters = new HashMap<>();
        urlParameters.put(ApiKeys.Url.APP_VERSION, appVersion);
        String url = appVersion == null ? AccuratEndpoints.GET_SETTINGS.getUrl() : AccuratEndpoints.GET_APP_VERSION_SETTINGS.getUrl(urlParameters);
        String path = appVersion == null ? AccuratEndpoints.GET_SETTINGS.getPath() : AccuratEndpoints.GET_APP_VERSION_SETTINGS.getPath(urlParameters);
        String endpoint = appVersion == null ? Configuration.ENDPOINT_GET_SETTINGS : Configuration.ENDPOINT_GET_APP_VERSION_SETTINGS;

        JsonObjectRequest parametersRequest = new JsonObjectRequest(
                Request.Method.GET,
                url,
                null,
                response -> {
                    AccuratLogger.logNetworkResponse(HttpMethod.GET, endpoint, response, false);
                    if (!response.has("data")) {
                        onCompleted.onCompleted(false);

                        return;
                    }

                    try {
                        JSONObject data = response.getJSONObject("data");
                        if (!data.has("parameters")) {
                            onCompleted.onCompleted(false);

                            return;
                        }

                        JSONObject parameters = data.getJSONObject("parameters");
                        AccuratSettings settings = AccuratSettings.fromServerJson(parameters);
                        if (settings == null) {
                            onCompleted.onCompleted(false);

                            return;
                        }
                        settings.addDefaultEndpoint();
                        storeSettings(settings);
                        AccuratSettingsManager.settings = settings;

                        onCompleted.onCompleted(true);
                    } catch (JSONException e) {
                        AccuratLogger.log(AccuratLogger.JSON_ERROR, TAG + ".fetchSettings(): " + e.getMessage());
                        e.printStackTrace();
                        onCompleted.onCompleted(false);
                    }
                },
                error -> {
                    AccuratLogger.logNetworkError(HttpMethod.GET, endpoint, error);
                    onCompleted.onCompleted(false);
                }
        ) {
            @Override
            public Map<String, String> getHeaders() {
                return AccuratApi.getHeaders(
                        storage,
                        "GET",
                        "",
                        AccuratApi.getEncodedRequestBody(""),
                        path
                );
            }
        };

        parametersRequest.setTag(TAG)
                .setRetryPolicy(AccuratApi.defaultRetryPolicy)
                .setShouldCache(false);

        if (requestQueue == null) {
            AccuratLogger.log(AccuratLogger.ERROR, "Failed to make API call, requestQueue is NULL");
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".fetchSettings()");

            return;
        }

        AccuratLogger.logNetworkRequest(HttpMethod.GET, endpoint);
        requestQueue.add(parametersRequest);
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".fetchSettings()");
    }

    static void cancelNetworkRequests() {
        AccuratLogger.log(AccuratLogger.NONE, TAG + ".cancelNetworkRequests()");
        if (requestQueue != null) {
            requestQueue.cancelAll(TAG);
            // Note - the onCompleted callback in fetchSettings will not be called!
            // Todo - Check this doesn't block the AccuratSettingsWorker thread.
        }
    }

    /**
     * Ask the WorkManager to synchronize the settings.
     * Also plans a daily sync after the initial sync completion.
     */
    static void schedule() {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".schedule()");
        checkInitialized();

        WorkManager workManager = WorkManager.getInstance();
        if (workManager == null) {
            AccuratLogger.log(AccuratLogger.ERROR, "WorkManager.getInstance() is NULL");
            return;
        }

        // Cancel any previously running work
        workManager.cancelAllWorkByTag(Constants.SYNC_SETTINGS_WORK_TAG);

        // Needs to have an initial delay since a periodic task does not have such a constraint.
        long syncTime = getSyncTime();
        long delayDuration = syncTime - System.currentTimeMillis();
        // The delay should be positive (in the future). If not, delay by a day (in ms)
        long checkedDelayDuration = delayDuration > 0 ? delayDuration : 86400000;
        AccuratLogger.log(AccuratLogger.NONE, TAG + " should trigger around " + (System.currentTimeMillis() + checkedDelayDuration));

        OneTimeWorkRequest.Builder builder = new OneTimeWorkRequest.Builder(AccuratScheduleSettingsWorker.class)
                .setInitialDelay(checkedDelayDuration, TimeUnit.MILLISECONDS)
                .addTag(Constants.SYNC_SETTINGS_WORK_TAG);

        // Schedule the work
        workManager.enqueue(builder.build());
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".schedule()");
    }

    /**
     * Plan a daily synchronisation of the settings
     */
    static void scheduleFetch() {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".scheduleFetch()");
        checkInitialized();

        WorkManager workManager = WorkManager.getInstance();
        if (workManager == null) {
            AccuratLogger.log(AccuratLogger.ERROR, "WorkManager.getInstance() is NULL");
            return;
        }

        // Cancel any previously running work
        workManager.cancelAllWorkByTag(Constants.SYNC_SETTINGS_WORK_TAG);

        // Create a new WorkRequest
        PeriodicWorkRequest.Builder builder = new PeriodicWorkRequest.Builder(
                AccuratSettingsWorker.class,
                1,
                TimeUnit.DAYS
        ).setConstraints(getWorkerConstraints())
                .addTag(Constants.SYNC_SETTINGS_WORK_TAG);

        // Schedule the work
        workManager.enqueue(builder.build());
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".scheduleFetch()");
    }

    static void stop() {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".stop()");
        WorkManager workManager = WorkManager.getInstance();
        if (workManager == null) {
            AccuratLogger.log(AccuratLogger.ERROR, "WorkManager.getInstance() is NULL");
            return;
        }

        // Cancel any previously running work
        workManager.cancelAllWorkByTag(Constants.SYNC_SETTINGS_WORK_TAG);
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".stop()");
    }
    // </editor-fold>

    // <editor-fold desc="Public interface">
    public static void start(final AccuratCompletionCallback onCompleted) {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".start()");
        loadSettings();

        if (settings == null) {
            AccuratLogger.log(AccuratLogger.SETTINGS, "Settings are null");
            AccuratLogger.log(AccuratLogger.SDK_FLOW, "Fetching settings from server...");
            fetchSettings(success -> {
                AccuratLogger.log(AccuratLogger.SETTINGS, "Settings " + (success ? "successfully fetched" : "failed to fetch"));
                if (!success) {
                    AccuratLogger.log(AccuratLogger.SDK_FLOW, "Use default settings");
                    settings = AccuratSettings.getDefaultSettings();
                    settings.addDefaultEndpoint();
                }
                storeSettings(settings);
                onCompleted.onCompleted(settings.isSdkEnabled());
            });
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".start()");

            return;
        }

        AccuratLogger.log(AccuratLogger.SETTINGS, "Loaded settings from local storage");
        onCompleted.onCompleted(settings.isSdkEnabled());
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".start()");
    }

    public static AccuratSettings getSettings() {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".getSettings()");
        if (settings == null) {
            AccuratLogger.log(AccuratLogger.SDK_FLOW, "Settings are NULL, loading settings...");
            loadSettings();
            AccuratLogger.log(AccuratLogger.SETTINGS_DATA, (settings == null ? "null" : settings.toString()));
        }

        AccuratLogger.log(AccuratLogger.NONE, "Settings haven't changed");
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".getSettings()");
        return settings;
    }

    public static boolean isGeofencingEnabled() {
        AccuratSettings settings = getSettings();

        return settings == null || settings.isGeofencingEnabled();
    }
    // </editor-fold>

    // <editor-fold desc="Helpers">
    private static Constraints getWorkerConstraints() {
        return new Constraints.Builder()
                .setRequiresBatteryNotLow(true)
                .setRequiredNetworkType(NetworkType.CONNECTED)
                .build();
    }

    private static long getSyncTime() {
        checkInitialized();

        long fiveMinutes = 300000;
        long oneDay = 86400000;

        long syncMillis = generateSyncMillis();
        long now = System.currentTimeMillis();
        long midnight = (now / oneDay) * oneDay;
        long syncTime = midnight + syncMillis;

        if (syncTime <= now + fiveMinutes) {// Add an additional 5 minutes to allow worker threads to settle.
            syncTime += oneDay;
        }

        return syncTime;
    }

    private static long generateSyncMillis() {
        int oneHour = 3600;
        int oneDay = 86400;

        int windowSize = Configuration.SYNC_WINDOW_END_SECONDS - Configuration.SYNC_WINDOW_START_SECONDS;
        if (windowSize < 0) {
            windowSize = oneDay + windowSize;
        } else if (windowSize == 0) {
            windowSize = oneHour;// Window should be at least 1 hour long
        }

        int offset = new Random(System.currentTimeMillis()).nextInt(windowSize);
        int syncSeconds;
        if (Configuration.SYNC_WINDOW_START_SECONDS <= Configuration.SYNC_WINDOW_END_SECONDS) {
            syncSeconds = Configuration.SYNC_WINDOW_START_SECONDS + offset;
        } else {
            syncSeconds = Configuration.SYNC_WINDOW_END_SECONDS + offset;
        }

        return syncSeconds % oneDay * 1000;// In milliseconds
    }
    // </editor-fold>
}
