package ai.accurat.sdk.core;

import android.database.Cursor;
import android.database.DatabaseUtils;
import android.database.sqlite.SQLiteDatabase;
import android.database.sqlite.SQLiteStatement;
import android.provider.BaseColumns;
import androidx.annotation.NonNull;

import com.google.android.gms.common.util.CollectionUtils;
import com.google.gson.Gson;

import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.List;

/**
 * Contract for storing and searching through geofences
 */
public final class GeofenceContract {

    private static final String TAG = GeofenceContract.class.getSimpleName();

    private GeofenceContract() {
    }

    //<editor-fold desc="Geofence-related tables">

    /**
     * Defines the table contents for geofences.
     */
    public static class GeofenceEntry implements BaseColumns {
        static final String TABLE_NAME = "geofence";
        static final String COLUMN_RTREE_MIN_LAT = "rtree_min_lat";
        static final String COLUMN_RTREE_MAX_LAT = "rtree_max_lat";
        static final String COLUMN_RTREE_MIN_LNG = "rtree_min_lng";
        static final String COLUMN_RTREE_MAX_LNG = "rtree_max_lng";
        static final String COLUMN_GEOFENCE = "geofence_blob";
        static final String COLUMN_CREATED_AT = "created_at";
        static final String COLUMN_SERVER_ID = "server_id";
        static final String TIMESTAMP_INDEX = COLUMN_CREATED_AT + "_index";
        static final int COLUMN_INDEX_RTREE_MIN_LAT = 1;
        static final int COLUMN_INDEX_RTREE_MAX_LAT = 2;
        static final int COLUMN_INDEX_RTREE_MIN_LNG = 3;
        static final int COLUMN_INDEX_RTREE_MAX_LNG = 4;
        static final int COLUMN_INDEX_GEOFENCE = 5;
        static final int COLUMN_INDEX_SERVER_ID = 6;
        static final DateFormat ISO_8601_FORMAT = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");

        static final String CREATE_TABLE_SQL = "CREATE TABLE IF NOT EXISTS `" + TABLE_NAME + "` ("
                + "`" + _ID + "` INTEGER PRIMARY KEY AUTOINCREMENT, "
                + "`" + COLUMN_RTREE_MIN_LAT + "` DOUBLE, "
                + "`" + COLUMN_RTREE_MAX_LAT + "` DOUBLE, "
                + "`" + COLUMN_RTREE_MIN_LNG + "` DOUBLE, "
                + "`" + COLUMN_RTREE_MAX_LNG + "` DOUBLE, "
                + "`" + COLUMN_GEOFENCE + "` BLOB NOT NULL, "
                + "`" + COLUMN_CREATED_AT + "` datetime DEFAULT current_timestamp, "
                + "`" + COLUMN_SERVER_ID + "` TEXT)";

        static final String DROP_TABLE_SQL = "DROP TABLE IF EXISTS " + TABLE_NAME;

        static final String INSERT_SQL = "INSERT INTO `" + TABLE_NAME + "` ("
                + "`" + COLUMN_RTREE_MIN_LAT + "`, "
                + "`" + COLUMN_RTREE_MAX_LAT + "`, "
                + "`" + COLUMN_RTREE_MIN_LNG + "`, "
                + "`" + COLUMN_RTREE_MAX_LNG + "`, "
                + "`" + COLUMN_GEOFENCE + "`, "
                + "`" + COLUMN_SERVER_ID + "`) "
                + "VALUES (?, ?, ?, ?, ?, ?)";

        static final String CREATE_INDEX_SQL = "CREATE INDEX IF NOT EXISTS `" + TIMESTAMP_INDEX + "` "
                + " ON `" + TABLE_NAME + "` (`" + COLUMN_CREATED_AT + "` ASC);";
    }
    //</editor-fold>

    //<editor-fold desc="Scheme management">
    static void createIfRequired(SQLiteDatabase db) {
        db.execSQL(GeofenceEntry.CREATE_TABLE_SQL);
        db.execSQL(GeofenceEntry.CREATE_INDEX_SQL);
    }

    static void upgrade(SQLiteDatabase db, int oldVersion, int newVersion) {
        dropTables(db);
        createIfRequired(db);
    }

    static void dropTables(SQLiteDatabase db) {
        db.execSQL(GeofenceEntry.DROP_TABLE_SQL);
    }
    //</editor-fold>

    //<editor-fold desc="Database access">

    /**
     * Add all geofences to the database
     *
     * @param db        database to add geofences to
     * @param geofences geofences to add
     */
    synchronized static void addAll(SQLiteDatabase db, List<AccuratGeofence> geofences) {
        if (db == null || CollectionUtils.isEmpty(geofences)) {
            return;
        }

        // Try with resource management -> will closeHelper statement if not null after completion
        try (SQLiteStatement statement = db.compileStatement(GeofenceEntry.INSERT_SQL)) {
            Gson gson = new Gson();
            db.beginTransaction();
            for (AccuratGeofence geofence : geofences) {
                if (geofence == null) {
                    continue;
                }

                db.delete(GeofenceEntry.TABLE_NAME, GeofenceEntry.COLUMN_SERVER_ID + "=?", new String[]{geofence.getId()});

                statement.clearBindings();
                statement.bindDouble(GeofenceEntry.COLUMN_INDEX_RTREE_MIN_LAT, geofence.getLatitude());
                statement.bindDouble(GeofenceEntry.COLUMN_INDEX_RTREE_MAX_LAT, geofence.getLatitude());
                statement.bindDouble(GeofenceEntry.COLUMN_INDEX_RTREE_MIN_LNG, geofence.getLongitude());
                statement.bindDouble(GeofenceEntry.COLUMN_INDEX_RTREE_MAX_LNG, geofence.getLongitude());
                statement.bindBlob(GeofenceEntry.COLUMN_INDEX_GEOFENCE, gson.toJson(geofence).getBytes());
                statement.bindString(GeofenceEntry.COLUMN_INDEX_SERVER_ID, geofence.getId());

                statement.executeInsert();
            }

            db.setTransactionSuccessful();
        } finally {
            db.endTransaction();
        }
    }

    /**
     * Find a specific geofence in the local database
     *
     * @param db db to search
     * @param id server id of the geofence to locate
     * @return {@link AccuratGeofence} or null if the geofence has not been found
     */
    synchronized static AccuratGeofence findById(SQLiteDatabase db, String id) {
        if (db == null) {
            return null;
        }

        Cursor cursor = null;
        // Don't include resources in try definition for readability
        //noinspection TryFinallyCanBeTryWithResources
        try {
            cursor = db.query(GeofenceEntry.TABLE_NAME,
                    new String[]{GeofenceEntry.COLUMN_GEOFENCE, GeofenceEntry.COLUMN_SERVER_ID},
                    GeofenceEntry.COLUMN_SERVER_ID + "=?",
                    new String[]{id},
                    null, null, null);

            if (cursor == null || cursor.isClosed() || !cursor.moveToFirst()) {
                return null;
            }

            return getGeofenceFromCursor(cursor, new Gson());
        } finally {
            if (cursor != null) {
                cursor.close();
            }
        }
    }

    /**
     * Gets the number of geofences stored in the database.
     */
    synchronized static long count(SQLiteDatabase db) {
        if (db == null) {
            return 0;
        }
        return DatabaseUtils.queryNumEntries(db, GeofenceEntry.TABLE_NAME);
    }

    /**
     * Retrieves all geofences from storage
     *
     * @param db Readable database to get geofences from.
     * @return A list of all the geofences in local storage.
     */
    synchronized static List<AccuratGeofence> getAll(SQLiteDatabase db) {
        Cursor c = db.query(GeofenceEntry.TABLE_NAME,
                new String[]{GeofenceEntry._ID, GeofenceEntry.COLUMN_GEOFENCE},
                null, null, null, null, null);
        return getGeofencesFromCursor(c);
    }

    /**
     * Retrieve all geofences within a search range from storage
     *
     * @param db       Readable database to get geofences from
     * @param location location to search around
     * @param box      search box defining the search area
     * @return A list of all geofences within the search area
     */
    synchronized public static List<AccuratGeofence> getAllWithinBox(SQLiteDatabase db, LocationInterface location, AccuratGeofenceRange box) {
        Double minLatRange = location.getLocationInfo().getLatitude() - box.getLatitudeRange();
        Double maxLatRange = location.getLocationInfo().getLatitude() + box.getLatitudeRange();
        Double minLngRange = location.getLocationInfo().getLongitude() - box.getLongitudeRange();
        Double maxLngRange = location.getLocationInfo().getLongitude() + box.getLongitudeRange();

        String selection = GeofenceEntry.COLUMN_RTREE_MIN_LAT + ">=? AND " +
                GeofenceEntry.COLUMN_RTREE_MAX_LAT + "<=? AND " +
                GeofenceEntry.COLUMN_RTREE_MIN_LNG + ">=? AND " +
                GeofenceEntry.COLUMN_RTREE_MAX_LNG + "<=?";

        Cursor cursor = db.query(GeofenceEntry.TABLE_NAME,
                new String[]{"*"},
                selection,
                new String[]{String.valueOf(minLatRange), String.valueOf(maxLatRange), String.valueOf(minLngRange), String.valueOf(maxLngRange)},
                null, null, null);

        return getGeofencesFromCursor(cursor);
    }

    /**
     * Retrieve all geofences within a search box
     *
     * @param db       Readable database to get geofences from
     * @param searchBox      search box defining the search area
     * @return A list of all geofences within the search area
     */
    synchronized public static List<AccuratGeofence> getAllWithinBox(SQLiteDatabase db, SearchBox searchBox) {
        String selection = GeofenceEntry.COLUMN_RTREE_MIN_LAT + ">=? AND " +
                GeofenceEntry.COLUMN_RTREE_MAX_LAT + "<=? AND " +
                GeofenceEntry.COLUMN_RTREE_MIN_LNG + ">=? AND " +
                GeofenceEntry.COLUMN_RTREE_MAX_LNG + "<=?";

        Cursor cursor = db.query(GeofenceEntry.TABLE_NAME,
                new String[]{"*"},
                selection,
                new String[]{
                        String.valueOf(searchBox.getMinLatitude()),
                        String.valueOf(searchBox.getMaxLatitude()),
                        String.valueOf(searchBox.getMinLongitude()),
                        String.valueOf(searchBox.getMaxLongitude())
                },
                null, null, null);

        return getGeofencesFromCursor(cursor);
    }

    /**
     * Get the most recently created geofences
     *
     * @param db    Readable database to read from
     * @param limit Number of geofences to fetch
     * @return List of the most recently created geofences
     */
    synchronized public static List<AccuratGeofence> getRecent(SQLiteDatabase db, int limit) {
        Cursor c = db.query(GeofenceEntry.TABLE_NAME,
                new String[]{GeofenceEntry._ID, GeofenceEntry.COLUMN_GEOFENCE, GeofenceEntry.COLUMN_CREATED_AT},
                null, null, null, null,
                GeofenceEntry.COLUMN_CREATED_AT + " DESC", String.valueOf(limit));
        return getGeofencesFromCursor(c);
    }

    /**
     * Remove all geofences stored in the database.
     *
     * @param db Writable database via which to delete the geofences.
     * @return Number of geofences that have been deleted.
     */
    synchronized public static long removeGeofences(SQLiteDatabase db) {
        if (db == null) {
            return -1;
        }
        return db.delete(GeofenceEntry.TABLE_NAME, null, null);
    }
    //</editor-fold>

    //<editor-fold desc="Helpers">

    /**
     * Helper method that takes a cursor and creates a list of {@link AccuratGeofence} objects.
     *
     * @param cursor Cursor with custom query to loop through geofences.
     * @return List of geofences within the cursor, or an empty list if nothing has been found.
     */
    @NonNull
    private static List<AccuratGeofence> getGeofencesFromCursor(Cursor cursor) {
        List<AccuratGeofence> geofences = new ArrayList<>();

        try {
            // TODO: 30/10/2018 cursor is closed for moveToFirst?
            if (cursor == null || cursor.isClosed() || !cursor.moveToFirst()) {
                return geofences;
            }

            Gson gson = new Gson();
            do {
                geofences.add(getGeofenceFromCursor(cursor, gson));
            } while (cursor.moveToNext());

            return geofences;
        } finally {
            if (cursor != null) {
                cursor.close();
            }
        }
    }

    /**
     * Convert a {@link Cursor} to a {@link AccuratGeofence}.
     * Does not move the cursor.
     *
     * @param cursor Cursor moving through the result of a database query for geofences
     * @param gson   Gson object used to convert the geofence object in the cursor
     * @return The AccuratGeofence object resulting from Gson parsing
     */
    private static AccuratGeofence getGeofenceFromCursor(@NonNull Cursor cursor, @NonNull Gson gson) {
        // Parse geofence
        byte[] blob = cursor.getBlob(cursor.getColumnIndex(GeofenceEntry.COLUMN_GEOFENCE));
        String json = new String(blob);
        AccuratGeofence geofence = gson.fromJson(json, AccuratGeofence.class);

        // Parse and add creation date if present
        int creationDateIndex = cursor.getColumnIndex(GeofenceEntry.COLUMN_CREATED_AT);
        if (creationDateIndex > -1) {
            String datetime = cursor.getString(creationDateIndex);
            try {
                geofence.setCreatedAt(GeofenceEntry.ISO_8601_FORMAT.parse(datetime));
            } catch (ParseException e) {
                AccuratLogger.log(AccuratLogger.ERROR, "getGeofenceFromCursor(): Could not parse date " + datetime + "(" + e.getMessage() + ")");
            }
        }

        return geofence;
    }
    //</editor-fold>

}