package ai.accurat.sdk.core;

import android.text.TextUtils;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collections;
import java.util.Date;

import ai.accurat.sdk.config.Configuration;
import ai.accurat.sdk.constants.ServerDataKeys;
import ai.accurat.sdk.constants.StorageKeys;

public class GeofenceNotification extends JSON {

    private static final String TAG = GeofenceNotification.class.getSimpleName();

    // <editor-fold desc="Field">
    private long notificationId;
    private String data;
    private String title;
    private String message;
    private String brandColour;

    private long frequency;
    private ArrayList<TimeInterval> timeIntervals;
    // </editor-fold>

    // <editor-fold desc="Initialisation">
    public GeofenceNotification() {
    }

    public GeofenceNotification(long notificationId, String data, String title, String message, String brandColour, long frequency, ArrayList<TimeInterval> timeIntervals) {
        this.notificationId = notificationId;
        this.data = data;
        this.title = title;
        this.message = message;
        this.brandColour = brandColour;
        this.frequency = frequency;
        this.timeIntervals = timeIntervals;
    }
    // </editor-fold>

    // <editor-fold desc="JSON Handling">
    @Override
    public void update(JSONObject json) {
        notificationId = load(json, StorageKeys.GeofenceNotification.NOTIFICATION_ID, notificationId);
        data = load(json, StorageKeys.GeofenceNotification.DATA, data);
        title = load(json, StorageKeys.GeofenceNotification.TITLE, title);
        message = load(json, StorageKeys.GeofenceNotification.MESSAGE, message);
        brandColour = load(json, StorageKeys.GeofenceNotification.BRAND_COLOUR, brandColour);
        frequency = load(json, StorageKeys.GeofenceNotification.FREQUENCY, frequency);
        timeIntervals = loadObjectList(json, StorageKeys.GeofenceNotification.TIME_INTERVALS, timeIntervals, TimeInterval.class);
    }

    @Override
    public JSONObject toJson() {
        JSONObject json = new JSONObject();

        try {
            json.put(StorageKeys.GeofenceNotification.NOTIFICATION_ID, notificationId);
            json.put(StorageKeys.GeofenceNotification.DATA, data);
            json.put(StorageKeys.GeofenceNotification.TITLE, title);
            json.put(StorageKeys.GeofenceNotification.MESSAGE, message);
            json.put(StorageKeys.GeofenceNotification.BRAND_COLOUR, brandColour);
            json.put(StorageKeys.GeofenceNotification.FREQUENCY, frequency);
            json.put(StorageKeys.GeofenceNotification.TIME_INTERVALS, toJsonObjectArray(timeIntervals));
        } catch (JSONException e) {
            AccuratLogger.log(AccuratLogger.JSON_ERROR, TAG + ".toJson(): " + e.getMessage());
            e.printStackTrace();
        }

        return json;
    }

    public static GeofenceNotification fromServerJson(JSONObject json) {
        GeofenceNotification notification = new GeofenceNotification();

        notification.notificationId = load(json, StorageKeys.GeofenceNotification.Server.NOTIFICATION_ID, 0);
        notification.data = load(json, StorageKeys.GeofenceNotification.Server.DATA);
        notification.title = load(json, StorageKeys.GeofenceNotification.Server.TITLE);
        notification.message = load(json, StorageKeys.GeofenceNotification.Server.MESSAGE);
        notification.brandColour = load(json, StorageKeys.GeofenceNotification.Server.BRAND_COLOUR);
        notification.frequency = load(json, StorageKeys.GeofenceNotification.Server.FREQUENCY, 1);
        notification.timeIntervals = parseServerTimings(json);

        return notification;
    }

    public static GeofenceNotification fromOnlineJson(JSONObject json) {
        GeofenceNotification notification = new GeofenceNotification();

        try {
            notification.notificationId = Long.parseLong(load(json, ServerDataKeys.GeofenceTrigger.NOTIFICATION_ID));
        } catch (NumberFormatException e) {
            AccuratLogger.log(AccuratLogger.ERROR, "Error parsing " + TAG + ".notificationID from server JSON: " + e.getMessage());
        }
        notification.data = load(json, ServerDataKeys.GeofenceTrigger.NOTIFICATION_DATA);
        notification.title = load(json, ServerDataKeys.GeofenceTrigger.TITLE);
        notification.message = load(json, ServerDataKeys.GeofenceTrigger.MESSAGE);
        notification.brandColour = load(json, ServerDataKeys.GeofenceTrigger.BRAND_COLOR, Configuration.DEFAULT_NOTIFICATION_COLOUR);

        if (TextUtils.isEmpty(notification.brandColour) || VALUE_NULL_STRING.equals(notification.brandColour)) {
            notification.brandColour = Configuration.DEFAULT_NOTIFICATION_COLOUR;
        }

        return notification;
    }
    // </editor-fold>

    // <editor-fold desc="Private helpers">
    private static ArrayList<TimeInterval> parseServerTimings(JSONObject json) {
        JSONArray timings = load(json, StorageKeys.GeofenceNotification.Server.TIME_INTERVALS, new JSONArray());
        ArrayList<TimeInterval> timeIntervals = new ArrayList<>();

        for (int timingIndex = 0; timingIndex < timings.length(); timingIndex++) {
            // Loop over each timing-entry.
            try {
                JSONObject timing = timings.getJSONObject(timingIndex);
                // First, extract the day.
                int dayNumber = load(timing, StorageKeys.TimeInterval.Server.DAY, -1);
                if (dayNumber < 1 || dayNumber > 7) {
                    // Skip this timing-entry if the day is invalid. Should be in [1, 7]
                    continue;
                }
                Day day = Day.values()[dayNumber - 1];

                // Next, extract the hours.
                ArrayList<Integer> hours = loadIntegerList(timing, StorageKeys.TimeInterval.Server.HOURS, new ArrayList<>());
                if (hours.isEmpty()) {
                    // Skip if no hours are available.
                    continue;
                }

                // Sort the hours ascending.
                Collections.sort(hours);

                // Transform into TimeIntervals.
                // Set up the first interval.
                int startHour = hours.get(0);
                int endHour = hours.get(0);

                // Loop over each timings.hours entry.
                for (int hoursIndex = 1; hoursIndex < hours.size(); hoursIndex++) {
                    int hour = hours.get(hoursIndex);
                    switch (hour - endHour) {
                        case 0:
                            // A double entry in the timings.hours array. Skip it and continue.
                            continue;
                        case 1:
                            // The new hour follows the previous immediately.
                            endHour = hour;
                            continue;
                        default:
                            // This is the start of a new interval.
                            // First, save the previous one.
                            timeIntervals.add(new TimeInterval(day, startHour, endHour));

                            // Then, set up for the new interval.
                            startHour = hour;
                            endHour = hour;
                    }
                }
                // Save the last interval.
                timeIntervals.add(new TimeInterval(day, startHour, endHour));
            } catch (JSONException e) {
                AccuratLogger.log(AccuratLogger.JSON_ERROR, TAG + ".parseServerTimings(): " + e.getMessage());
                e.printStackTrace();
            }
        }

        return timeIntervals;
    }
    // </editor-fold>

    // <editor-fold desc="Getters">
    public long getNotificationId() {
        return notificationId;
    }

    public String getData() {
        return data;
    }

    public String getTitle() {
        return title;
    }

    public String getMessage() {
        return message;
    }

    public String getBrandColour() {
        return brandColour;
    }

    public long getFrequency() {
        return frequency;
    }

    public ArrayList<TimeInterval> getTimeIntervals() {
        return timeIntervals;
    }
    // </editor-fold>

    // <editor-fold desc="Helpers">
    public boolean canShowAt(Date date) {
        if (timeIntervals == null || timeIntervals.size() == 0) {
            // If no timings are available, interpret as "show always"
            return true;
        }

        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);

        int hour = calendar.get(Calendar.HOUR_OF_DAY);

        return contains(hour);
    }

    public boolean contains(int hour) {
        if (timeIntervals == null || timeIntervals.size() == 0) {
            // If no timings are available, interpret as "show always"
            return true;
        }

        for (TimeInterval interval : timeIntervals) {
            if (interval.contains(hour)) {
                return true;
            }
        }

        return false;
    }
    // </editor-fold>
}
