package ai.accurat.sdk.core;

import android.content.Context;
import androidx.annotation.NonNull;

import androidx.work.Constraints;
import androidx.work.ExistingPeriodicWorkPolicy;
import androidx.work.NetworkType;
import androidx.work.PeriodicWorkRequest;
import androidx.work.WorkManager;
import androidx.work.Worker;
import androidx.work.WorkerParameters;

import java.util.concurrent.TimeUnit;

import ai.accurat.sdk.callbacks.AccuratActionableCallback;
import ai.accurat.sdk.managers.RealmManager;

/**
 * A {@link Worker} class for keeping the local collection of geofences up to date.
 */
public class GeofencesSyncWorker extends Worker {
    private static final String TAG = GeofencesSyncWorker.class.getSimpleName();
    private static final String UNIQUE_WORK_NAME = GeofencesSyncWorker.class.getCanonicalName();

    public GeofencesSyncWorker(@NonNull Context context, @NonNull WorkerParameters workerParams) {
        super(context, workerParams);
    }

    @NonNull
    @Override
    public Result doWork() {
        AccuratLogger.init(getApplicationContext());
        AccuratLogger.log(AccuratLogger.WORKMANAGER, TAG + ".doWork()");
        RealmManager.init(getApplicationContext());
        CustomGeofenceSyncManager.init(this.getApplicationContext());
        CustomGeofenceSyncManager.sync(this.getApplicationContext(), new AccuratActionableCallback() {
            @Override
            public void onCompleted(boolean success) {
                if (success) {
                    AccuratLogger.log(AccuratLogger.SDK_FLOW, "Successfully completed periodic sync of geofences");
                } else {
                    AccuratLogger.log(AccuratLogger.WARNING, "Failed to complete periodic sync of geofences");
                }
            }

            @Override
            public void onActionPotentiallyRequired(boolean isActionRequired) {
                AccuratLogger.log(AccuratLogger.NONE, (isActionRequired ? "Action" : "No action") + " required after periodic geofence sync");
            }
        }, true);

        AccuratLogger.log(AccuratLogger.WORKMANAGER, TAG + " - Work done, returning Result.success()");
        return Result.success();
    }

    /**
     * Builds a {@link PeriodicWorkRequest} which will run this worker once every 24 hours
     * while charging and connected to an unmetered network connection.
     *
     * @return A daily work request.
     */
    static void planGeofencesSync() {
        Constraints constraints = new Constraints.Builder()
                .setRequiresCharging(true)
                .setRequiredNetworkType(NetworkType.UNMETERED)
                .build();
        PeriodicWorkRequest periodicWorkRequest = new PeriodicWorkRequest.Builder(GeofencesSyncWorker.class, 1L, TimeUnit.DAYS)
                .setConstraints(constraints)
                .build();
        WorkManager.getInstance().enqueueUniquePeriodicWork(UNIQUE_WORK_NAME, ExistingPeriodicWorkPolicy.REPLACE, periodicWorkRequest);
        AccuratLogger.log(AccuratLogger.SDK_FLOW, "Planned geofence sync");
    }
}
