package ai.accurat.sdk.core;

import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;
import android.location.Location;
import android.os.AsyncTask;

import com.google.android.gms.location.LocationResult;

import java.util.List;

import ai.accurat.sdk.Accurat;

/**
 * Receiver for handling location updates.
 * <p>
 * For apps targeting API level O
 * {@link android.app.PendingIntent#getBroadcast(Context, int, Intent, int)} should be used when
 * requesting location updates. Due to limits on background services,
 * {@link android.app.PendingIntent#getService(Context, int, Intent, int)} should not be used.
 * <p>
 * Note: Apps running on "O" devices (regardless of targetSdkVersion) may receive updates
 * less frequently than the interval specified in the
 * {@link com.google.android.gms.location.LocationRequest} when the app is no longer in the
 * foreground.
 */
public class LocationUpdatesBroadcastReceiver extends BroadcastReceiver {
    private static final String TAG = LocationUpdatesBroadcastReceiver.class.getSimpleName();

    static final String ACTION_LOCATION_UPDATES = "ai.accurat.sdk.action.LOCATION_UPDATES";

    @Override
    public void onReceive(Context context, Intent intent) {
        if (intent == null) {
            return;
        }

        final String action = intent.getAction();
        if (!ACTION_LOCATION_UPDATES.equals(action)) {
            return;
        }

        AccuratLogger.init(context);
        AccuratLogger.log(AccuratLogger.SDK_FLOW, TAG + ".onReceive()");

        AccuratSettingsManager.init(context);
        AccuratSettings setting = AccuratSettingsManager.getSettings();
        if (setting != null && !setting.isSdkEnabled()) {
            AccuratLogger.log(AccuratLogger.WARNING, "Not storing received locations, because the SDK is disabled");

            return;
        }

        LocationResult result = LocationResult.extractResult(intent);
        //noinspection ConstantConditions
        if (result == null) {
            return;
        }

        List<Location> locations = result.getLocations();
        int numLocations = locations.size();
        if (numLocations > 0) {
            //noinspection unchecked
            new SendLocationUpdateTask().execute(locations);
        }
        AccuratLogger.log(AccuratLogger.SDK_FLOW, "Received " + numLocations + " new locations");
        int lastLocationIndex = numLocations - 1;

        for (int i = 0; i < numLocations; i++) {
            Location location = locations.get(i);
            if (location == null) {
                continue;
            }

            // Only update geofences for the last location update
            AccuratLocationManager.getInstance().onLocationChanged(
                    context,
                    location,
                    LocationContext.getLocationContext(),
                    i == lastLocationIndex
            );
        }
    }

    public static class SendLocationUpdateTask extends AsyncTask<List<Location>, Void, Void> {

        @SafeVarargs
        @Override
        protected final Void doInBackground(List<Location>... lists) {
            if (Accurat.onLocationUpdateCallback != null) {
                for (List<Location> locations : lists) {
                    Accurat.onLocationUpdateCallback.onLocationUpdated(locations);
                }
            }

            return null;
        }
    }
}