package ai.accurat.sdk.core;

import android.content.Context;
import androidx.annotation.NonNull;
import android.util.Log;
import android.util.Pair;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;

import ai.accurat.sdk.constants.StorageKeys;

/**
 * A class for mimicking multi-process SharedPreferences.
 * Warning: Race conditions do apply, you should only write from a single process!
 * Warning: Getting a value might commit changes.
 * <p>
 * Reasoning for the warnings: We'll only use this type of storage in specific use cases, for which
 * the warnings don't apply. We'll only store in one process and we don't mind data being committed
 * when getting values.
 *
 * @author Kenneth Saey
 * @Accurat
 * @since 22-06-2018 09:46.
 */
public class MultiProcessStorage {

    private static final String TAG = MultiProcessStorage.class.getSimpleName();

    // <editor-fold desc="Fields">
    private File mStorageFile;

    private boolean mIsDirty;
    private HashMap<String, DataValue> mValues = new HashMap<>();
    // </editor-fold>

    // <editor-fold desc="Construction">
    private MultiProcessStorage(Context context, String name) {
        mStorageFile = new File(context.getFilesDir() + File.separator + name);
    }

    public static MultiProcessStorage getAccuratStorage(@NonNull Context context) {
        AccuratLogger.init(context);

        return getStorage(context, StorageKeys.ACCURAT_MULTI_PROCESS_STORAGE);
    }

    public static MultiProcessStorage getStorage(@NonNull Context context, @NonNull String name) {
        if (name.isEmpty()) {
            return null;
        }

        if (name.contains(":")) {
            throw new IllegalArgumentException("Storage name may not contain colons (:).");
        }

        return new MultiProcessStorage(context, name);
    }
    // </editor-fold>

    // <editor-fold desc="Loading">
    public String dump() {
        StringBuilder builder = new StringBuilder();
        try {
            BufferedReader reader = new BufferedReader(new FileReader(mStorageFile));
            String line;
            while ((line = reader.readLine()) != null) {
                builder.append(line)
                        .append(" | ");
            }
            reader.close();

            String dump = builder.toString();
            return dump == null ? " " : " " + dump;
        } catch (java.io.IOException e) {
            return null;
        }
    }

    private synchronized HashMap<String, DataValue> fetch() {
        List<String> lines = new ArrayList<>();
        try {
            BufferedReader reader = new BufferedReader(new FileReader(mStorageFile));
            HashMap<String, DataValue> values = new HashMap<>();
            String line;
            while ((line = reader.readLine()) != null) {
                Pair<String, DataValue> keyValuePair = parseLine(line);
                if (keyValuePair != null) {
                    values.put(keyValuePair.first, keyValuePair.second);
                }
            }
            reader.close();

            return values;
        } catch (java.io.IOException e) {
            AccuratLogger.log(AccuratLogger.ERROR, "Could not read MultiProcessStorage: " + e.getMessage());
            return null;
        }
    }

    private Pair<String, DataValue> parseLine(String line) {
        String[] parts = line.split(":", 3);
        if (parts.length < 2 || parts.length > 3 || parts[0].length() <= 0 || parts[1].length() <= 0) {
            return null;
        }

        if (parts.length == 2) {
            return new Pair<>(parts[0], DataValue.create());
        }

        DataType dataType;
        try {
            dataType = DataType.valueOf(parts[1]);
        } catch (IllegalArgumentException e) {
            Log.e(TAG, "Illegal data type '" + parts[1] + "': " + e.getMessage());

            return null;
        }

        DataValue dataValue = null;
        try {
            switch (dataType) {
                case BOOLEAN:
                    dataValue = DataValue.create(DataType.BOOLEAN, Boolean.valueOf(parts[2]));
                    break;
                case INTEGER:
                    dataValue = DataValue.create(dataType, Integer.valueOf(parts[2]));
                    break;
                case LONG:
                    dataValue = DataValue.create(dataType, Long.valueOf(parts[2]));
                    break;
                case FLOAT:
                    dataValue = DataValue.create(dataType, Float.valueOf(parts[2]));
                    break;
                case DOUBLE:
                    dataValue = DataValue.create(dataType, Double.valueOf(parts[2]));
                    break;
                case STRING:
                    dataValue = DataValue.create(dataType, parts[2]);
                    break;
            }
        } catch (Exception e) {
            Log.e(TAG, e.getMessage());
            e.printStackTrace();
            Log.v(TAG, dump());
        }

        if (dataValue == null) {
            return null;
        }

        return new Pair<>(parts[0], dataValue);
    }

    private void load() {
        HashMap<String, DataValue> storageValues = fetch();
        if (storageValues == null) {
            storageValues = new HashMap<>();
        }

        // Merge the in-memory values into the values fetched from File storage.
        for (String key : mValues.keySet()) {
            storageValues.put(key, mValues.get(key));
        }

        mValues = storageValues;
    }
    // </editor-fold>

    // <editor-fold desc="Storing">

    /**
     * Commit the changes to device storage.
     */
    public void commit() {
        if (!mIsDirty) {
            // Nothing to commit to storage.
            return;
        }

        load();
        store(mValues);

        mIsDirty = false;
    }

    private synchronized boolean store(HashMap<String, DataValue> values) {
        try {
            mStorageFile.createNewFile();
            FileWriter fw = new FileWriter(mStorageFile);
            BufferedWriter out = new BufferedWriter(fw);
            out.write(serialize(values));
            out.flush();
            out.close();

            return true;
        } catch (IOException e) {
            AccuratLogger.log(AccuratLogger.ERROR, "Could not store MultiProcessStorage: " + e.getMessage());
            return false;
        }
    }

    private String serialize(HashMap<String, DataValue> values) {
        if (values == null || values.isEmpty()) {
            return "";
        }

        List<String> lines = new ArrayList<>();
        for (String key : values.keySet()) {
            DataValue value = values.get(key);
            if (value == null) {
                continue;
            }

            lines.add(
                    key + ":" + value.mDataType.name() +
                            (value.mDataType == DataType.NULL ? "" : ":" + value.getValue()));
        }

        return implode(lines, "\n");
    }

    private String implode(List<String> parts, String separator) {
        if (parts == null || parts.isEmpty()) {
            return "";
        }

        StringBuilder implosion = new StringBuilder(parts.get(0));
        for (int i = 1; i < parts.size(); i++) {
            implosion.append(separator).append(parts.get(i));
        }

        return implosion.toString();
    }
    // </editor-fold>

    // <editor-fold desc="Setters">
    public MultiProcessStorage remove(@NonNull String key) {
        load();
        mValues.remove(key);
        store(mValues);

        return this;
    }

    public MultiProcessStorage setValue(@NonNull String key, boolean value) {
        DataValue dataValue = DataValue.create(DataType.BOOLEAN, value);
        if (dataValue != null) {
            mIsDirty = true;
            mValues.put(key, dataValue);
        }

        return this;
    }

    public MultiProcessStorage setValue(@NonNull String key, int value) {
        DataValue dataValue = DataValue.create(DataType.INTEGER, value);
        if (dataValue != null) {
            mIsDirty = true;
            mValues.put(key, dataValue);
        }

        return this;
    }

    public MultiProcessStorage setValue(@NonNull String key, long value) {
        DataValue dataValue = DataValue.create(DataType.LONG, value);
        if (dataValue != null) {
            mIsDirty = true;
            mValues.put(key, dataValue);
        }

        return this;
    }

    public MultiProcessStorage setValue(@NonNull String key, float value) {
        DataValue dataValue = DataValue.create(DataType.FLOAT, value);
        if (dataValue != null) {
            mIsDirty = true;
            mValues.put(key, dataValue);
        }

        return this;
    }

    public MultiProcessStorage setValue(@NonNull String key, double value) {
        DataValue dataValue = DataValue.create(DataType.DOUBLE, value);
        if (dataValue != null) {
            mIsDirty = true;
            mValues.put(key, dataValue);
        }

        return this;
    }

    public MultiProcessStorage setValue(@NonNull String key, String value) {
        DataValue dataValue = DataValue.create(DataType.STRING, value);
        if (dataValue != null) {
            mIsDirty = true;
            mValues.put(key, dataValue);
        }

        return this;
    }
    // </editor-fold>

    // <editor-fold desc="Getters">
    public boolean hasKey(@NonNull String key) {
        if (mValues == null || !mValues.containsKey(key)) {
            load();
        }

        return mValues != null && mValues.containsKey(key);
    }

    public boolean getBoolean(@NonNull String key, boolean defaultValue) {
        if (mValues == null || !mValues.containsKey(key)) {
            load();
        }

        if (mValues == null || !mValues.containsKey(key)) {
            return defaultValue;
        }

        DataValue dataValue = mValues.get(key);
        if (dataValue == null) {
            return defaultValue;
        }

        if (dataValue.mDataType != DataType.BOOLEAN) {
            return defaultValue;
        }

        return dataValue.getValue();
    }

    public int getInt(@NonNull String key, int defaultValue) {
        if (mValues == null || !mValues.containsKey(key)) {
            load();
        }

        if (mValues == null || !mValues.containsKey(key)) {
            return defaultValue;
        }

        DataValue dataValue = mValues.get(key);
        if (dataValue == null) {
            return defaultValue;
        }

        if (dataValue.mDataType != DataType.INTEGER) {
            return defaultValue;
        }

        return dataValue.getValue();
    }

    public long getLong(@NonNull String key, long defaultValue) {
        if (mValues == null || !mValues.containsKey(key)) {
            load();
        }

        if (mValues == null || !mValues.containsKey(key)) {
            return defaultValue;
        }

        DataValue dataValue = mValues.get(key);
        if (dataValue == null) {
            return defaultValue;
        }

        if (dataValue.mDataType != DataType.LONG) {
            return defaultValue;
        }

        return dataValue.getValue();
    }

    public float getFloat(@NonNull String key, float defaultValue) {
        if (mValues == null || !mValues.containsKey(key)) {
            load();
        }

        if (mValues == null || !mValues.containsKey(key)) {
            return defaultValue;
        }

        DataValue dataValue = mValues.get(key);
        if (dataValue == null) {
            return defaultValue;
        }

        if (dataValue.mDataType != DataType.FLOAT) {
            return defaultValue;
        }

        return dataValue.getValue();
    }

    public double getDouble(@NonNull String key, double defaultValue) {
        if (mValues == null || !mValues.containsKey(key)) {
            load();
        }

        if (mValues == null || !mValues.containsKey(key)) {
            return defaultValue;
        }

        DataValue dataValue = mValues.get(key);
        if (dataValue == null) {
            return defaultValue;
        }

        if (dataValue.mDataType != DataType.DOUBLE) {
            return defaultValue;
        }

        return dataValue.getValue();
    }

    public String getString(@NonNull String key, String defaultValue) {
        if (mValues == null || !mValues.containsKey(key)) {
            load();
        }

        if (mValues == null || !mValues.containsKey(key)) {
            return defaultValue;
        }

        DataValue dataValue = mValues.get(key);
        if (dataValue == null) {
            return defaultValue;
        }

        if (dataValue.mDataType == DataType.NULL) {
            return null;
        }

        if (dataValue.mDataType != DataType.STRING) {
            return defaultValue;
        }

        return dataValue.getValue();
    }
    // </editor-fold>

    // <editor-fold desc="DataType">
    private enum DataType {
        NULL,
        BOOLEAN,
        INTEGER,
        LONG,
        FLOAT,
        DOUBLE,
        STRING
    }
    // </editor-fold>

    // <editor-fold desc="DataValue">
    private static class DataValue {

        private DataType mDataType;
        private Object mValue;

        private DataValue(DataType dataType, Object value) {
            mDataType = dataType;
            mValue = value;
        }

        // <editor-fold desc="Creators">
        static DataValue create() {
            return new DataValue(DataType.NULL, null);
        }

        static DataValue create(DataType dataType) {
            if (dataType != DataType.NULL) {
                return null;
            }

            return new DataValue(dataType, null);
        }

        static DataValue create(DataType dataType, boolean value) {
            if (dataType != DataType.BOOLEAN) {
                return null;
            }

            return new DataValue(dataType, value);
        }

        static DataValue create(DataType dataType, int value) {
            if (dataType != DataType.INTEGER) {
                return null;
            }

            return new DataValue(dataType, value);
        }

        static DataValue create(DataType dataType, long value) {
            if (dataType != DataType.LONG) {
                return null;
            }

            return new DataValue(dataType, value);
        }

        static DataValue create(DataType dataType, float value) {
            if (dataType != DataType.FLOAT) {
                return null;
            }

            return new DataValue(dataType, value);
        }

        static DataValue create(DataType dataType, double value) {
            if (dataType != DataType.DOUBLE) {
                return null;
            }

            return new DataValue(dataType, value);
        }

        static DataValue create(DataType dataType, String value) {
            if (dataType != DataType.STRING) {
                return null;
            }

            return new DataValue(dataType, value);
        }
        // </editor-fold>

        // <editor-fold desc="Getters">
        public <T> T getValue() {
            return (T) mValue;
        }
        // </editor-fold>
    }
    // </editor-fold>
}
