package ai.accurat.sdk.core;

import android.os.Parcel;
import android.os.Parcelable;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Endpoint class replacing OpenLocate.Endpoint for the general implementation.
 *
 * @since 1.0.0
 */
public class OpenLocateBasedEndpoint implements Parcelable {

    public static final String URL = "url";
    public static final String HEADERS = "headers";
    public static final String HEADERS_KEY = "key";
    public static final String HEADERS_VALUE = "value";

    public static List<OpenLocateBasedEndpoint> fromJson(String json) throws JSONException {

        JSONArray jsonArray = new JSONArray(json);
        List<OpenLocateBasedEndpoint> result = new ArrayList<>(jsonArray.length());

        for (int i = 0; i < jsonArray.length(); i++) {
            JSONObject jsonOpenLocateBasedEndpoint = jsonArray.getJSONObject(i);
            Builder builder = OpenLocateBasedEndpoint.builder(jsonOpenLocateBasedEndpoint.getString(URL));
            JSONArray headers = jsonOpenLocateBasedEndpoint.getJSONArray(HEADERS);
            for (int j = 0; j < headers.length(); j++) {
                JSONObject header = headers.getJSONObject(j);
                builder.withHeader(header.getString(HEADERS_KEY), header.getString(HEADERS_VALUE));
            }
            result.add(builder.build());
        }

        return result;
    }

    public static String toJson(List<OpenLocateBasedEndpoint> endpoints) throws JSONException {
        JSONArray jsonArray = new JSONArray();

        for (OpenLocateBasedEndpoint endpoint : endpoints) {

            JSONObject jsonOpenLocateBasedEndpoint = new JSONObject();
            jsonOpenLocateBasedEndpoint.put(URL, endpoint.url);

            JSONArray jsonHeaders = new JSONArray();
            if (endpoint.getHeaders() != null) {
                for (Map.Entry<String, String> entry : endpoint.getHeaders().entrySet()) {
                    JSONObject header = new JSONObject();
                    header.put(HEADERS_KEY, entry.getKey());
                    header.put(HEADERS_VALUE, entry.getValue());
                    jsonHeaders.put(header);
                }
            }
            jsonOpenLocateBasedEndpoint.put(HEADERS, jsonHeaders);

            jsonArray.put(jsonOpenLocateBasedEndpoint);
        }

        return jsonArray.toString();
    }

    private String url;

    private HashMap<String, String> headers;

    public OpenLocateBasedEndpoint(String url, HashMap<String, String> headers) {
        this.url = url;
        this.headers = headers;
    }

    private OpenLocateBasedEndpoint(Builder builder) {
        this.url = builder.url;
        this.headers = builder.headers;
    }

    public String getUrl() {
        return url;
    }

    public HashMap<String, String> getHeaders() {
        return headers;
    }

    public static Builder builder(String url) {
        return new Builder(url);
    }

    public static class Builder {

        private String url;

        private HashMap<String, String> headers;

        public Builder(String url) {
            this.url = url;
        }

        public Builder withHeader(String key, String value) {

            if (headers == null) {
                headers = new HashMap<>();
            }

            headers.put(key, value);
            return this;
        }

        public Builder withHeaders(Map<String, String> headers) {

            if (this.headers == null) {
                this.headers = new HashMap<>();
            }

            this.headers.putAll(headers);
            return this;
        }

        public OpenLocateBasedEndpoint build() {
            return new OpenLocateBasedEndpoint(this.url, this.headers);
        }
    }


    @Override
    public int describeContents() {
        return 0;
    }

    @Override
    public void writeToParcel(Parcel dest, int flags) {
        dest.writeString(this.url);
        dest.writeInt(this.headers.size());
        for (Map.Entry<String, String> entry : this.headers.entrySet()) {
            dest.writeString(entry.getKey());
            dest.writeString(entry.getValue());
        }
    }

    protected OpenLocateBasedEndpoint(Parcel in) {
        this.url = in.readString();
        int headersSize = in.readInt();
        this.headers = new HashMap<>(headersSize);
        for (int i = 0; i < headersSize; i++) {
            String key = in.readString();
            String value = in.readString();
            this.headers.put(key, value);
        }
    }

    public static final Parcelable.Creator<OpenLocateBasedEndpoint> CREATOR = new Creator<OpenLocateBasedEndpoint>() {
        @Override
        public OpenLocateBasedEndpoint createFromParcel(Parcel source) {
            return new OpenLocateBasedEndpoint(source);
        }

        @Override
        public OpenLocateBasedEndpoint[] newArray(int size) {
            return new OpenLocateBasedEndpoint[size];
        }
    };

    @Override
    public String toString() {
        return "{url:" + url + "}";
    }

}
