package ai.accurat.sdk.constants;

import android.Manifest;
import android.content.Context;
import android.content.pm.PackageManager;
import android.os.Build;

import androidx.core.content.ContextCompat;

import ai.accurat.sdk.core.AccuratLogger;

/**
 * @author Kenneth Saey
 * @since 22-06-2018 09:09.
 */
public enum ConsentType {

    TRACKING(
            Build.VERSION.SDK_INT >= Build.VERSION_CODES.Q
                    ? new String[]{Manifest.permission.ACCESS_BACKGROUND_LOCATION, Manifest.permission.ACCESS_COARSE_LOCATION, Manifest.permission.ACCESS_FINE_LOCATION}
                    : new String[]{Manifest.permission.ACCESS_COARSE_LOCATION, Manifest.permission.ACCESS_FINE_LOCATION}
            , RequestCodes.LOCATION_PERMISSIONS),
    GDPR(new String[0], RequestCodes.NONE),
    LOCATION(new String[0], RequestCodes.NONE);

    private final String[] permissions;
    private final int requestCode;

    ConsentType(String[] permissions, int requestCode) {
        this.permissions = permissions;
        this.requestCode = requestCode;
    }

    // <editor-fold desc="Getters">
    public String[] getPermissions() {
        return permissions;
    }

    public int getRequestCode() {
        return requestCode;
    }

    public static ConsentType[] all() {
        return new ConsentType[]{
                TRACKING
        };
    }
    // </editor-fold>

    public boolean hasPermissions(Context context) {
        boolean granted = true;
        for (String permission : permissions) {
            if (permission.contains("BACKGROUND")) {
                // BACKGROUND permissions are not required, the SDK should work when only foreground permissions are accepted
                continue;
            }

            AccuratLogger.log(AccuratLogger.SDK_FLOW, "Checking permission " + permission);
            granted &= ContextCompat.checkSelfPermission(context, permission) == PackageManager.PERMISSION_GRANTED;
            AccuratLogger.log(AccuratLogger.SDK_FLOW, "Granted?: " + (ContextCompat.checkSelfPermission(context, permission) == PackageManager.PERMISSION_GRANTED));
        }

        return granted;
    }

    public boolean hasBackgroundPermission(Context context) {
        boolean granted = true;
        for (String permission : permissions) {
            if (!permission.contains("BACKGROUND")) {
                // We're only checking background permissions here, so skip this one
                continue;
            }

            AccuratLogger.log(AccuratLogger.SDK_FLOW, "Checking background permission: " + permission);
            granted &= ContextCompat.checkSelfPermission(context, permission) == PackageManager.PERMISSION_GRANTED;
            AccuratLogger.log(AccuratLogger.SDK_FLOW, "Granted?: " + (ContextCompat.checkSelfPermission(context, permission) == PackageManager.PERMISSION_GRANTED));
        }

        return granted;
    }

    public static ConsentType fromServerType(String serverType) {
        switch (serverType) {
            case "tracking":
            case "gdpr":
                return TRACKING;
            default:
                return null;
        }
    }

    public static ConsentType normalise(ConsentType consentType) {
        if (consentType == null) {
            return null;
        }

        switch (consentType) {
            case TRACKING:
            case GDPR:
            case LOCATION:
            default:
                return TRACKING;
        }
    }
}
