package ai.accurat.sdk.core;

import android.location.Location;
import android.text.TextUtils;

import com.google.gson.annotations.SerializedName;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.Date;


/**
 * Geofence model class.
 *
 * @since 2.0.0
 */
public class AccuratGeofence implements JsonObjectType {

    private static final String TAG = AccuratGeofence.class.getSimpleName();

    // <editor-fold desc="JSON keys">
    private static final String JSON_KEY_ID = "AccuratGeofence.id";
    private static final String JSON_KEY_LATITUDE = "AccuratGeofence.latitude";
    private static final String JSON_KEY_LONGITUDE = "AccuratGeofence.longitude";
    private static final String JSON_KEY_NIDS = "AccuratGeofence.nids";
    private static final String JSON_KEY_POLYGON = "AccuratGeofence.polygon";
    private static final String JSON_KEY_RADIUS = "AccuratGeofence.radius";
    private static final String JSON_KEY_CREATED_AT = "AccuratGeofence.createdAt";
    private static final String JSON_KEY_TYPE = "AccuratGeofence.type";
    private static final String JSON_KEY_DWELL_TIME = "AccuratGeofence.dwell_time";
    private static final String JSON_KEY_DATA = "AccuratGeofence.data";
    // </editor-fold>

    // <editor-fold desc="Members">
    @SerializedName("id")
    private String id;
    @SerializedName("lat")
    private Double latitude;
    @SerializedName("lng")
    private Double longitude;
    @SerializedName("nids")
    private String nids;
    @SerializedName("polygon")
    private Double[][] polygon;
    @SerializedName("radius")
    private int radius;
    @SerializedName("type")
    private String type;
    @SerializedName("dwell_time")
    private int dwellTime;
    @SerializedName("data")
    private String data;

    private transient Date createdAt;
    // </editor-fold>

    // <editor-fold desc="Initialisation">
    public AccuratGeofence() {
    }

    public AccuratGeofence(String id, Double latitude, Double longitude, String nids, Double[][] polygon, int radius) {
        this.id = id;
        this.latitude = latitude;
        this.longitude = longitude;
        this.nids = nids;
        this.polygon = polygon;
        this.radius = radius;
    }
    // </editor-fold>

    // <editor-fold desc="Getters & Setters">
    public String getId() {
        return this.id;
    }

    public String getNids() {
        return this.nids;
    }

    public Double getLatitude() {
        return this.latitude;
    }

    public Double getLongitude() {
        return this.longitude;
    }

    public Date getCreatedAt() {
        return createdAt;
    }

    public void setCreatedAt(Date createdAt) {
        this.createdAt = createdAt;
    }

    public int getRadius() {
        return radius;
    }

    public Long[] getExtractedNotificationIds() {
        if (TextUtils.isEmpty(nids)) {
            return new Long[0];
        }

        String[] notificationIds = nids.split("\\|");
        ArrayList<Long> extractedNotificationIds = new ArrayList<>();

        for (String notificationId : notificationIds) {
            try {
                extractedNotificationIds.add(Long.parseLong(notificationId));
            } catch (NumberFormatException ignored) {
            }
        }

        return extractedNotificationIds.toArray(new Long[0]);
    }

    public String getType() {
        return type;
    }

    public int getDwellTime() {
        return dwellTime;
    }

    public String getData() {
        return data;
    }
    // </editor-fold>

    // <editor-fold desc="Helpers">
    public boolean contains(LocationInterface currentLocation) {
        double distance = LocationUtils.distance(longitude, latitude, currentLocation.getLocationInfo().getLongitude(), currentLocation.getLocationInfo().getLatitude());
        boolean contains = distance <= radius;
        AccuratLogger.log(AccuratLogger.GEOFENCE, "Checking if geofence "
                + id + " [ "
                + latitude + " | "
                + longitude + " ; r = "
                + radius + "] contains location [ "
                + currentLocation.getLocationInfo().getLatitude() + " | "
                + currentLocation.getLocationInfo().getLongitude() + " ] => "
                + (contains ? "YES" : "NO") + " (distance = "
                + distance + " )"
        );
        return contains;
    }

    public boolean contains(Location location) {
        double distance = LocationUtils.distance(longitude, latitude, location.getLongitude(), location.getLatitude());
        boolean contains = distance <= radius;
        AccuratLogger.log(AccuratLogger.GEOFENCE, "Checking if geofence "
                + id + " [ "
                + latitude + " | "
                + longitude + " ; r = "
                + radius + "] contains location [ "
                + location.getLatitude() + " | "
                + location.getLongitude() + " ] => "
                + (contains ? "YES" : "NO") + " (distance = "
                + distance + " )"
        );
        return contains;
    }
    // </editor-fold>

    // <editor-fold desc="JSON Handling">
    public static AccuratGeofence fromJson(String jsonString) throws JSONException {
        if (jsonString == null || jsonString.isEmpty()) {
            return null;
        }

        JSONObject json = new JSONObject(jsonString);
        AccuratGeofence geofence = new AccuratGeofence();

        geofence.id = json.has(JSON_KEY_ID) ? json.getString(JSON_KEY_ID) : geofence.id;
        geofence.latitude = json.has(JSON_KEY_LATITUDE) ? json.getDouble(JSON_KEY_LATITUDE) : geofence.latitude;
        geofence.longitude = json.has(JSON_KEY_LONGITUDE) ? json.getDouble(JSON_KEY_LONGITUDE) : geofence.longitude;
        geofence.nids = json.has(JSON_KEY_NIDS) ? json.getString(JSON_KEY_NIDS) : geofence.nids;
        geofence.polygon = json.has(JSON_KEY_POLYGON) ? fromJson(json.getJSONArray(JSON_KEY_POLYGON)) : geofence.polygon;
        geofence.radius = json.has(JSON_KEY_RADIUS) ? json.getInt(JSON_KEY_RADIUS) : geofence.radius;
        geofence.createdAt = json.has(JSON_KEY_CREATED_AT) && !json.isNull(JSON_KEY_CREATED_AT) ? new Date(json.getLong(JSON_KEY_CREATED_AT)) : geofence.createdAt;
        geofence.type = json.optString(JSON_KEY_TYPE);
        geofence.dwellTime = json.optInt(JSON_KEY_DWELL_TIME, 0);
        geofence.data = json.optString(JSON_KEY_DATA);

        return geofence;
    }

    @Override
    public JSONObject getJson() {
        JSONObject json = new JSONObject();
        try {
            json.put(JSON_KEY_ID, id);
            json.put(JSON_KEY_LATITUDE, latitude);
            json.put(JSON_KEY_LONGITUDE, longitude);
            json.put(JSON_KEY_NIDS, nids);
            json.put(JSON_KEY_POLYGON, toJson(polygon));
            json.put(JSON_KEY_RADIUS, radius);
            json.put(JSON_KEY_CREATED_AT, createdAt == null ? null : createdAt.getTime());
            json.put(JSON_KEY_TYPE, type);
            json.put(JSON_KEY_DWELL_TIME, dwellTime);
            json.put(JSON_KEY_DATA, data);
        } catch (JSONException e) {
            AccuratLogger.log(AccuratLogger.JSON_ERROR, TAG + ".getJson(): " + e.getMessage());
            e.printStackTrace();
        }

        return json;
    }

    // </editor-fold>

    // <editor-fold desc="JSON Helpers">
    private static Double[][] fromJson(JSONArray jsonMatrix) {
        if (jsonMatrix == null) {
            return null;
        }

        int rowCount = jsonMatrix.length();
        if (rowCount == 0) {
            return null;
        }

        try {
            JSONArray firstRow = jsonMatrix.getJSONArray(0);
            if (firstRow == null) {
                return null;
            }

            int columnCount = firstRow.length();
            if (columnCount == 0) {
                return null;
            }

            Double[][] matrix = new Double[rowCount][columnCount];
            for (int i = 0; i < rowCount; i++) {
                JSONArray row = jsonMatrix.getJSONArray(i);
                if (row == null || row.length() == 0) {
                    continue;
                }

                for (int j = 0; j < row.length(); j++) {
                    matrix[i][j] = row.getDouble(j);
                }
            }

            return matrix;
        } catch (JSONException e) {
            AccuratLogger.log(AccuratLogger.JSON_ERROR, TAG + ".fromJson(): " + e.getMessage());

            return null;
        }
    }

    private static JSONArray toJson(Double[][] matrix) {
        if (matrix == null) {
            return null;
        }

        JSONArray jsonMatrix = new JSONArray();
        for (int i = 0; i < matrix.length; i++) {
            JSONArray jsonRow = new JSONArray();
            for (int j = 0; j < matrix[i].length; j++) {
                jsonRow.put(matrix[i][j]);
            }
            jsonMatrix.put(jsonRow);
        }

        return jsonMatrix;
    }
    // </editor-fold>


    @Override
    public String toString() {
        return getJson().toString();
    }

    public String getLogString() {
        return "[ " + latitude + ", " + longitude + " , r = " + radius + "m]";
    }
}
