package ai.accurat.sdk.core;

import android.content.Context;
import android.text.TextUtils;
import android.util.Log;

import com.android.volley.VolleyError;

import org.json.JSONArray;
import org.json.JSONObject;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.PrintWriter;
import java.nio.charset.StandardCharsets;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;
import java.util.Locale;

import ai.accurat.sdk.config.Configuration;
import ai.accurat.sdk.constants.HttpMethod;
import ai.accurat.sdk.constants.StorageKeys;
import ai.accurat.sdk.managers.AccuratConfigurationManager;
import ai.accurat.sdk.managers.RealmManager;

/**
 * @author Kenneth Saey
 * @Accurat
 * @since 31-05-2018 09:58.
 */
public class AccuratLogger {

    // <editor-fold desc="Tags">
    public static final String METHOD_START = "SDK_FLOW - METHOD_START";
    public static final String METHOD_END = "SDK_FLOW - METHOD_END";
    public static final String SDK_FLOW = "SDK_FLOW";
    public static final String ADVERTISING = "ADVERTISING";
    public static final String TRACKER = "TRACKER";
    public static final String GEOFENCE = "GEOFENCE";
    public static final String NOTIFICATION = "NOTIFICATION";
    public static final String DISPATCHER = "DISPATCHER";
    public static final String DISPATCH = "DISPATCH";
    public static final String DATABASE = "DATABASE";
    public static final String STORAGE = "STORAGE";
    public static final String STORAGE_DATA = "STORAGE - DATA";
    public static final String NETWORK = "NETWORK";
    public static final String NETWORK_REQUEST = "NETWORK - REQUEST";
    public static final String NETWORK_RESPONSE = "NETWORK - RESPONSE";
    public static final String NETWORK_ERROR = "NETWORK - ERROR";
    public static final String NETWORK_REQUEST_DATA = "NETWORK - REQUEST - DATA";
    public static final String NETWORK_RESPONSE_DATA = "NETWORK - RESPONSE - DATA";
    public static final String NETWORK_ERROR_DATA = "NETWORK - ERROR - DATA";
    public static final String SETTINGS = "SETTINGS";
    public static final String SETTINGS_DATA = "SETTINGS - DATA";
    public static final String WORKMANAGER = "WORKMANAGER";
    public static final String WARNING = "WARNING";
    public static final String ERROR = "ERROR";
    public static final String JSON_ERROR = "JSON_ERROR";
    public static final String NONE = null;
    // </editor-fold>

    private static final String LOG_FILE_NAME = "accurat_debug_log";
    private static final String LOG_FILE_EXT = ".txt";
    public static final int MAX_LOG_LENGTH = 1000;

    private static final AccuratLogger INSTANCE = new AccuratLogger();
    public static final String TAG = "AccuratLogger";
    private static final String LAST_LOG_TIMESTAMP = "last_log_timestamp";

    private boolean initialized = false;
    private SimpleDateFormat timeFormat = new SimpleDateFormat("dd-MM-yyyy HH:mm:ss", Locale.getDefault());
    private SimpleDateFormat logFileTimeFormat = new SimpleDateFormat("yyyyMMdd_HHmm", Locale.getDefault());
    private SharedPreferenceUtils storage;

    private File logFile;
    private OutputStreamWriter logStream;

    private Context context;

    // <editor-fold desc="Initialisation">
    private AccuratLogger() {
    }

    public static void init(Context context) {
        if (!INSTANCE.initialized) {
            INSTANCE.context = context;
            INSTANCE.createFile(context);
            INSTANCE.storage = SharedPreferenceUtils.getInstance(context);

            INSTANCE.initialized = true;
            RealmManager.init(context);
        }
    }

    private void createFile(Context context) {
        // Todo - Write to a more secure location
        logFile = new File(context.getFilesDir() + File.separator + LOG_FILE_NAME + LOG_FILE_EXT);
    }

    private static void checkInitialized() {
        if (!INSTANCE.initialized) {
            throw new IllegalStateException("AccuratLogger has not been initialized.");
        }
    }

    public static File getLogFile() {
        checkInitialized();

        return INSTANCE.logFile;
    }

    // </editor-fold>
    public static String getDebugLog() {
        if (INSTANCE.initialized) {
            StringBuilder builder = new StringBuilder();

            try {
                BufferedReader reader = new BufferedReader(new FileReader(INSTANCE.logFile));
                String line;
                while ((line = reader.readLine()) != null) {
                    builder.append(line).append("\n");
                }
                reader.close();

                return builder.toString();
            } catch (IOException e) {
                // File doesn't exist yet
            }
        }

        return "";
    }
    // </editor-fold>

    // <editor-fold desc="Setters">
    public static void log(String tag, String message) {
        if (!INSTANCE.isLoggingEnabled()) {
            return;
        }

        if (TextUtils.isEmpty(message)) {
            return;
        }

        StringBuilder logMessage = new StringBuilder(prependCurrentTime(""));
        if (!TextUtils.isEmpty(tag)) {
            logMessage.append(tag)
                    .append(" - ");
        }

        logMessage.append(
                message,
                0,
                Math.min(MAX_LOG_LENGTH, message.length())
        ).append("\n");

        try {
            //noinspection ResultOfMethodCallIgnored
            INSTANCE.logFile.createNewFile();
            FileOutputStream fos = new FileOutputStream(INSTANCE.logFile, true);
            OutputStreamWriter logStream = new OutputStreamWriter(fos, StandardCharsets.UTF_8);
            PrintWriter logWriter = new PrintWriter(logStream);
            logWriter.append(logMessage);
            logWriter.flush();
            logWriter.close();
            logToConsole(tag, message);
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    public static void logNetworkRequest(HttpMethod method, String url) {
        logNetworkRequest(method, url, null, true);
    }

    public static void logNetworkRequest(HttpMethod method, String url, JSONObject data, boolean expectNullData) {
        log(AccuratLogger.NETWORK_REQUEST, "Calling " + method.name() + " " + url);
        boolean expected = (data == null) == expectNullData;
        log(AccuratLogger.NETWORK_REQUEST_DATA, (expected ? "[expected] " : "[unexpected]") + nullSafe(data));
    }

    public static void logNetworkResponse(HttpMethod method, String url, HttpResponse response, boolean expectResponse) {
        log(AccuratLogger.NETWORK_RESPONSE, "Response from " + method.name() + " " + url);
        log(
                AccuratLogger.NETWORK_RESPONSE_DATA,
                expectResponse
                        ? nullSafe(response)
                        : "Status code = " + (response == null ? "null" : response.getStatusCode())
        );
    }

    public static void logNetworkResponse(HttpMethod method, String url, JSONObject response, boolean expectNullResponse) {
        log(AccuratLogger.NETWORK_RESPONSE, "Response from " + method.name() + " " + url);
        log(AccuratLogger.NETWORK_RESPONSE_DATA, nullSafe(response, expectNullResponse));
    }

    public static void logNetworkResponse(HttpMethod method, String url, JSONArray response, boolean expectNullResponse) {
        log(AccuratLogger.NETWORK_RESPONSE, "Response from " + method.name() + " " + url);
        log(AccuratLogger.NETWORK_RESPONSE_DATA, nullSafe(response, expectNullResponse));
    }

    public static void logNetworkError(HttpMethod method, String url, HttpResponse error) {
        log(AccuratLogger.NETWORK_ERROR, "Error from " + method.name() + " " + url);
        log(AccuratLogger.NETWORK_ERROR_DATA, "Status code = " + (error == null ? "null" : error.getStatusCode()));
        log(AccuratLogger.NETWORK_ERROR_DATA, error == null || error.getError() == null ? "null" : error.getError().getMessage());
    }

    public static void logNetworkError(HttpMethod method, String url, VolleyError error) {
        log(AccuratLogger.NETWORK_ERROR, "Error from " + method.name() + " " + url);
        log(AccuratLogger.NETWORK_ERROR_DATA, error == null ? "null" : error.getMessage());
    }
    // </editor-fold>

    private static String nullSafe(Object object) {
        return object == null ? "null" : object.toString();
    }

    private static String nullSafe(Object object, boolean expectNull) {
        boolean expected = (object == null) == expectNull;
        return (expected ? "[expected] " : "[unexpected] ") + (object == null ? "null" : object.toString());
    }

    private static void logToConsole(String tag, String message) {
        if (tag == null) {
            Log.i(TAG, message);

            return;
        }

        switch (tag) {
            case WARNING:
                Log.w(TAG, message);
                break;
            case ERROR:
                Log.e(TAG, message);
                break;
            case METHOD_START:
            case METHOD_END:
            case SDK_FLOW:
            case STORAGE_DATA:
            case NETWORK_REQUEST_DATA:
            case NETWORK_RESPONSE_DATA:
            case NETWORK_ERROR_DATA:
                Log.v(TAG, message);
                break;
            default:
                Log.d(TAG, message);
        }
    }

    public static void removeDebugLog() {
        checkInitialized();
        try {
            //noinspection ResultOfMethodCallIgnored
            INSTANCE.logFile.createNewFile();
            FileOutputStream fos = new FileOutputStream(INSTANCE.logFile);
            OutputStreamWriter logStream = new OutputStreamWriter(fos, StandardCharsets.UTF_8);
            PrintWriter logWriter = new PrintWriter(logStream);
            logWriter.write("");
            logWriter.flush();
            logWriter.close();
        } catch (IOException e) {
            Log.e(TAG, "Something went wrong clearing the log file", e);
            e.printStackTrace();
        }
    }
    // </editor-fold>

    // <editor-fold desc="Helpers">
    private static String prependCurrentTime(String message) {
        Date now = Calendar.getInstance().getTime();
        INSTANCE.branchLogsIfNeeded(now);
        INSTANCE.storeLogTime(now);
        String timestamp = INSTANCE.timeFormat.format(now).replace(' ', '\t');

        return timestamp + '\t' + message;
    }

    private boolean isLoggingEnabled() {
        return initialized && (
                storage.getBooleanValue(StorageKeys.ACCURAT_LOGGING_ENABLED, Configuration.DEFAULT_LOGGING_ENABLED)
                        || AccuratConfigurationManager.hasDebugLogsEnabled()
        );
    }

    private void storeLogTime(Date date) {
        if (date == null || storage == null) {
            return;
        }

        storage.setValue(LAST_LOG_TIMESTAMP, date.getTime());
    }

    private long getLastLogTime() {
        if (storage == null) {
            return 0;
        }

        return storage.getLongValue(LAST_LOG_TIMESTAMP, 0);
    }

    /**
     * If the new log is in a new interval of 4 hours, the previous logs should be branched.
     *
     * @param date The current date and time
     */
    private void branchLogsIfNeeded(Date date) {
        if (!initialized || context == null || date == null) {
            return;
        }

        long previous = getLastLogTime();
        if (previous == 0) {
            return;
        }

        long now = date.getTime();
        long fourHoursInMillis = 14400000;
        if (now / fourHoursInMillis <= previous / fourHoursInMillis) {
            // Still in the same interval of 4 hours
            return;
        }

        File externalDir = context.getExternalFilesDir(null);
        if (externalDir == null) {
            return;
        }

        String logFileTime = logFileTimeFormat.format(new Date((previous/fourHoursInMillis) * fourHoursInMillis));
        File branchedFile = new File(externalDir + File.separator + LOG_FILE_NAME + "_" + logFileTime + LOG_FILE_EXT);
        try {
            //noinspection ResultOfMethodCallIgnored
            branchedFile.createNewFile();
            copyFile(logFile, branchedFile);
            removeDebugLog();
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    public static String[] getLogFiles(Context context) {
        if (!INSTANCE.initialized || context == null) {
            return new String[0];
        }

        File externalDir = context.getExternalFilesDir(null);
        if (externalDir == null) {
            return new String[0];
        }

        return externalDir.list();
    }

    private void copyFile(File source, File destination) throws IOException {
        try (InputStream in = new FileInputStream(source)) {
            try(OutputStream out = new FileOutputStream(destination)) {
                byte[] buffer = new byte[1024];
                int len;
                while((len = in.read(buffer)) > 0) {
                    out.write(buffer, 0, len);
                }
            }
        }
    }
    // </editor-fold>
}
