package ai.accurat.sdk.core;

import android.content.Context;

import androidx.work.BackoffPolicy;
import androidx.work.Constraints;
import androidx.work.Data;
import androidx.work.NetworkType;
import androidx.work.PeriodicWorkRequest;
import androidx.work.WorkManager;

import com.google.android.gms.common.util.CollectionUtils;

import java.util.concurrent.TimeUnit;

import ai.accurat.sdk.callbacks.AccuratProcessCallback;
import ai.accurat.sdk.constants.StorageKeys;

class Dispatcher implements DispatcherInterface {

    private static final String TAG = Dispatcher.class.getSimpleName();
    private Context context;
    private long transmissionIntervalInSecs;
    private MultiProcessStorage storage;
    private String endpoints;

    @Override
    public void initDispatcher(Context context) {
        this.context = context;
        this.storage = MultiProcessStorage.getStorage(context, StorageKeys.ACCURAT_MULTI_PROCESS_STORAGE);
        this.transmissionIntervalInSecs = AccuratSettingsManager.getSettings().getTransmissionIntervalInSecs();
    }

    @Override
    public boolean startDispatcher(AccuratProcessCallback callback) {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".startDispatcher()");
        WorkManagerProvider.initialize(context);
        // Based on commit 9a5dee6248ea231c353ea8763f6234ab62183930
        if (CollectionUtils.isEmpty(AccuratSettingsManager.getSettings().getEndpoints())) {
            AccuratLogger.log(AccuratLogger.WARNING, "No endpoints to dispatch to");
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".startDispatcher()");

            return false;
        }

        WorkManager workManager = WorkManager.getInstance();
        if (workManager != null) {
            // Cancel any previously running dispatch work
            workManager.cancelAllWorkByTag(Constants.DISPATCH_WORK_TAG);

            // Create a new WorkRequest
            PeriodicWorkRequest dispatchWork = new PeriodicWorkRequest.Builder(DispatchWorker.class, transmissionIntervalInSecs, TimeUnit.SECONDS)
                    .setBackoffCriteria(BackoffPolicy.EXPONENTIAL, transmissionIntervalInSecs / 100, TimeUnit.SECONDS)
                    .setConstraints(getDispatchConstraints())
                    .setInputData(getDispatchData())
                    .addTag(Constants.DISPATCH_WORK_TAG)
                    .build();

            // Schedule the dispatch work
            workManager.enqueue(dispatchWork);
            AccuratLogger.log(AccuratLogger.SDK_FLOW, "Scheduled Dispatcher");
        } else {
            AccuratLogger.log(AccuratLogger.ERROR, "Can't schedule Dispatcher, WorkManager.getInstance() is NULL");
        }
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".startDispatcher()");

        return false;
    }

    @Override
    public void stopDispatcher(Context context) {
        stop(context);
    }

    private Constraints getDispatchConstraints() {
        return new Constraints.Builder()
                .setRequiresBatteryNotLow(true)
                .setRequiredNetworkType(NetworkType.CONNECTED)
                .build();
    }

    private Data getDispatchData() {
        return new Data.Builder()
                .putString(Constants.ENDPOINTS_KEY, storage.getString(Constants.ENDPOINTS_KEY, ""))
                .build();
    }

    /**
     * Stops all open work tagged with {@link Constants#DISPATCH_WORK_TAG}.
     */
    public static void stop(Context context) {
        WorkManagerProvider.initialize(context);
        // Cancel all open dispatching work requests
        WorkManager workManager = WorkManager.getInstance(context);
        workManager.cancelAllWorkByTag(Constants.DISPATCH_WORK_TAG);
    }
}
