package ai.accurat.sdk.core;

import android.content.Context;
import androidx.annotation.NonNull;

import androidx.work.Constraints;
import androidx.work.Data;
import androidx.work.NetworkType;
import androidx.work.OneTimeWorkRequest;
import androidx.work.WorkManager;
import androidx.work.Worker;
import androidx.work.WorkerParameters;

import com.google.gson.Gson;

/**
 * Worker which will fetch geofence notifications once network is available.
 */
public class GeofencesNotificationPlanner extends Worker {
    private static final String TAG = GeofencesNotificationPlanner.class.getSimpleName();
    private static final String TRIGGERED_GEOFENCE = "triggered_geofence";
    private static final String TRIGGER_TIMESTAMP = "trigger_timestamp";
    private final long HALF_AN_HOUR_IN_MILLIS = 30 * 60 * 1000;

    public GeofencesNotificationPlanner(@NonNull Context context, @NonNull WorkerParameters workerParams) {
        super(context, workerParams);
    }

    @NonNull
    @Override
    public Result doWork() {
        AccuratLogger.init(getApplicationContext());
        AccuratLogger.log(AccuratLogger.WORKMANAGER, TAG + ".doWork()");
        final long currentTime = System.currentTimeMillis();
        final AccuratGeofence geofence = new Gson().fromJson(getInputData().getString(TRIGGERED_GEOFENCE), AccuratGeofence.class);
        final long timestamp = getInputData().getLong(TRIGGER_TIMESTAMP, 0L);

        // Verify whether the notification is still relevant.
        if (timestamp + HALF_AN_HOUR_IN_MILLIS < currentTime) {
            AccuratLogger.log(AccuratLogger.NOTIFICATION, "Delayed notification for geofence " + geofence.getId() + " has expired. Will not fetch and post the notification.");
            AccuratLogger.log(AccuratLogger.WORKMANAGER, TAG + " - Work done, returning Result.success()");
            return Result.success();
        }

        // Attempt to show the notification again, will only retry once since this is a retry already.
        GeofencesManager.showNotificationForGeofence(getApplicationContext(), geofence, true,
                success -> {
                    if (success) {
                        AccuratLogger.log(AccuratLogger.NOTIFICATION, "Successfully posted notification for geofence with ID " + geofence.getId() + " after the network connection was restored");
                    } else {
                        AccuratLogger.log(AccuratLogger.WARNING, "Failed to post notification for geofence with ID " + geofence.getId() + " after the network connection was restored");
                    }
                });

        AccuratLogger.log(AccuratLogger.WORKMANAGER, TAG + " - Work done, returning Result.success()");
        return Result.success();
    }

    /**
     * Plan to show a notification when network access is regained.
     * Should expire after half an hour.
     *
     * @param geofence Geofence that has been triggered and for which a notification should be shown.
     */
    static void planNotificationForGeofence(@NonNull AccuratGeofence geofence) {
        Constraints constraints = new Constraints.Builder()
                .setRequiredNetworkType(NetworkType.CONNECTED)
                .build();
        OneTimeWorkRequest workRequest = new OneTimeWorkRequest.Builder(GeofencesNotificationPlanner.class)
                .setConstraints(constraints)
                .setInputData(new Data.Builder()
                        .putString(TRIGGERED_GEOFENCE, new Gson().toJson(geofence))
                        .putLong(TRIGGER_TIMESTAMP, System.currentTimeMillis())
                        .build())
                .build();
        WorkManager.getInstance().enqueue(workRequest);
        AccuratLogger.log(AccuratLogger.SDK_FLOW, "Planned to fetch and show a geofence notification when network is available");
    }

}
