package ai.accurat.sdk.core;

import android.content.Context;
import androidx.annotation.NonNull;

import java.util.Date;
import java.util.Random;

import ai.accurat.sdk.config.Configuration;
import ai.accurat.sdk.constants.StorageKeys;

public class PeriodicSync {

    // <editor-fold desc="Fields">
    private static final String TAG = PeriodicSync.class.getSimpleName();
    private static final PeriodicSync INSTANCE = new PeriodicSync();

    private static MultiProcessStorage storage;
    // </editor-fold>

    // <editor-fold desc="Initialisation">
    public static void init(@NonNull Context context) {
        if (!isInitialized()) {
            AccuratLogger.init(context);
            storage = MultiProcessStorage.getStorage(context, StorageKeys.ACCURAT_MULTI_PROCESS_STORAGE);
        }
    }

    private static boolean isInitialized() {
        return storage != null;
    }

    private static void checkInitialized() {
        if (!isInitialized()) {
            throw new IllegalStateException("PeriodicSync has not yet been initialised.");
        }
    }
    // </editor-fold>

    // <editor-fold desc="Public interface">
    static long getSyncTime(String key) {
        return getSyncTime(key, Configuration.SYNC_WINDOW_START_SECONDS, Configuration.SYNC_WINDOW_END_SECONDS);
    }

    private static long getSyncTime(String key, int syncWindowStartSeconds, int syncWindowEndSeconds) {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".getSyncTime(" + key + ")");
        checkInitialized();

        long fiveMinutes = 300000;
        long oneDay = 86400000;

        long syncMillis = generateSyncMillis(syncWindowStartSeconds, syncWindowEndSeconds);
        long now = System.currentTimeMillis();
        long midnight = (now / oneDay) * oneDay;
        long syncTime = midnight + syncMillis;

        if (syncTime <= now + fiveMinutes) {// Add an additional 5 minutes to allow worker threads to settle.
            syncTime += oneDay;
        }

        AccuratLogger.log(AccuratLogger.SDK_FLOW, "syncTime = " + Configuration.LOG_TIME_FORMAT.format(new Date(syncTime)) + " (" + syncTime + ")");
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".getSyncTime(" + key + ")");

        return syncTime;
    }

    private static long generateSyncMillis(int syncWindowStartSeconds, int syncWindowEndSeconds) {
        int oneHour = 3600;
        int oneDay = 86400;

        int windowSize = syncWindowEndSeconds - syncWindowStartSeconds;
        if (windowSize < 0) {
            windowSize = oneDay + windowSize;
        } else if (windowSize == 0) {
            windowSize = oneHour;// Window should be at least 1 hour long
        }

        int offset = new Random(System.currentTimeMillis()).nextInt(windowSize);
        int syncSeconds;
        if (syncWindowStartSeconds <= syncWindowEndSeconds) {
            syncSeconds = syncWindowStartSeconds + offset;
        } else {
            syncSeconds = syncWindowEndSeconds + offset;
        }

        return syncSeconds % oneDay * 1000;// In milliseconds
    }
    // </editor-fold>

}
