package ai.accurat.sdk.core;

import android.Manifest;
import android.content.Context;
import android.content.pm.PackageManager;
import android.provider.Settings;
import androidx.core.content.ContextCompat;

/**
 * Utility functions to access the platform settings and app permissions.
 */
public final class SettingsUtils {

    /**
     * Private constructor to avoid instantiation of utility class.
     */
    private SettingsUtils() {
    }

    /**
     * Verifies whether location is enabled on the device.
     *
     * @param context Required to fetch the device settings
     * @return True when the location mode is set to any other value than {@link Settings.Secure#LOCATION_MODE_OFF}.
     * @throws IllegalStateException When the location setting is not found.
     */
    static boolean isLocationEnabled(Context context) throws IllegalStateException {
        try {
            int locationMode = Settings.Secure.getInt(context.getContentResolver(), Settings.Secure.LOCATION_MODE);
            return locationMode != Settings.Secure.LOCATION_MODE_OFF;
        } catch (Settings.SettingNotFoundException exception) {
            throw new IllegalStateException(exception);
        }
    }

    /**
     * Verifies whether the user has given us the required permissions.
     *
     * @param context Required to access the permissions
     * @return True when the necessary permissions have been given
     */
    static boolean hasLocationPermission(Context context) {
        return (ContextCompat.checkSelfPermission(context, Manifest.permission.ACCESS_FINE_LOCATION) == PackageManager.PERMISSION_GRANTED
                || ContextCompat.checkSelfPermission(context, Manifest.permission.ACCESS_COARSE_LOCATION) == PackageManager.PERMISSION_GRANTED);
    }
}
