package ai.accurat.sdk.data

import ai.accurat.sdk.data.enums.ConsentType
import ai.accurat.sdk.managers.RealmManager
import io.realm.Realm
import io.realm.RealmModel
import io.realm.RealmObject
import io.realm.RealmQuery
import io.realm.kotlin.where
import java.util.EnumMap

inline fun <reified T : RealmModel> Realm.firstOr(defaultIfNull: T, query: (RealmQuery<T>) -> RealmQuery<T>): T {
    val realmItem = query(this.where())
            .findFirst()
    val item = realmItem?.let {
        this.copyFromRealm(realmItem)
    } ?: defaultIfNull
    this.close()

    return item
}

fun RealmObject.save() {
    val realm = RealmManager.getAccuratInstance()
    realm.executeTransaction {
        it.copyToRealmOrUpdate(this)
    }
    realm.close()
}

inline fun <reified T : RealmModel> RealmObject.all(): MutableList<T> {
    val realm = RealmManager.getAccuratInstance()
    val realmObjects = realm.where<T>()
        .findAll()
    val objects: MutableList<T> = realm.copyFromRealm(realmObjects)
    realm.close()

    return objects
}

inline fun <reified T : RealmModel> RealmObject.count(): Long {
    val realm = RealmManager.getAccuratInstance()
    val count = realm.where<T>()
        .count()
    realm.close()

    return count
}

fun Boolean.toInt(): Int = if (this) 1 else 0

fun String.toConsentTypeMap(): Map<ConsentType, Boolean> {
    val pairs = split(",").filter {
        it.isNotBlank()
    }
    val map = EnumMap<ConsentType, Boolean>(ConsentType::class.java)
    pairs.forEach { pair ->
        val parts = pair.split(":")
        map.put(ConsentType.valueOf(parts[0]), parts[1].toBoolean())
    }

    return map
}

fun Map<ConsentType, Boolean>.stringify(): String {
    return map { entry ->
        "${entry.key}:${entry.value}"
    }.joinToString(",")
}