package ai.accurat.sdk.data.models

import ai.accurat.sdk.config.Configuration
import ai.accurat.sdk.data.RealmHelper
import ai.accurat.sdk.data.enums.ConsentType
import io.realm.RealmObject
import io.realm.annotations.PrimaryKey
import org.json.JSONObject

/**
 * @author Kenneth
 * @since 2020-11-10
 */
open class ConsentModel : RealmObject() {

    // <editor-fold desc="Fields">
    @PrimaryKey
    var id: Long = RealmHelper.NO_ID
    private var typeString: String = ConsentType.UNKNOWN.name
    var type: ConsentType
        get() = ConsentType.from(typeString)
        set(value) {
            typeString = value.name
        }
    var isSystemPermission: Boolean = false
    var maxRefuseCount: Int = if (isSystemPermission) {
        Configuration.MAX_PERMISSION_REFUSE_COUNT
    } else {
        Configuration.MAX_CONSENT_REFUSE_COUNT
    }
    var title: String = ""
    var message: String = ""
    var accept: String = ""
    var refute: String = ""
    var next: String = ""
    var brandColour: String = ""
    // </editor-fold>

    override fun toString(): String {
        return "{id: $id, type: $type, maxRefuseCount: $maxRefuseCount, isSystemPermission: $isSystemPermission, accept = $accept, refute = $refute, next = $next}"
    }

    private fun getDefaultRefuseCount(isSystemPermission: Boolean): Int = if (isSystemPermission) {
        Configuration.MAX_PERMISSION_REFUSE_COUNT
    } else {
        Configuration.MAX_CONSENT_REFUSE_COUNT
    }

    // <editor-fold desc="Objects">
    object RealmColumns {
        @JvmField val _TABLE: String = "ConsentModel"

        const val ID: String = "id"
        const val TYPE_STRING: String = "typeString"
        const val IS_SYSTEM_PERMISSION: String = "isSystemPermission"
        const val TITLE: String = "title"
        const val MESSAGE: String = "message"
        const val ACCEPT: String = "accept"
        const val REFUTE: String = "refute"
        const val NEXT: String = "next"
        const val BRAND_COLOUR: String = "brandColour"
        const val MAX_REFUSE_COUNT: String = "maxRefuseCount"
    }

    object ServerKeys {
        const val ID: String = "id"
        const val TYPE: String = "type"
        const val IS_SYSTEM_PERMISSION: String = "is_system_permission"
        const val TITLE: String = "title"
        const val MESSAGE: String = "message"
        const val ACCEPT: String = "accept"
        const val REFUTE: String = "refute"
        const val NEXT: String = "next"
        const val BRAND_COLOUR: String = "brand_color"
        const val SCREEN_REPEAT: String = "screen_repeat"
    }

    companion object {
        fun fromServerJson(json: JSONObject): ConsentModel = ConsentModel().apply {
            id = json.optLong(ServerKeys.ID, id)
            type = ConsentType.fromValue(json.optString(ServerKeys.TYPE, type.value))
            isSystemPermission = json.optBoolean(ServerKeys.IS_SYSTEM_PERMISSION, isSystemPermission)
            title = json.optString(ServerKeys.TITLE, title)
            message = json.optString(ServerKeys.MESSAGE, message)
            accept = json.optString(ServerKeys.ACCEPT, accept)
            refute = json.optString(ServerKeys.REFUTE, refute)
            next = json.optString(ServerKeys.NEXT, next)
            brandColour = json.getNullSafeString(ServerKeys.BRAND_COLOUR, brandColour)
            maxRefuseCount = json.optInt(ServerKeys.SCREEN_REPEAT, getDefaultRefuseCount(isSystemPermission))
        }
    }
    // </editor-fold>
}

fun List<ConsentModel>.find(type: ConsentType): ConsentModel? = find {
    it.type == type
}