package ai.accurat.sdk.core;

import android.annotation.SuppressLint;
import android.content.Context;
import android.net.ConnectivityManager;
import android.net.NetworkInfo;
import android.text.TextUtils;
import android.util.Base64;

import androidx.annotation.NonNull;

import com.android.volley.DefaultRetryPolicy;
import com.android.volley.RetryPolicy;

import java.nio.charset.StandardCharsets;
import java.security.InvalidKeyException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.HashMap;
import java.util.Map;
import java.util.TimeZone;

import javax.crypto.Mac;
import javax.crypto.spec.SecretKeySpec;

import ai.accurat.sdk.config.Configuration;
import ai.accurat.sdk.constants.ApiKeys;
import ai.accurat.sdk.managers.AccuratConfigurationManager;

/**
 * @author Kenneth Saey
 * @Accurat
 * @since 03-07-2018 13:24.
 */
public class AccuratApi {

    private static final String TAG = AccuratApi.class.getSimpleName();
    private static final TimeZone UTC = TimeZone.getTimeZone("UTC");
    public static RetryPolicy defaultRetryPolicy = new DefaultRetryPolicy(
            Configuration.DEFAULT_TIMEOUT_MS,
            Configuration.DEFAULT_MAX_RETRIES,
            Configuration.DEFAULT_BACKOFF_MULTIPLIER
    );

    public static boolean isNetworkAvailable(Context context) {
        ConnectivityManager connectivityManager = (ConnectivityManager) context.getSystemService(
                Context.CONNECTIVITY_SERVICE
        );

        if (connectivityManager == null) {
            return false;
        }
        NetworkInfo networkInfo = connectivityManager.getActiveNetworkInfo();

        return networkInfo != null && networkInfo.isConnected();
    }

    public static Map<String, String> getHeaders(MultiProcessStorage storage, String requestMethod, String contentType, String requestBodyHash, String path) {
        Map<String, String> headers = new HashMap<>();
        if (storage == null) {
            return headers;
        }
        final String appSecret = AccuratConfigurationManager.getPassword();
        if (TextUtils.isEmpty(appSecret)) {
            return headers;
        }

        @SuppressLint("SimpleDateFormat") final SimpleDateFormat authXDateFormat = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss.SSS'Z'");
        authXDateFormat.setTimeZone(UTC);
        final Calendar cal = Calendar.getInstance(UTC);
        final String xDate = authXDateFormat.format(cal.getTime());
        final String baseSignature = requestMethod + contentType + requestBodyHash + path + xDate;

        try {
            String stringHash = getSignatureHash(appSecret, baseSignature);
            String signature = encodeByteArray(stringHash);

            String auth = Configuration.AUTH_HEADER_NAME + " ";
            auth += AccuratConfigurationManager.getUsername();
            auth += ":" + signature;

            if (!auth.isEmpty()) {
                headers.put(ApiKeys.Header.AUTHORIZATON, auth);
                headers.put(ApiKeys.Header.X_DATE, xDate);
                if (TextUtils.isEmpty(contentType)) {
                    headers.put(ApiKeys.Header.CONTENT_TYPE, contentType);
                }
            }
        } catch (NoSuchAlgorithmException e) {
            AccuratLogger.log(AccuratLogger.ERROR, "NoSuchAlgorithmException: " + e.getMessage());
            AccuratLogger.log(AccuratLogger.WARNING, "getHeaders(" + path + ") could not get HMAC SHA256 instance");
        } catch (InvalidKeyException e) {
            AccuratLogger.log(AccuratLogger.ERROR, "InvalidKeyException: " + e.getMessage());
            AccuratLogger.log(AccuratLogger.WARNING, "getHeaders(" + path + ") could not get key spec for authorization header");
        }

        return headers;
    }

    @NonNull
    static String getSignatureHash(String appSecret, String baseSignature) throws NoSuchAlgorithmException, InvalidKeyException {
        Mac hasher = Mac.getInstance("HmacSHA256");
        hasher.init(new SecretKeySpec(appSecret.getBytes(), "HmacSHA256"));

        byte[] hash = hasher.doFinal(baseSignature.getBytes());
        return printHexBinary(hash);
    }

    /**
     * Calculates a Base64-encoded MD5 hash of the request body
     *
     * @param requestBody Request body to encode
     * @return A Base-64-encoded MD5 hash of the request body
     */
    public static String getEncodedRequestBody(@NonNull String requestBody) {
        return encodeByteArray(getRequestBodyHash(requestBody));
    }

    /**
     * Calculates an MD5 hash of the request body
     *
     * @param requestBody Request body to encode
     * @return an MD5 hash of the request body
     */
    static String getRequestBodyHash(@NonNull String requestBody) {
        MessageDigest digest;
        try {
            digest = MessageDigest.getInstance("MD5");
        } catch (NoSuchAlgorithmException e) {
            AccuratLogger.log(AccuratLogger.ERROR, "NoSuchAlgorithmException: " + e.getMessage());
            AccuratLogger.log(AccuratLogger.WARNING, TAG + ".getRequestBodyHash(): MD5 is not supported on this device");
            return "";
        }

        digest.update(requestBody == null ? "".getBytes() : requestBody.getBytes());
        byte[] hash = digest.digest();
        return printHexBinary(hash).toLowerCase();
    }

    /**
     * Prints a hex-representation of a byte array
     *
     * @param binary byte array to convert
     * @return A Hex-based textual representation of the binary input
     */
    private static String printHexBinary(byte[] binary) {
        int len = binary.length;
        StringBuilder sb = new StringBuilder(len << 1);
        for (byte b : binary) {
            sb.append(Character.forDigit((b & 0xf0) >> 4, 16));
            sb.append(Character.forDigit(b & 0x0f, 16));
        }
        return sb.toString();
    }

    /**
     * Encodes a byte array to a Base64-encoded String
     *
     * @param input Input byte array to encode
     * @return Hex-representation of the byte array
     */
    static String encodeByteArray(String input) {
        byte[] bytes = input.getBytes(StandardCharsets.UTF_8);
        byte[] encoding = Base64.encode(bytes, Base64.DEFAULT | Base64.NO_WRAP);

        return new String(encoding, StandardCharsets.UTF_8);
    }
}