package ai.accurat.sdk.core;

import android.content.Context;

import androidx.annotation.NonNull;
import androidx.work.Worker;
import androidx.work.WorkerParameters;

import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;

import ai.accurat.sdk.config.Configuration;
import ai.accurat.sdk.managers.RealmManager;

/**
 * @author Kenneth Saey
 * @Accurat
 */
public class AccuratSettingsWorker extends Worker {

    private static final String TAG = AccuratSettingsWorker.class.getSimpleName();

    public AccuratSettingsWorker(@NonNull Context context, @NonNull WorkerParameters workerParams) {
        super(context, workerParams);
    }

    @NonNull
    @Override
    public Result doWork() {
        AccuratLogger.init(getApplicationContext());
        AccuratLogger.log(AccuratLogger.WORKMANAGER, TAG + ".doWork()");
        RealmManager.init(getApplicationContext());

        // If we wouldn't use an array, we wouldn't be able to set another result value.
        final Result[] result = {Result.retry()};
        AccuratSettingsManager.init(getApplicationContext());

        // The doWork()-method requires a synchronous return.
        // However, making network requests is usually async. We could make a sync request, but any
        // exceptions will be thrown on the main thread, not on this worker thread. To handle that
        // situation, we would need to set a timeout here, but the timeout might be shorter than the
        // network response time in random cases, which would cause us to miss the response.
        // Instead we will use a CountDownLatch to block this worker thread and wait for all
        // responses of the network thread.
        final CountDownLatch countDownLatch = new CountDownLatch(1);

        AccuratSettingsManager.fetchSettings(success -> {
            result[0] = success ? Result.success() : Result.failure();
            if (success) {
                checkSdkEnabled();
            }
            countDownLatch.countDown();
        });

        try {
            // Await the network response.
            countDownLatch.await(Configuration.DEFAULT_TIMEOUT_MS, TimeUnit.MILLISECONDS);
        } catch (InterruptedException e) {
            AccuratLogger.log(AccuratLogger.JSON_ERROR, TAG + ".doWork() InterruptedException: " + e.getMessage());
        }

        AccuratLogger.log(AccuratLogger.WORKMANAGER, TAG + " - Work done, returning result[0]");
        return result[0];
    }

    @Override
    public void onStopped() {
        AccuratSettingsManager.init(getApplicationContext());
        AccuratSettingsManager.cancelNetworkRequests();
    }

    private void checkSdkEnabled() {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + "checkSdkEnabled()");
        AccuratSettings settings = AccuratSettingsManager.getSettings();
        AccuratLocationManager locationManager = AccuratLocationManager.getInstance();
        locationManager.init(getApplicationContext());
        if (settings == null || settings.isSdkEnabled()) {
            if (locationManager.isRunning()) {
                AccuratLogger.log(AccuratLogger.SDK_FLOW, "SDK is still enabled");
            } else {
                AccuratLogger.log(AccuratLogger.SDK_FLOW, "SDK is disabled, restarting it...");
                locationManager.start(getApplicationContext(), started -> {
                    AccuratLogger.log(AccuratLogger.SDK_FLOW, "SDK restarted");
                });
            }
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + "checkSdkEnabled()");

            return;
        }

        AccuratLogger.log(AccuratLogger.SDK_FLOW, "SDK is disabled");
        locationManager.stop(getApplicationContext());
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + "checkSdkEnabled()");
    }
}
