package ai.accurat.sdk.core;

import android.content.Context;

import androidx.annotation.NonNull;

import com.android.volley.AuthFailureError;
import com.android.volley.Request;
import com.android.volley.RequestQueue;
import com.android.volley.toolbox.JsonObjectRequest;
import com.android.volley.toolbox.Volley;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;

import ai.accurat.sdk.callbacks.AccuratCompletionCallback;
import ai.accurat.sdk.callbacks.AccuratResultCallback;
import ai.accurat.sdk.config.Configuration;
import ai.accurat.sdk.constants.AccuratEndpoints;
import ai.accurat.sdk.constants.AccuratLanguage;
import ai.accurat.sdk.constants.ApiKeys;
import ai.accurat.sdk.constants.HttpMethod;
import ai.accurat.sdk.constants.ServerDataKeys;
import ai.accurat.sdk.constants.StorageKeys;
import ai.accurat.sdk.data.enums.ConsentType;
import ai.accurat.sdk.data.models.ConsentManagerState;
import ai.accurat.sdk.managers.AccuratManager;

/**
 * @author Kenneth Saey
 * <p>
 * Manager to help with the storage and synchronisation of an AccuratUser.
 * @Accurat
 * @since 06-08-2018 15:31.
 */
public class AccuratUserManager {

    // <editor-fold desc="Fields">
    private static final String TAG = AccuratUserManager.class.getSimpleName();

    private static MultiProcessStorage storage;
    private static AccuratUser user;
    private static RequestQueue requestQueue;
    private static final String JSON_REQUEST_BODY_PARAM_GROUP = "group";
    private static final String JSON_REQUEST_BODY_PARAM_CAMPAIGN = "campaign";
    private static final String JSON_REQUEST_BODY_PARAM_TOUCHPOINT = "touchpoint";
    private static final String JSON_RESPONSE_ROOT = "data";
    private static final String JSON_RESPONSE_BODY_INSERTED_AT = "inserted_at";
    // </editor-fold>

    // <editor-fold desc="Initialisation">
    public static void init(@NonNull Context context) {
        if (!isInitialized()) {
            AccuratLogger.init(context);
            AccuratLogger.log(AccuratLogger.SDK_FLOW, "Initialising " + TAG);
            storage = MultiProcessStorage.getStorage(context, StorageKeys.ACCURAT_MULTI_PROCESS_STORAGE);
            requestQueue = Volley.newRequestQueue(context);
        } else if (requestQueue == null) {
            requestQueue = Volley.newRequestQueue(context);
        }
    }

    private static boolean isInitialized() {
        return storage != null;
    }

    private static void checkInitialized() {
        if (!isInitialized()) {
            throw new IllegalStateException("AccuratUserManager has not yet been initialised.");
        }
    }
    // </editor-fold>

    // <editor-fold desc="Storage">
    private static void load() {
        checkInitialized();

        String jsonString = storage.getString(StorageKeys.ACCURAT_USER, "");
        if (jsonString == null || jsonString.isEmpty()) {
            user = new AccuratUser();
            return;
        }

        try {
            user = AccuratUser.fromJson(jsonString);
        } catch (JSONException e) {
            AccuratLogger.log(AccuratLogger.JSON_ERROR, TAG + ".load(): " + e.getMessage());
            e.printStackTrace();
        }
    }

    private static void store() {
        checkInitialized();

        if (user == null) {
            storage.remove(StorageKeys.ACCURAT_USER)
                    .commit();
        } else {
            storage.setValue(StorageKeys.ACCURAT_USER, user.toJson().toString())
                    .commit();
        }
    }
    // </editor-fold>

    // <editor-fold desc="Public interface">
    public static AccuratUser getUser() {
        checkInitialized();
        load();

        return user;
    }

    public static String getUserLanguage() {
        AccuratUser user = getUser();
        if (user == null || user.language == null) {
            return Locale.getDefault().toString();
        }

        return user.language.getCode();
    }

    public static void setLanguage(AccuratLanguage language) {
        load();

        if (user == null) {
            user = new AccuratUser();
        } else if (user.language == language) {
            // Nothing changed
            return;
        }

        user.language = language;
        store();
    }

    /**
     * Add interaction for consumer, based on given campaign and touchpoint.
     * If campaign and/or touchpoint does not exist, they will be created.
     *
     * @param group      The group which contains the active campaign
     * @param campaign   The campaign representing the presented advertising messages
     * @param touchpoint The way the consumer interacted with the business
     * @param callback   Indicates whether the interaction has been processed correctly by the server.
     */
    public static void interact(String group, String campaign, String touchpoint, AccuratCompletionCallback callback) {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".interact()");
        checkInitialized();

        // Only interact if the user has consented
        try {
//            if (ConsentManager.getConsentState(ConsentType.TRACKING) != ConsentState.ACCEPTED) {
            if (AccuratManager.INSTANCE.check(ConsentType.GDPR)) {
                return;
            }
        } catch (IllegalStateException e) {
            // ConsentManager was not initialised
            return;
        }

        HashMap<String, Object> urlParameters = getUrlParameters();
        JSONObject requestBody = new JSONObject();
        try {
            requestBody.put(JSON_REQUEST_BODY_PARAM_GROUP, group);
            requestBody.put(JSON_REQUEST_BODY_PARAM_CAMPAIGN, campaign);
            requestBody.put(JSON_REQUEST_BODY_PARAM_TOUCHPOINT, touchpoint);
        } catch (JSONException e) {
            AccuratLogger.log(AccuratLogger.JSON_ERROR, TAG + ".interact(): " + e.getMessage());
            e.printStackTrace();
        }
        JsonObjectRequest interactRequest = new JsonObjectRequest(
                Request.Method.POST,
                AccuratEndpoints.POST_INTERACT.getUrl(urlParameters),
                requestBody,
                response -> {
                    AccuratLogger.logNetworkResponse(HttpMethod.POST, Configuration.ENDPOINT_POST_INTERACT, response, false);
                    boolean result = false;
                    if (response != null) {
                        try {
                            result = response.has(JSON_RESPONSE_ROOT) && response.getJSONObject(JSON_RESPONSE_ROOT).has(JSON_RESPONSE_BODY_INSERTED_AT);
                        } catch (JSONException e) {
                            AccuratLogger.log(AccuratLogger.JSON_ERROR, "Could not parse response: " + e.getMessage());
                        }
                    }
                    if (callback != null) {
                        callback.onCompleted(result);
                    }
                },
                error -> {
                    AccuratLogger.logNetworkError(HttpMethod.POST, Configuration.ENDPOINT_POST_INTERACT, error);
                    if (callback != null) {
                        callback.onCompleted(false);
                    }
                }
        ) {
            @Override
            public Map<String, String> getHeaders() throws AuthFailureError {
                return AccuratApi.getHeaders(storage, "POST", getBodyContentType(),
                        AccuratApi.getEncodedRequestBody(requestBody.toString()), AccuratEndpoints.POST_INTERACT.getPath(urlParameters));
            }
        };
        interactRequest.setTag(TAG)
                .setRetryPolicy(AccuratApi.defaultRetryPolicy)
                .setShouldCache(false);
        if (requestQueue == null) {
            AccuratLogger.log(AccuratLogger.ERROR, "Failed to make API call, requestQueue is NULL");
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".interact()");

            return;
        }

        AccuratLogger.logNetworkRequest(HttpMethod.POST, Configuration.ENDPOINT_POST_INTERACT, requestBody, false);
        requestQueue.add(interactRequest);
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".interact()");
    }

    /**
     * Fetch segments the current consumer belongs to.
     * If the consumer does not exist, an empty list is returned.
     *
     * @param callback Contains the segments the current consumer belongs to.
     */
    public static void getSegments(AccuratResultCallback<String[]> callback) {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".getSegments()");
        checkInitialized();

        HashMap<String, Object> urlParameters = getUrlParameters();
        JsonObjectRequest segmentsRequest = new JsonObjectRequest(
                Request.Method.GET,
                AccuratEndpoints.GET_SEGMENTS.getUrl(urlParameters),
                null,
                response -> {
                    AccuratLogger.logNetworkResponse(HttpMethod.GET, Configuration.ENDPOINT_GET_SEGMENTS, response, false);
                    if (response == null || !response.has(JSON_RESPONSE_ROOT)) {
                        callback.onCompleted(new String[0]);
                        return;
                    }

                    try {
                        JSONArray jsonArray = response.getJSONArray(JSON_RESPONSE_ROOT);
                        if (jsonArray == null || jsonArray.length() == 0) {
                            callback.onCompleted(new String[0]);
                            return;
                        }

                        int numSegments = jsonArray.length();
                        final String[] segments = new String[numSegments];
                        for (int i = 0; i < numSegments; i++) {
                            segments[i] = jsonArray.getString(i);
                        }

                        callback.onCompleted(segments);
                    } catch (JSONException e) {
                        AccuratLogger.log(AccuratLogger.JSON_ERROR, "Could not parse segments: " + e.getMessage());
                    }
                },
                error -> {
                    AccuratLogger.logNetworkError(HttpMethod.GET, Configuration.ENDPOINT_GET_SEGMENTS, error);
                    if (callback != null) {
                        callback.onError(error);
                    }
                }
        ) {
            @Override
            public Map<String, String> getHeaders() throws AuthFailureError {
                return AccuratApi.getHeaders(storage, "GET", "",
                        AccuratApi.getEncodedRequestBody(""), AccuratEndpoints.GET_SEGMENTS.getPath(urlParameters));
            }
        };
        segmentsRequest.setTag(TAG)
                .setRetryPolicy(AccuratApi.defaultRetryPolicy)
                .setShouldCache(false);
        if (requestQueue == null) {
            AccuratLogger.log(AccuratLogger.ERROR, "Failed to make API call, requestQueue is NULL");
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".getSegments()");

            return;
        }

        AccuratLogger.logNetworkRequest(HttpMethod.GET, Configuration.ENDPOINT_GET_SEGMENTS);
        requestQueue.add(segmentsRequest);
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".getSegments()");
    }

    private static void respondEmptyToGetMeta(MetaCallback callback) {
        respondToGetMeta(callback, new ArrayList<>());
    }

    private static void respondToGetMeta(MetaCallback callback, ArrayList<Meta> metas) {
        if (callback == null) {
            return;
        }

        callback.onResponse(metas);
    }

    public static void getMeta(MetaCallback callback) {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".getMeta()");
        checkInitialized();

        HashMap<String, Object> urlParameters = getUrlParameters();
        JsonObjectRequest metaRequest = new JsonObjectRequest(
                Request.Method.GET,
                AccuratEndpoints.GET_META.getUrl(urlParameters),
                null,
                response -> {
                    AccuratLogger.logNetworkResponse(HttpMethod.GET, Configuration.ENDPOINT_GET_META, response, false);
                    if (response == null || !response.has(JSON_RESPONSE_ROOT)) {
                        respondEmptyToGetMeta(callback);

                        return;
                    }

                    try {
                        JSONArray jsonArray = response.getJSONArray(JSON_RESPONSE_ROOT);
                        if (jsonArray == null || jsonArray.length() == 0) {
                            respondEmptyToGetMeta(callback);

                            return;
                        }

                        respondToGetMeta(callback, JSON.loadObjectList(jsonArray, new ArrayList<>(), Meta.class));
                    } catch (JSONException e) {
                        AccuratLogger.log(AccuratLogger.JSON_ERROR, "Could not parse metas: " + e.getMessage());
                    }
                },
                error -> {
                    AccuratLogger.logNetworkError(HttpMethod.GET, Configuration.ENDPOINT_GET_META, error);
                    respondEmptyToGetMeta(callback);
                }
        ) {
            @Override
            public Map<String, String> getHeaders() throws AuthFailureError {
                return AccuratApi.getHeaders(storage, "GET", "",
                        AccuratApi.getEncodedRequestBody(""), AccuratEndpoints.GET_META.getPath(urlParameters));
            }
        };
        metaRequest.setTag(TAG)
                .setRetryPolicy(AccuratApi.defaultRetryPolicy)
                .setShouldCache(false);

        if (requestQueue == null) {
            AccuratLogger.log(AccuratLogger.ERROR, "Failed to make API call, requestQueue is NULL");
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".getMeta()");

            return;
        }

        AccuratLogger.logNetworkRequest(HttpMethod.GET, Configuration.ENDPOINT_GET_META);
        requestQueue.add(metaRequest);
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".getMeta()");
    }

    public static void setMeta(Meta... metas) {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".setMeta()");
        checkInitialized();

        HashMap<String, Object> urlParameters = getUrlParameters();
        JSONObject requestBody = new JSONObject();
        JSONArray data = new JSONArray();
        try {
            for (Meta meta : metas) {
                data.put(meta.toJson());
            }
            requestBody.put(ServerDataKeys.Meta.DATA, data);
        } catch (JSONException e) {
            AccuratLogger.log(AccuratLogger.JSON_ERROR, TAG + ".setMeta(): " + e.getMessage());
            e.printStackTrace();
        }
        JsonObjectRequest metaRequest = new JsonObjectRequest(
                Request.Method.POST,
                AccuratEndpoints.GET_META.getUrl(urlParameters),
                requestBody,
                response -> AccuratLogger.logNetworkResponse(HttpMethod.POST, Configuration.ENDPOINT_GET_META, response, false),
                error -> AccuratLogger.logNetworkError(HttpMethod.POST, Configuration.ENDPOINT_GET_META, error)
        ) {
            @Override
            public Map<String, String> getHeaders() throws AuthFailureError {
                return AccuratApi.getHeaders(storage, "POST", getBodyContentType(),
                        AccuratApi.getEncodedRequestBody(requestBody.toString()), AccuratEndpoints.GET_META.getPath(urlParameters));
            }
        };
        metaRequest.setTag(TAG)
                .setRetryPolicy(AccuratApi.defaultRetryPolicy)
                .setShouldCache(false);

        if (requestQueue == null) {
            AccuratLogger.log(AccuratLogger.ERROR, "Failed to make API call, requestQueue is NULL");
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".setMeta()");

            return;
        }

        AccuratLogger.logNetworkRequest(HttpMethod.POST, Configuration.ENDPOINT_GET_META, requestBody, false);
        requestQueue.add(metaRequest);
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".setMeta()");
    }
    // </editor-fold>

    // <editor-fold desc="Network">
    public static void upload() {// TODO - Check usages
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".upload()");
        load();
        if (user == null) {
            AccuratLogger.log(AccuratLogger.SDK_FLOW, "User is null, nothing to upload");
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".upload()");

            return;
        }

        HashMap<String, Object> urlParameters = getUrlParameters();
        if (urlParameters == null) {
            AccuratLogger.log(AccuratLogger.SDK_FLOW, "No ad ID available yet, nothing to upload");
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".upload()");

            return;
        }

        JSONObject requestBody = getRequestBody();
        if (requestBody == null) {
            AccuratLogger.log(AccuratLogger.WARNING, "Couldn't create request body, nothing to upload");
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".upload()");

            return;
        }

        JsonObjectRequest consumerRequest = new JsonObjectRequest(
                Request.Method.POST,
                AccuratEndpoints.POST_CONSUMERS.getUrl(urlParameters),
                requestBody,
                response -> {
                    AccuratLogger.logNetworkResponse(HttpMethod.POST, Configuration.ENDPOINT_POST_CONSUMERS, response, false);
                },// Don't handle responses
                error -> {
                    AccuratLogger.logNetworkError(HttpMethod.POST, Configuration.ENDPOINT_POST_CONSUMERS, error);
                }// Don't handle errors
        ) {
            @Override
            public Map<String, String> getHeaders() {
                return AccuratApi.getHeaders(storage, "POST", getBodyContentType(),
                        AccuratApi.getEncodedRequestBody(requestBody.toString()), AccuratEndpoints.POST_CONSUMERS.getPath(urlParameters));
            }
        };

        consumerRequest.setTag(TAG)
                .setRetryPolicy(AccuratApi.defaultRetryPolicy)
                .setShouldCache(false);

        if (requestQueue == null) {
            AccuratLogger.log(AccuratLogger.ERROR, "Failed to make API call, requestQueue is NULL");
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".upload()");

            return;
        }
        AccuratLogger.logNetworkRequest(HttpMethod.POST, Configuration.ENDPOINT_POST_CONSUMERS, requestBody, false);
        requestQueue.add(consumerRequest);
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".upload()");
    }

    private static HashMap<String, Object> getUrlParameters() {
//        String adId = AdvertisingManager.getAdId();
        String adId = ConsentManagerState.Companion.getAdId();
        if (adId == null) {
            return null;
        }

        HashMap<String, Object> urlParameters = new HashMap<>();
        urlParameters.put(ApiKeys.Url.AD_ID, adId);

        return urlParameters;
    }

    private static JSONObject getRequestBody() {
        JSONObject requestParameters = new JSONObject();
        try {
            requestParameters.put("lang", user.language.getCode());

            return requestParameters;
        } catch (JSONException e) {
            return null;
        }
    }
    // </editor-fold>
}
