package ai.accurat.sdk.core;

import android.content.Context;

import androidx.annotation.NonNull;
import androidx.work.OneTimeWorkRequest;
import androidx.work.WorkManager;

import com.android.volley.AuthFailureError;
import com.android.volley.Request;
import com.android.volley.RequestQueue;
import com.android.volley.toolbox.JsonObjectRequest;
import com.android.volley.toolbox.Volley;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;

import ai.accurat.sdk.callbacks.AccuratCompletionCallback;
import ai.accurat.sdk.config.Configuration;
import ai.accurat.sdk.constants.AccuratEndpoints;
import ai.accurat.sdk.constants.ApiKeys;
import ai.accurat.sdk.constants.HttpMethod;
import ai.accurat.sdk.constants.ServerDataKeys;
import ai.accurat.sdk.constants.StorageKeys;
import ai.accurat.sdk.data.enums.ConsentType;
import ai.accurat.sdk.data.models.ConsentManagerState;
import ai.accurat.sdk.managers.RealmManager;
import io.realm.Realm;

public class CampaignManager {

    // <editor-fold desc="Properties">
    private static final String TAG = CampaignManager.class.getSimpleName();
    private static final String DEFAULT_CAMPAIGN_GROUP = "_default_";
    private static final String DEFAULT_CAMPAIGN = "_default_";
    private static final String DEFAULT_TOUCH_POINT_PREFIX = "_notification_";
    private static final String DEFAULT_TOUCH_POINT_SUFFIX = "_";

    private static MultiProcessStorage storage;
    private static SimpleRealmStorage<CampaignInteraction> campaignStorage;
    private static boolean isUploading;

    private static RequestQueue requestQueue;
    // </editor-fold>

    // <editor-fold desc="Initialisation">
    public static void init(@NonNull Context context) {
        if (isInitialized()) {
            return;
        }

        AccuratLogger.init(context);
        AccuratLogger.log(AccuratLogger.SDK_FLOW, "Initialising " + TAG);
        ConsentManager.init(context);
        RealmManager.init(context);
        storage = MultiProcessStorage.getStorage(context, StorageKeys.ACCURAT_MULTI_PROCESS_STORAGE);
        campaignStorage = new SimpleRealmStorage<>(context);
        requestQueue = Volley.newRequestQueue(context);
    }

    private static boolean isInitialized() {
        return requestQueue != null;
    }

    private static void checkInitialized() {
        if (isInitialized()) {
            return;
        }

        throw new IllegalStateException(TAG + " has not yet been initialised.");
    }
    // </editor-fold>

    // <editor-fold desc="Public interface">
    public static boolean add(CampaignInteraction interaction, AccuratCompletionCallback callback) {
        checkInitialized();
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".add()");
        AccuratLogger.log(AccuratLogger.NONE, "Interaction = " + interaction.toString());
        if (!ConsentManagerState.Companion.isConsentAccepted(ConsentType.GDPR)) {
            AccuratLogger.log(AccuratLogger.SDK_FLOW, "Tracking not allowed, won't store interaction");
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".add()");

            return false;
        }

        AccuratLogger.log(AccuratLogger.SDK_FLOW, "Adding interaction");
        campaignStorage.store(interaction);
        upload(callback);
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".add()");

        return true;
    }

    public static boolean addOfflineNotificationInteraction(long notificationId) {
        return add(new CampaignInteraction(
                DEFAULT_CAMPAIGN_GROUP,
                DEFAULT_CAMPAIGN,
                DEFAULT_TOUCH_POINT_PREFIX + notificationId + DEFAULT_TOUCH_POINT_SUFFIX
        ), null);
    }

    public static void upload(AccuratCompletionCallback callback) {
        upload(false, callback);
    }

    public static void upload(boolean calledFromWorker, AccuratCompletionCallback callback) {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".upload()");
        if (!shouldUpload()) {
            AccuratLogger.log(AccuratLogger.SDK_FLOW, TAG + " No upload allowed at this time");
            succeed(callback);
            if (!calledFromWorker) {
                scheduleCampaignUploadWorker();
            }

            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".upload()");
            return;
        }

        uploadBatch(callback);
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".upload()");
    }

    private static void scheduleCampaignUploadWorker() {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".scheduleCampaignUploadWorker()");
        WorkManager workManager = WorkManager.getInstance();
        if (workManager == null) {
            AccuratLogger.log(AccuratLogger.ERROR, "WorkManager.getInstance() is NULL");
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".scheduleCampaignUploadWorker()");

            return;
        }

        // Cancel any previously running work
        workManager.cancelAllWorkByTag(Constants.UPLOAD_CAMPAIGN_WORK_TAG);

        // Build the request
        OneTimeWorkRequest.Builder builder = new OneTimeWorkRequest.Builder(CampaignUploadWorker.class)
                .setInitialDelay(Configuration.DEFAULT_CAMPAIGN_UPLOAD_INTERVAL_MILLIS, TimeUnit.MILLISECONDS)
                .addTag(Constants.UPLOAD_CAMPAIGN_WORK_TAG);

        // Schedule the work
        workManager.enqueue(builder.build());
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".scheduleCampaignUploadWorker()");
    }
    // </editor-fold>

    // <editor-fold desc="Storage">
    private static long getLastUpload() {
        checkInitialized();

        try (Realm realm = RealmManager.getAccuratInstance()) {
            Settings settings = realm.where(Settings.class).findFirst();

            if (settings == null) {
                settings = Settings.defaultSettings();
            }

            return settings.campaignLastUploaded;
        }
    }

    private static void setLastUpload(long timestamp) {
        checkInitialized();

        try (Realm realm = RealmManager.getAccuratInstance()) {
            realm.executeTransaction(
                    transactionRealm -> {
                        Settings settings = realm.where(Settings.class).findFirst();

                        if (settings == null) {
                            settings = Settings.defaultSettings();
                        }

                        settings.campaignLastUploaded = timestamp;

                        realm.copyToRealm(settings);
                    });
        } finally {
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".store()");
        }
    }

    private static List<CampaignInteraction> getCurrentInteractions() {
        return campaignStorage.getAll(CampaignInteraction.class);
    }

    private static void deleteInteractions(List<CampaignInteraction> interactions) {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".deleteInteractions()");
        AccuratLogger.log(AccuratLogger.NONE, "Deleting " + interactions.size() + " interactions");
        ArrayList<Long> timestamps = new ArrayList<>();
        for (CampaignInteraction interaction : interactions) {
            timestamps.add(interaction.timestamp);
        }

        try (Realm realm = RealmManager.getAccuratInstance()) {
            realm.executeTransaction(
                    transactionRealm -> transactionRealm.where(CampaignInteraction.class)
                            .in(CampaignInteraction.RealmKeys.TIMESTAMP, timestamps.toArray(new Long[0]))
                            .findAll()
                            .deleteAllFromRealm()
            );
        } finally {
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".deleteInteractions()");
        }
    }
    // </editor-fold>

    // <editor-fold desc="Helpers">
    private static boolean shouldUpload() {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".shouldUpload()");

        if (isUploading) {
            AccuratLogger.log(AccuratLogger.SDK_FLOW, TAG + " -- No, upload is already in progress!");
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".shouldUpload()");
            return false;
        }

        checkInitialized();

        if (!ConsentManagerState.Companion.isConsentAccepted(ConsentType.GDPR)) {
            AccuratLogger.log(AccuratLogger.SDK_FLOW, TAG + " -- No, no consent given");

            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".shouldUpload()");
            return false;
        }

        Realm realm = RealmManager.getAccuratInstance();
        long interactionCount = realm.where(CampaignInteraction.class).count();
        realm.close();
        if (interactionCount <= 0) {
            AccuratLogger.log(AccuratLogger.SDK_FLOW, TAG + " -- No, no interactions in storage");

            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".shouldUpload()");
            return false;
        }

        long getLastUpload = getLastUpload();
        long uploadedAgoMillis = System.currentTimeMillis() - getLastUpload;
        AccuratLogger.log(AccuratLogger.SDK_FLOW, TAG + " Last uploaded " + (uploadedAgoMillis / 60000) + " minutes ago [" + uploadedAgoMillis + "ms]");
        if (uploadedAgoMillis < Configuration.DEFAULT_CAMPAIGN_UPLOAD_INTERVAL_MILLIS) {
            AccuratLogger.log(AccuratLogger.SDK_FLOW, TAG + " -- No, not long ago enough");

            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".shouldUpload()");
            return false;
        }

        AccuratLogger.log(AccuratLogger.SDK_FLOW, TAG + " -- Yes");
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".shouldUpload()");
        return true;
    }

    private static void succeed(AccuratCompletionCallback callback) {
        if (callback == null) {
            return;
        }

        callback.onCompleted(true);
        isUploading = false;
    }

    private static void fail(AccuratCompletionCallback callback) {
        if (callback == null) {
            return;
        }

        callback.onCompleted(false);
        isUploading = false;
    }
    // </editor-fold>

    // <editor-fold desc="Network">
    private static void uploadBatch(AccuratCompletionCallback callback) {
        isUploading = true;

        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".uploadBatch()");
        List<CampaignInteraction> interactions = getCurrentInteractions();
        JSONObject requestBody = getUploadBody(interactions);
        if (requestBody == null) {
            fail(callback);

            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".uploadBatch()");
            return;
        }

        HashMap<String, Object> urlParameters = getUrlParameters();
        JsonObjectRequest batchInteractRequest = new JsonObjectRequest(
                Request.Method.POST,
                AccuratEndpoints.POST_BATCH_INTERACT.getUrl(urlParameters),
                requestBody,
                response -> {
                    AccuratLogger.logNetworkResponse(HttpMethod.POST, Configuration.ENDPOINT_POST_BATCH_INTERACT, response, false);
                    if (response == null) {
                        AccuratLogger.log(AccuratLogger.WARNING, "uploadBatch: response is null");
                        fail(callback);

                        return;
                    }

                    if (!response.has(ServerDataKeys.BatchInteract.DATA) || response.isNull(ServerDataKeys.BatchInteract.DATA)) {
                        AccuratLogger.log(AccuratLogger.ERROR, "Could not process batch interact, response is NULL or missing 'data' element");
                        fail(callback);

                        return;
                    }

                    int data = response.optInt(ServerDataKeys.BatchInteract.DATA, 0);

                    if (data == 1) {
                        AccuratLogger.log(AccuratLogger.SDK_FLOW, "Batch interact succeeded");
                        deleteInteractions(interactions);
                        setLastUpload(System.currentTimeMillis());
                        succeed(callback);
                    } else {
                        AccuratLogger.log(AccuratLogger.WARNING, "Batch interact failed, data is " + data);
                        fail(callback);
                    }
                },
                error -> {
                    AccuratLogger.logNetworkError(HttpMethod.POST, Configuration.ENDPOINT_POST_BATCH_INTERACT, error);
                    fail(callback);
                }
        ) {
            @Override
            public Map<String, String> getHeaders() throws AuthFailureError {
                return AccuratApi.getHeaders(storage, "POST", getBodyContentType(),
                        AccuratApi.getEncodedRequestBody(requestBody.toString()), AccuratEndpoints.POST_BATCH_INTERACT.getPath(urlParameters));
            }
        };
        batchInteractRequest.setTag(TAG)
                .setRetryPolicy(AccuratApi.defaultRetryPolicy)
                .setShouldCache(false);

        if (requestQueue == null) {
            AccuratLogger.log(AccuratLogger.ERROR, "Failed to make API call, requestQueue is NULL");
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".uploadBatch()");
            fail(callback);

            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".uploadBatch()");
            return;
        }

        AccuratLogger.logNetworkRequest(HttpMethod.POST, Configuration.ENDPOINT_POST_BATCH_INTERACT, requestBody, false);
        requestQueue.add(batchInteractRequest);
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".uploadBatch()");
    }

    private static HashMap<String, Object> getUrlParameters() {
        String adId = AdvertisingManager.getAdId();
        if (adId == null) {
            return null;
        }

        HashMap<String, Object> urlParameters = new HashMap<>();
        urlParameters.put(ApiKeys.Url.AD_ID, adId);

        return urlParameters;
    }

    private static JSONObject getUploadBody(List<CampaignInteraction> interactions) {
        JSONObject json = new JSONObject();

        try {
            JSONArray data = JSON.toServerJsonObjectArray(interactions);
            json.put(ApiKeys.BatchInteract.INTERACTIONS, data);
            AccuratLogger.log(AccuratLogger.SDK_FLOW, "Campaign data: " + data.toString());

        } catch (JSONException e) {
            AccuratLogger.log(AccuratLogger.JSON_ERROR, e.getMessage());

            return null;
        }

        return json;
    }
    // </editor-fold>
}
