package ai.accurat.sdk.core;

import android.content.Context;

import org.json.JSONException;
import org.json.JSONObject;

import ai.accurat.sdk.config.Configuration;
import ai.accurat.sdk.constants.ConsentState;
import ai.accurat.sdk.constants.ConsentType;
import ai.accurat.sdk.constants.ServerDataKeys;
import ai.accurat.sdk.managers.VersionManager;

/**
 * @author Kenneth Saey
 * @Accurat
 * @since 22-06-2018 09:11.
 */
public class Consent implements JsonObjectType {

    private static final String JSON_KEY_TYPE = "Consent.type";
    private static final String JSON_KEY_STATE = "Consent.state";
    private static final String JSON_KEY_TITLE = "Consent.title";
    private static final String JSON_KEY_DESCRIPTION = "Consent.description";
    private static final String JSON_KEY_BUTTON_ACCEPT = "Consent.button_accept";
    private static final String JSON_KEY_BUTTON_REFUSE = "Consent.button_refuse";
    private static final String JSON_KEY_BRAND_COLOR = "Consent.brand_color";
    private static final String JSON_KEY_REFUSE_COUNT = "Consent.refuse_count";
    private static final String JSON_KEY_MAX_CONSENT_REFUSE_COUNT = "Consent.max_consent_refuse_count";

    private static final String JSON_KEY_SYSTEM_PERMISSION_STATE = "Consent.system_permission_state";
    private static final String JSON_KEY_SYSTEM_PERMISSION_BACKGROUND_STATE = "Consent.system_permission_background_state";
    private static final String JSON_KEY_SYSTEM_PERMISSION_TITLE = "Consent.system_permission_title";
    private static final String JSON_KEY_SYSTEM_PERMISSION_DESCRIPTION = "Consent.system_permission_description";
    private static final String JSON_KEY_SYSTEM_PERMISSION_BUTTON_NEXT = "Consent.system_permission_button_next";
    private static final String JSON_KEY_SYSTEM_PERMISSION_REFUSE_COUNT = "Consent.system_permission_refuse_count";
    private static final String JSON_KEY_MAX_PERMISSION_REFUSE_COUNT = "Consent.max_permission_refuse_count";

    private static final String JSON_KEY_SERVER_ID = "Consent.server_id";
    private static final String JSON_KEY_SERVER_TYPE = "Consent.server_type";
    private static final String JSON_KEY_SERVER_PERMISSION_TYPE = "Consent.server_permission_type";
    private static final String JSON_KEY_SERVER_PERMISSION_ID = "Consent.server_permission_id";
    private static final String JSON_EMPTY_STRING_VALUE_AFTER_PARSING = "null";
    private static final String TAG = Consent.class.getSimpleName();

    private ConsentType type;
    private ConsentState state = ConsentState.UNKNOWN;
    private String title;
    private String description;
    private String buttonAccept;
    private String buttonRefuse;
    private String brandColor;
    private int refuseCount;
    private int maxConsentRefuseCount = Configuration.MAX_CONSENT_REFUSE_COUNT;

    private ConsentState systemPermissionState = ConsentState.UNKNOWN;
    private ConsentState systemPermissionBackgroundState = ConsentState.UNKNOWN;
    private String systemPermissionTitle;
    private String systemPermissionDescription;
    private String systemPermissionButtonNext;
    private int systemPermissionRefuseCount;
    private int maxPermissionRefuseCount = Configuration.MAX_PERMISSION_REFUSE_COUNT;

    private int serverId;
    private String serverType;
    private String serverPermissionType;
    private int serverPermissionId;

    // <editor-fold desc="JSON Handling">
    public static Consent fromJson(String jsonString) throws JSONException {
        JSONObject json = new JSONObject(jsonString);
        Consent consent = new Consent();
        try {
            consent.type = json.has(JSON_KEY_TYPE) ? ConsentType.valueOf(json.getString(JSON_KEY_TYPE)) : consent.type;
        } catch (IllegalArgumentException ignored) {
        }
        try {
            consent.state = json.has(JSON_KEY_STATE) ? ConsentState.valueOf(json.getString(JSON_KEY_STATE)) : consent.state;
        } catch (IllegalArgumentException ignored) {
        }
        consent.title = json.has(JSON_KEY_TITLE) ? json.getString(JSON_KEY_TITLE) : consent.title;
        consent.description = json.has(JSON_KEY_DESCRIPTION) ? json.getString(JSON_KEY_DESCRIPTION) : consent.description;
        consent.buttonAccept = json.has(JSON_KEY_BUTTON_ACCEPT) ? json.getString(JSON_KEY_BUTTON_ACCEPT) : consent.buttonAccept;
        consent.buttonRefuse = json.has(JSON_KEY_BUTTON_REFUSE) ? json.getString(JSON_KEY_BUTTON_REFUSE) : consent.buttonRefuse;
        consent.brandColor = json.has(JSON_KEY_BRAND_COLOR) ? json.getString(JSON_KEY_BRAND_COLOR) : consent.brandColor;
        if (JSON_EMPTY_STRING_VALUE_AFTER_PARSING.equals(consent.brandColor)) {
            consent.brandColor = null;
        }
        consent.refuseCount = json.has(JSON_KEY_REFUSE_COUNT) ? json.getInt(JSON_KEY_REFUSE_COUNT) : consent.refuseCount;
        consent.maxConsentRefuseCount = json.has(JSON_KEY_MAX_CONSENT_REFUSE_COUNT) ? json.getInt(JSON_KEY_MAX_CONSENT_REFUSE_COUNT) : consent.maxConsentRefuseCount;

        try {
            consent.systemPermissionState = json.has(JSON_KEY_SYSTEM_PERMISSION_STATE) ? ConsentState.valueOf(json.getString(JSON_KEY_SYSTEM_PERMISSION_STATE)) : consent.systemPermissionState;
        } catch (IllegalArgumentException ignored) {
        }
        try {
            consent.systemPermissionBackgroundState = json.has(JSON_KEY_SYSTEM_PERMISSION_BACKGROUND_STATE) ? ConsentState.valueOf(json.getString(JSON_KEY_SYSTEM_PERMISSION_BACKGROUND_STATE)) : consent.systemPermissionBackgroundState;
        } catch (IllegalArgumentException ignored) {
        }
        consent.systemPermissionTitle = json.has(JSON_KEY_SYSTEM_PERMISSION_TITLE) ? json.getString(JSON_KEY_SYSTEM_PERMISSION_TITLE) : consent.systemPermissionTitle;
        consent.systemPermissionDescription = json.has(JSON_KEY_SYSTEM_PERMISSION_DESCRIPTION) ? json.getString(JSON_KEY_SYSTEM_PERMISSION_DESCRIPTION) : consent.systemPermissionDescription;
        consent.systemPermissionButtonNext = json.has(JSON_KEY_SYSTEM_PERMISSION_BUTTON_NEXT) ? json.getString(JSON_KEY_SYSTEM_PERMISSION_BUTTON_NEXT) : consent.systemPermissionButtonNext;
        consent.systemPermissionRefuseCount = json.has(JSON_KEY_SYSTEM_PERMISSION_REFUSE_COUNT) ? json.getInt(JSON_KEY_SYSTEM_PERMISSION_REFUSE_COUNT) : consent.systemPermissionRefuseCount;
        consent.maxPermissionRefuseCount = json.has(JSON_KEY_MAX_PERMISSION_REFUSE_COUNT) ? json.getInt(JSON_KEY_MAX_PERMISSION_REFUSE_COUNT) : consent.maxPermissionRefuseCount;

        consent.serverId = json.has(JSON_KEY_SERVER_ID) ? json.getInt(JSON_KEY_SERVER_ID) : consent.serverId;
        consent.serverType = json.has(JSON_KEY_SERVER_TYPE) ? json.getString(JSON_KEY_SERVER_TYPE) : consent.serverType;
        consent.serverPermissionType = json.has(JSON_KEY_SERVER_PERMISSION_TYPE) ? json.getString(JSON_KEY_SERVER_PERMISSION_TYPE) : consent.serverPermissionType;
        consent.serverPermissionId = json.has(JSON_KEY_SERVER_PERMISSION_ID) ? json.getInt(JSON_KEY_SERVER_PERMISSION_ID) : consent.serverPermissionId;

        return consent;
    }

    @Override
    public JSONObject getJson() {
        JSONObject json = new JSONObject();
        try {
            json.put(JSON_KEY_TYPE, type.name());
            json.put(JSON_KEY_STATE, state.name());
            json.put(JSON_KEY_TITLE, title);
            json.put(JSON_KEY_DESCRIPTION, description);
            json.put(JSON_KEY_BUTTON_ACCEPT, buttonAccept);
            json.put(JSON_KEY_BUTTON_REFUSE, buttonRefuse);
            json.put(JSON_KEY_BRAND_COLOR, brandColor);
            json.put(JSON_KEY_REFUSE_COUNT, refuseCount);
            json.put(JSON_KEY_MAX_CONSENT_REFUSE_COUNT, maxConsentRefuseCount);

            json.put(JSON_KEY_SYSTEM_PERMISSION_STATE, systemPermissionState.name());
            json.put(JSON_KEY_SYSTEM_PERMISSION_BACKGROUND_STATE, systemPermissionBackgroundState.name());
            json.put(JSON_KEY_SYSTEM_PERMISSION_TITLE, systemPermissionTitle);
            json.put(JSON_KEY_SYSTEM_PERMISSION_DESCRIPTION, systemPermissionDescription);
            json.put(JSON_KEY_SYSTEM_PERMISSION_BUTTON_NEXT, systemPermissionButtonNext);
            json.put(JSON_KEY_SYSTEM_PERMISSION_REFUSE_COUNT, systemPermissionRefuseCount);
            json.put(JSON_KEY_MAX_PERMISSION_REFUSE_COUNT, maxPermissionRefuseCount);

            json.put(JSON_KEY_SERVER_ID, serverId);
            json.put(JSON_KEY_SERVER_TYPE, serverType);
            json.put(JSON_KEY_SERVER_PERMISSION_TYPE, serverPermissionType);
            json.put(JSON_KEY_SERVER_PERMISSION_ID, serverPermissionId);
        } catch (JSONException e) {
            AccuratLogger.log(AccuratLogger.JSON_ERROR, TAG + ".getJson(): " + e.getMessage());
            e.printStackTrace();
        }

        return json;
    }
    // </editor-fold>

    // <editor-fold desc="Server JSON Handling">
    public static Consent fromServerJson(JSONObject json, boolean isSystemPermission) {
        try {
            Consent consent = new Consent();

            consent.serverId = json.getInt("id");

            boolean isInAppPermission = false;
            try {
                isInAppPermission = "inapp_permission".equals(json.getString(ServerDataKeys.Consent.TYPE));
            } catch (JSONException ignored) {
            }
            if (isInAppPermission) {
                consent.systemPermissionState = ConsentState.UNKNOWN;
            } else if (isSystemPermission) {
                consent.serverPermissionType = json.getString(ServerDataKeys.Consent.TYPE);
                consent.systemPermissionBackgroundState = ConsentState.UNKNOWN;
                consent.systemPermissionTitle = cleanServerHtml(json.getString(ServerDataKeys.Consent.TITLE));
                consent.systemPermissionDescription = cleanServerHtml(json.getString(ServerDataKeys.Consent.MESSAGE));
                consent.systemPermissionButtonNext = json.getString(ServerDataKeys.Consent.NEXT);
                consent.systemPermissionRefuseCount = 0;
                consent.maxPermissionRefuseCount = json.has(ServerDataKeys.Consent.SCREEN_REPEAT) ? json.getInt(ServerDataKeys.Consent.SCREEN_REPEAT) : consent.maxPermissionRefuseCount;
            } else {
                consent.serverType = json.getString(ServerDataKeys.Consent.TYPE);
                consent.type = ConsentType.fromServerType(consent.serverType);
                consent.state = ConsentState.UNKNOWN;
                consent.title = cleanServerHtml(json.getString(ServerDataKeys.Consent.TITLE));
                consent.description = cleanServerHtml(json.getString(ServerDataKeys.Consent.MESSAGE));
                consent.buttonAccept = json.getString(ServerDataKeys.Consent.ACCEPT);
                consent.buttonRefuse = json.getString(ServerDataKeys.Consent.REFUTE);
                consent.serverPermissionId = json.getInt(ServerDataKeys.Consent.SYSTEM_CONSENT_ID);
                if (json.has(ServerDataKeys.Consent.BRAND_COLOR)) {
                    String brandColor = json.getString(ServerDataKeys.Consent.BRAND_COLOR);
                    // JSONObject.getString() will return "null" if there's a mapping, but no value.
                    if (!JSON_EMPTY_STRING_VALUE_AFTER_PARSING.equals(brandColor)) {
                        consent.brandColor = brandColor;
                    }
                }
                consent.refuseCount = json.has(ServerDataKeys.Consent.REFUSE_COUNT) ? json.getInt(ServerDataKeys.Consent.REFUSE_COUNT) : 0;
                consent.maxConsentRefuseCount = json.has(ServerDataKeys.Consent.SCREEN_REPEAT) ? json.getInt(ServerDataKeys.Consent.SCREEN_REPEAT) : consent.maxConsentRefuseCount;
            }

            return consent;
        } catch (JSONException e) {
            AccuratLogger.log(AccuratLogger.JSON_ERROR, TAG + ".fromServerJson(): " + e.getMessage());
            e.printStackTrace();

            return null;
        }
    }

    public void mergePermission(Consent consent) {
        serverPermissionType = consent.serverPermissionType;
        systemPermissionState = consent.systemPermissionState;
        if (!state.equals(systemPermissionState)) {
            state = systemPermissionState;
        }
        systemPermissionTitle = consent.systemPermissionTitle;
        systemPermissionDescription = consent.systemPermissionDescription;
        systemPermissionButtonNext = consent.systemPermissionButtonNext;
        systemPermissionRefuseCount = consent.systemPermissionRefuseCount;
        maxPermissionRefuseCount = consent.maxPermissionRefuseCount;
    }

    private static String cleanServerHtml(String html) {
        if (html == null) {
            return null;
        }

        return html.replaceAll("\\r\\n", "");
    }
    // </editor-fold>

    // <editor-fold desc="Getters">
    public ConsentType getType() {
        return type;
    }

    public ConsentState getState() {
        return state;
    }

    public String getTitle() {
        return title;
    }

    public String getDescription() {
        return description;
    }

    public String getButtonAccept() {
        return buttonAccept;
    }

    public String getButtonRefuse() {
        return buttonRefuse;
    }

    public String getBrandColor() {
        if (brandColor == null) {
            return null;
        }

        if (!brandColor.startsWith("#")) {
            return "#" + brandColor.toUpperCase();
        }

        return brandColor.toUpperCase();
    }

    public int getRefuseCount() {
        return refuseCount;
    }

    public int getMaxConsentRefuseCount() {
        return maxConsentRefuseCount;
    }

    public ConsentState getSystemPermissionState() {
        return systemPermissionState;
    }

    public ConsentState getSystemPermissionBackgroundState() {
        return systemPermissionBackgroundState;
    }

    /**
     * Get the live version of the systemPermissionState (and cache it in this object).
     *
     * @param context The current Context.
     * @return The ConsentState of the system permission.
     */
    public ConsentState getSystemPermissionState(Context context) {
        if (type == null) {
            // This Consent isn't (properly) initialised, so we shouldn't assume anything about
            // the systemPermissionState.
            systemPermissionState = ConsentState.UNKNOWN;
        } else {
            if (type.hasPermissions(context)) {
                systemPermissionState = ConsentState.ACCEPTED;
            } else if (systemPermissionRefuseCount == 0) {
                // The system permission has not yet been actively refused, so it is in it's default
                // state and we should presume UNKNOWN.
                systemPermissionState = ConsentState.UNKNOWN;
            } else {
                // The system permission has been actively refused.
                systemPermissionState = ConsentState.REFUSED;
            }
        }

        return systemPermissionState;
    }

    /**
     * Get the live version of the systemPermissionBackgroundState (and cache it in this object).
     *
     * @param context The current Context.
     * @return The ConsentState of the system permission.
     */
    public ConsentState getSystemPermissionBackgroundState(Context context) {
        if (type == null) {
            // This Consent isn't (properly) initialised, so we shouldn't assume anything about
            // the systemPermissionBackgroundState.
            systemPermissionBackgroundState = ConsentState.UNKNOWN;
        } else {
            if(VersionManager.isBelowAndroid10()) {
                // When on an Android version without background permissions, this should return the fine/coarse permission state
                systemPermissionBackgroundState = getSystemPermissionState(context);
            } else if (type.hasBackgroundPermission(context)) {
                systemPermissionBackgroundState = ConsentState.ACCEPTED;
            } else if (systemPermissionRefuseCount == 0) {
                // The system permission has not yet been actively refused, so it is in it's default
                // state and we should presume UNKNOWN.
                systemPermissionBackgroundState = ConsentState.UNKNOWN;
            } else {
                // The system permission has been actively refused.
                systemPermissionBackgroundState = ConsentState.REFUSED;
            }
        }

        return systemPermissionBackgroundState;
    }

    public String getSystemPermissionTitle() {
        return systemPermissionTitle;
    }

    public String getSystemPermissionDescription() {
        return systemPermissionDescription;
    }

    public String getSystemPermissionButtonNext() {
        return systemPermissionButtonNext;
    }

    public int getSystemPermissionRefuseCount() {
        return systemPermissionRefuseCount;
    }

    public int getMaxPermissionRefuseCount() {
        return maxPermissionRefuseCount;
    }

    public int getServerId() {
        return serverId;
    }

    public String getServerType() {
        if ("tracking".equals(serverType)) {
            // Upgrade to API v2
            return "gdpr";
        }

        return serverType;
    }

    public String getServerPermissionType() {
        if ("tracking_permission".equals(serverPermissionType)) {
            // Upgrade to API v2
            return "inapp_permission";
        }

        return serverPermissionType;
    }

    public String getServerBackgroundPermissionType() {
        return "always_permission";
    }

    public int getServerPermissionId() {
        return serverPermissionId;
    }
    // </editor-fold>

    // <editor-fold desc="Setters">
    public void setType(ConsentType type) {
        this.type = type;
    }

    public void approve() {
        state = ConsentState.ACCEPTED;
    }

    public void refuse() {
        state = ConsentState.REFUSED;
    }

    public void reset() {
        state = ConsentState.UNKNOWN;
        systemPermissionState = ConsentState.UNKNOWN;
        systemPermissionBackgroundState = ConsentState.UNKNOWN;
        systemPermissionRefuseCount = 0;
    }

    public void approvePermission() {
        systemPermissionState = ConsentState.ACCEPTED;
        VersionManager.whenBelow10(() -> systemPermissionRefuseCount = 0);
    }

    public void refusePermission() {
        systemPermissionState = ConsentState.REFUSED;
    }

    public void increasePermissionRefuseCount() {
        systemPermissionRefuseCount++;
    }

    public void updatePermissionRefuseCount(int refuseCount) {
        systemPermissionRefuseCount = refuseCount;
    }

    public void approveBackgroundPermission() {
        systemPermissionBackgroundState = ConsentState.ACCEPTED;
        VersionManager.when10OrHigher(() -> systemPermissionRefuseCount = 0);
    }

    public void refuseBackgroundPermission() {
        systemPermissionBackgroundState = ConsentState.REFUSED;
    }

    public void setState(ConsentState consentState) {
        state = consentState;
    }

    public void setSystemPermissionState(ConsentState systemPermissionState) {
        this.systemPermissionState = systemPermissionState;
    }

    public void setSystemPermissionBackgroundState(ConsentState systemPermissionBackgroundState) {
        this.systemPermissionBackgroundState = systemPermissionBackgroundState;
    }

    public void increaseRefuseCount() {
        this.refuseCount++;
    }

    public void updateRefuseCount(int refuseCount) {
        this.refuseCount = refuseCount;
    }
    // </editor-fold>


    @Override
    public String toString() {
        int sysPermLength = systemPermissionDescription == null ? 0 : systemPermissionDescription.length();
        return "Consent{" +
                "type=" + type +
                ", state=" + state +
                ", title='" + title + '\'' +
                ", description='" + description + '\'' +
                ", buttonAccept='" + buttonAccept + '\'' +
                ", buttonRefuse='" + buttonRefuse + '\'' +
                ", brandColor='" + brandColor + '\'' +
                ", refuseCount='" + refuseCount + '\'' +
                ", maxConsentRefuseCount='" + maxConsentRefuseCount + '\'' +
                ", systemPermissionState=" + systemPermissionState +
                ", systemPermissionBackgroundState=" + systemPermissionBackgroundState +
                ", systemPermissionTitle='" + systemPermissionTitle + '\'' +
                ", systemPermissionDescription='" + (sysPermLength == 0 ? null : systemPermissionDescription.substring(0, Math.min(sysPermLength, 100))) + '\'' +
                ", systemPermissionButtonNext='" + systemPermissionButtonNext + '\'' +
                ", systemPermissionRefuseCount=" + systemPermissionRefuseCount +
                ", maxPermissionRefuseCount=" + maxPermissionRefuseCount +
                ", serverId=" + serverId +
                ", serverType='" + serverType + '\'' +
                ", serverPermissionType='" + serverPermissionType + '\'' +
                ", serverPermissionId=" + serverPermissionId +
                '}';
    }
}
