package ai.accurat.sdk.core;

import android.content.Context;
import androidx.annotation.NonNull;

import java.util.concurrent.TimeUnit;

import ai.accurat.sdk.Accurat;
import androidx.work.Constraints;
import androidx.work.ExistingPeriodicWorkPolicy;
import androidx.work.NetworkType;
import androidx.work.PeriodicWorkRequest;
import androidx.work.WorkManager;
import androidx.work.Worker;
import androidx.work.WorkerParameters;

/**
 * Checks whether the user has given their consent for personalised ads.
 */
public class ConsentVerifier extends Worker {

    private static final String TAG = ConsentVerifier.class.getSimpleName();
    private static final String UNIQUE_WORK_NAME = ConsentVerifier.class.getCanonicalName();

    public ConsentVerifier(@NonNull Context context, @NonNull WorkerParameters workerParams) {
        super(context, workerParams);
    }

    @NonNull
    @Override
    public Result doWork() {
        AccuratLogger.init(getApplicationContext());
        AccuratLogger.log(AccuratLogger.WORKMANAGER, TAG + ".doWork()");
        AdvertisingManager.init(this.getApplicationContext());
        AdvertisingManager.start(this.getApplicationContext(), success -> {
            if (success) {
                AccuratLogger.log(AccuratLogger.SDK_FLOW, "Continue tracking");

                return;
            }

            AccuratLogger.log(AccuratLogger.ADVERTISING, "Consent for personalised ads has been revoked, stop location tracking");

            // Tracking the location is only allowed when the user has given their permission to personalize ads.
            Accurat.stopTracking(this.getApplicationContext());

            // Cancel periodic location tracking
            WorkManager.getInstance().cancelUniqueWork(UNIQUE_WORK_NAME);
        });

        AccuratLogger.log(AccuratLogger.WORKMANAGER, TAG + " - Work done, returning Result.success()");
        return Result.success();
    }

    /**
     * Helper function to periodically verify whether the user has still given their permission to personalize ads.
     */
    public static void planConsentVerification() {
        Constraints constraints = new Constraints.Builder()
                .setRequiredNetworkType(NetworkType.UNMETERED)
                .setRequiresBatteryNotLow(true)
                .build();
        PeriodicWorkRequest workRequest = new PeriodicWorkRequest.Builder(ConsentVerifier.class, 1L, TimeUnit.DAYS)
                .setConstraints(constraints)
                .build();

        WorkManager.getInstance().enqueueUniquePeriodicWork(ConsentVerifier.UNIQUE_WORK_NAME, ExistingPeriodicWorkPolicy.REPLACE, workRequest);
        AccuratLogger.log(AccuratLogger.SDK_FLOW, "Planned periodic consent verification");
    }
}
