package ai.accurat.sdk.core;

import android.content.Context;
import androidx.annotation.NonNull;

import com.android.volley.AuthFailureError;
import com.android.volley.Request;
import com.android.volley.RequestQueue;
import com.android.volley.toolbox.JsonArrayRequest;
import com.android.volley.toolbox.JsonObjectRequest;
import com.android.volley.toolbox.Volley;
import com.google.gson.Gson;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import ai.accurat.sdk.callbacks.AccuratActionableCallback;
import ai.accurat.sdk.config.Configuration;
import ai.accurat.sdk.constants.AccuratEndpoints;
import ai.accurat.sdk.constants.ApiKeys;
import ai.accurat.sdk.constants.HttpMethod;
import ai.accurat.sdk.constants.ServerDataKeys;
import ai.accurat.sdk.constants.StorageKeys;
import ai.accurat.sdk.data.enums.GeofenceType;

/**
 * A class responsible for fetching geofences from the server.
 */
public class CustomGeofenceSyncManager {

    // <editor-fold desc="Properties">
    private static final String TAG = CustomGeofenceSyncManager.class.getSimpleName();

    private static MultiProcessStorage storage;
    private static RequestQueue requestQueue;
    // </editor-fold>

    // <editor-fold desc="Initialisation">
    public static void init(Context context) {
        if (!isInitialized()) {
            AccuratLogger.init(context);
            AccuratLogger.log(AccuratLogger.SDK_FLOW, "Initialising " + TAG);
            AccuratSettingsManager.init(context);
            CustomGeofenceStorageManager.init(context);
            storage = MultiProcessStorage.getStorage(context, StorageKeys.ACCURAT_MULTI_PROCESS_STORAGE);
        }

        if (requestQueue == null) {
            requestQueue = Volley.newRequestQueue(context);
        }
    }

    private static boolean isInitialized() {
        return storage != null;
    }

    private static void checkInitialized() {
        if (isInitialized()) {
            return;
        }

        throw new IllegalStateException(TAG + " has not yet been initialised.");
    }
    // </editor-fold>

    // <editor-fold desc="Public interface">

    /**
     * Updates the geofences in storage.
     *
     * @param context  Used to access the network and storage.
     * @param callback Used to communicate API request and parsing success.
     *                 {@link AccuratActionableCallback#onActionPotentiallyRequired(boolean)} will be used to communicate storage changes.
     */
    public static void sync(Context context, AccuratActionableCallback callback) {
        sync(context, callback, false);
    }

    /**
     * Updates the geofences in storage.
     *
     * @param context           Used to access the network and storage.
     * @param callback          Used to communicate API request and parsing success.
     *                          {@link AccuratActionableCallback#onActionPotentiallyRequired(boolean)} will be used to communicate storage changes.
     * @param startedFromWorker Should be true when the manager has been started from a periodic build job to avoid endless loops.
     */
    public static void sync(Context context, AccuratActionableCallback callback, boolean startedFromWorker) {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".sync()");
        checkInitialized();

        if (hasGeofences()) {
            // If geofences are present in the database, check if they should be updated
            AccuratLogger.log(AccuratLogger.STORAGE, "Geofences found in storage");
            AccuratLogger.log(AccuratLogger.SDK_FLOW, "Going to check geofences status");
            checkGeofenceStatus(context, callback);
        } else {
            // If no geofences are present in the database, fetch them from the server
            AccuratLogger.log(AccuratLogger.STORAGE, "No geofences in storage");
            AccuratLogger.log(AccuratLogger.SDK_FLOW, "Going to fetch geofences");
            fetchGeofences(context, callback);
        }

        // Plan periodic synchronisation of geofences
        if (!startedFromWorker) {
            AccuratLogger.log(AccuratLogger.SDK_FLOW, "Going to plan sync");
            GeofencesSyncWorker.planGeofencesSync();
        }
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".sync()");
    }

    public static void stop() {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".stop()");
        if(requestQueue != null) {
            AccuratLogger.log(AccuratLogger.SDK_FLOW, "Stopping requestQueue...");
            requestQueue.stop();
        }
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".stop()");
    }
    // </editor-fold>

    // <editor-fold desc="Network calls">
    private static void checkGeofenceStatus(Context context, AccuratActionableCallback callback) {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".checkGeofenceStatus()");
        // Check if the app has a network connection
        if (!AccuratApi.isNetworkAvailable(context)) {
            AccuratLogger.log(AccuratLogger.NETWORK, "Network not available");
            fail(callback);
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".checkGeofenceStatus()");

            return;
        }

        // Build geofences state request body
        JSONObject requestBody = new JSONObject();
        try {
            String currentChecksum = storage.getString(StorageKeys.ACCURAT_GEOFENCES_STATUS, null);
            AccuratLogger.log(AccuratLogger.STORAGE, "Local geofence checksum = " + (currentChecksum == null ? "[unset]" : currentChecksum));
            requestBody.put(ApiKeys.GeofenceStatus.CHECKSUM, currentChecksum == null ? "unknown" : currentChecksum);
        } catch (JSONException e) {
            AccuratLogger.log(AccuratLogger.JSON_ERROR, "Failed to load local geofence checksum from storage: " + e.getMessage());
        }

        // Build geofences state request
        JsonObjectRequest getGeofenceStatusRequest = new JsonObjectRequest(
                Request.Method.POST,
                AccuratEndpoints.GET_GEOFENCES_STATE.getUrl(),
                requestBody,
                response -> {
                    AccuratLogger.logNetworkResponse(HttpMethod.POST, Configuration.ENDPOINT_GET_GEOFENCES_STATE, response, false);
                    handleGeofenceStatusResponse(context, response, callback);
                },
                error -> {
                    AccuratLogger.logNetworkError(HttpMethod.POST, Configuration.ENDPOINT_GET_GEOFENCES_STATE, error);
                    fail(callback);
                }) {
            @Override
            public Map<String, String> getHeaders() throws AuthFailureError {
                return AccuratApi.getHeaders(storage, "POST", getBodyContentType(),
                        AccuratApi.getEncodedRequestBody(requestBody.toString()),
                        AccuratEndpoints.GET_GEOFENCES_STATE.getPath());
            }
        };

        getGeofenceStatusRequest.setTag(TAG)
                .setRetryPolicy(AccuratApi.defaultRetryPolicy)
                .setShouldCache(false);

        // Request geofences file state
        if (requestQueue == null) {
            AccuratLogger.log(AccuratLogger.WARNING, "requestQueue is NULL, re-initialising queue....");
            requestQueue = Volley.newRequestQueue(context);
        }

        AccuratLogger.logNetworkRequest(HttpMethod.POST, Configuration.ENDPOINT_GET_GEOFENCES_STATE, requestBody, false);
        requestQueue.add(getGeofenceStatusRequest);
    }

    /**
     * Fetches the geofences.
     *
     * @param context  Used for network and storage access.
     * @param callback Used to communicate success, failure and potential actions (when storage has been updated)
     */
    private static void fetchGeofences(final Context context, @NonNull final AccuratActionableCallback callback) {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".fetchGeofences()");
        if (!AccuratSettingsManager.isGeofencingEnabled()) {
            AccuratLogger.log(AccuratLogger.SDK_FLOW, "Not fetching custom geofences: Geofencing is disabled");
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".fetchGeofences()");
            succeed(callback);

            return;
        }

        // Check if the app has a network connection
        if (!AccuratApi.isNetworkAvailable(context)) {
            AccuratLogger.log(AccuratLogger.NETWORK, "Network not available");
            fail(callback);
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".fetchGeofences()");

            return;
        }

        // Build the geofences request
        JsonArrayRequest geofencesRequest = new JsonArrayRequest(
                Request.Method.GET,
                AccuratEndpoints.GET_GEOFENCES.getUrl(),
                null,
                response -> {
                    AccuratLogger.logNetworkResponse(HttpMethod.GET, Configuration.ENDPOINT_GET_GEOFENCES, response, false);
                    handleGeofencesResponse(context, response, callback);
                },
                error -> {
                    AccuratLogger.logNetworkError(HttpMethod.GET, Configuration.ENDPOINT_GET_GEOFENCES, error);
                    fail(callback);
                }) {
            @Override
            public Map<String, String> getHeaders() throws AuthFailureError {
                return AccuratApi.getHeaders(storage, "GET", "",
                        AccuratApi.getEncodedRequestBody(""), AccuratEndpoints.GET_GEOFENCES.getPath());
            }
        };

        geofencesRequest.setTag(TAG)
                .setRetryPolicy(AccuratApi.defaultRetryPolicy)
                .setShouldCache(false);

        // Request geofences
        if (requestQueue == null) {
            AccuratLogger.log(AccuratLogger.WARNING, "requestQueue is NULL, re-initialising queue....");
            AccuratLogger.log(AccuratLogger.ERROR, "Failed to make API call, requestQueue is null");
            requestQueue = Volley.newRequestQueue(context);
        }

        AccuratLogger.logNetworkRequest(HttpMethod.GET, Configuration.ENDPOINT_GET_GEOFENCES);
        requestQueue.add(geofencesRequest);
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".fetchGeofences()");
    }
    // </editor-fold>

    // <editor-fold desc="Network response helpers">
    /**
     * Determines the next steps with the state of the geofences file
     *
     * @param context  Context to access storage, and network if need be
     * @param response Response of {@link AccuratEndpoints#GET_GEOFENCES_STATE}
     * @param callback Used to communicate success, failure and potential actions (when storage has been updated)
     */
    private static void handleGeofenceStatusResponse(Context context, JSONObject response, final AccuratActionableCallback callback) {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".handleGeofenceStatusResponse()");
        init(context);
        if (response == null || !response.has(ServerDataKeys.GeofenceStatus.DATA)) {
            AccuratLogger.log(AccuratLogger.ERROR, "Can't process geofence state, response is NULL or missing 'data' element");
            fail(callback);
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".handleGeofenceStatusResponse()");

            return;
        }

        // Verify validity of response
        final JSONObject data;
        try {
            data = response.getJSONObject(ServerDataKeys.GeofenceStatus.DATA);
        } catch (JSONException e) {
            AccuratLogger.log(AccuratLogger.JSON_ERROR, "Failed to parse geofence state: " + e.getMessage());
            fail(callback);
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".handleGeofenceStatusResponse()");

            return;
        }

        if (data == null || !data.has(ServerDataKeys.GeofenceStatus.STATUS)) {
            AccuratLogger.log(AccuratLogger.ERROR, "'data' element is NULL or missing 'status' attribute");
            fail(callback);
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".handleGeofenceStatusResponse()");

            return;
        }

        // Check whether new geofences are available and download them if need be
        try {
            String status = data.getString(ServerDataKeys.GeofenceStatus.STATUS);
            if (ServerDataKeys.GeofenceStatus.StatusValues.UPDATED.equals(status)) {
                AccuratLogger.log(AccuratLogger.SDK_FLOW, "Geofences have been updated on the server");
                AccuratLogger.log(AccuratLogger.SDK_FLOW, "Going to fetch geofences");
                // Fetching geofences from the server
                fetchGeofences(context, new AccuratActionableCallback() {
                    @Override
                    public void onCompleted(boolean success) {
                        try {
                            // Save the new checksum to storage
                            String checksum = data.getString(ServerDataKeys.GeofenceStatus.CHECKSUM);
                            AccuratLogger.log(AccuratLogger.SDK_FLOW, "Setting local geofences checksum to " + checksum);
                            storage.setValue(StorageKeys.ACCURAT_GEOFENCES_STATUS, checksum).commit();
                        } catch (JSONException e) {
                            AccuratLogger.log(AccuratLogger.JSON_ERROR, "Failed to set local geofence checksum: " + e.getMessage());
                        }

                        succeed(callback);
                    }

                    @Override
                    public void onActionPotentiallyRequired(boolean isActionRequired) {
                        if(callback != null) {
                            callback.onActionPotentiallyRequired(isActionRequired);
                        }
                    }
                });
            } else {
                AccuratLogger.log(AccuratLogger.SDK_FLOW, "Geofences have not been updated on the server");
                succeed(callback);
            }
        } catch (JSONException e) {
            AccuratLogger.log(AccuratLogger.JSON_ERROR, "Failed to parse geofence status: " + e.getMessage());
        }
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".handleGeofenceStatusResponse()");
    }

    /**
     * Handle the response from fetching the geofences from the server.
     *
     * @param context  Used to access local storage.
     * @param response The geofences that should be stored.
     * @param callback Callback to indicate that the geofences in local storage have been updated.
     */
    private static void handleGeofencesResponse(Context context, JSONArray response, AccuratActionableCallback callback) {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".handleGeofencesResponse()");
        init(context);
        // Parse the geofences (Gson from Niels)
        final Gson gson = new Gson();
        final int numGeofences = response.length();
        if (numGeofences == 0) {
            AccuratLogger.log(AccuratLogger.STORAGE, "No geofences to store");
            succeed(callback);
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".storeGeofences()");

            return;
        }
        AccuratLogger.log(AccuratLogger.DATABASE, "Storing " + numGeofences + " geofences");
        List<AccuratGeofence> geofences = new ArrayList<>(numGeofences);
        for (int i = 0; i < numGeofences; i++) {
            try {
                JSONObject geofenceData = response.getJSONObject(i);
                AccuratGeofence geofence = gson.fromJson(geofenceData.toString(), AccuratGeofence.class);
                geofences.add(geofence);
            } catch (JSONException e) {
                AccuratLogger.log(AccuratLogger.JSON_ERROR, "Failed to parse geofences response: " + e.getMessage());
            }
        }
        logGeofences(geofences);

        CustomGeofenceStorageManager.storeServerGeofences(geofences);
        succeed(callback);

        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".handleGeofencesResponse()");
    }
    // </editor-fold>

    private static void logGeofences(List<AccuratGeofence> geofences) {
        List<AccuratGeofence> entryGeofences = new ArrayList<>();
        List<AccuratGeofence> dwellGeofences = new ArrayList<>();
        List<AccuratGeofence> exitGeofences = new ArrayList<>();
        List<AccuratGeofence> invalidGeofences = new ArrayList<>();
        for (AccuratGeofence geofence : geofences) {
            if (GeofenceType.ENTER.matches(geofence.getType())) {
                entryGeofences.add(geofence);
            } else if (GeofenceType.DWELL.matches(geofence.getType())) {
                dwellGeofences.add(geofence);
            } else if (GeofenceType.EXIT.matches(geofence.getType())) {
                exitGeofences.add(geofence);
            } else {
                invalidGeofences.add(geofence);
            }
        }
        AccuratLogger.log(AccuratLogger.SDK_FLOW, "-- Of which " + entryGeofences.size() + " are ENTER geofences:");
        for (AccuratGeofence geofence : entryGeofences) {
            AccuratLogger.log(AccuratLogger.NONE, "-- -- " + geofence.toString());
        }
        AccuratLogger.log(AccuratLogger.SDK_FLOW, "-- Of which " + dwellGeofences.size() + " are DWELL geofences:");
        for (AccuratGeofence geofence : dwellGeofences) {
            AccuratLogger.log(AccuratLogger.NONE, "-- -- " + geofence.toString());
        }
        AccuratLogger.log(AccuratLogger.SDK_FLOW, "-- Of which " + exitGeofences.size() + " are EXIT geofences:");
        for (AccuratGeofence geofence : exitGeofences) {
            AccuratLogger.log(AccuratLogger.NONE, "-- -- " + geofence.toString());
        }
        if (!invalidGeofences.isEmpty()) {
            AccuratLogger.log(AccuratLogger.WARNING, "-- Of which " + invalidGeofences.size() + " are invalid geofences:");
            for (AccuratGeofence geofence : invalidGeofences) {
                AccuratLogger.log(AccuratLogger.NONE, "-- -- " + geofence.toString());
            }
        }
    }

    // <editor-fold desc="Helpers">
    private static boolean hasGeofences() {
        return CustomGeofenceStorageManager.getStoredGeofenceCount() > 0;
    }

    /**
     * Helper method to indicate that the call has failed to complete.
     */
    private static void fail(AccuratActionableCallback callback) {
        if(callback == null) {
            return;
        }

        callback.onCompleted(false);
    }

    private static void succeed(AccuratActionableCallback callback) {
        if(callback == null) {
            return;
        }

        callback.onCompleted(true);
    }
    // </editor-fold>
}
