package ai.accurat.sdk.core;

import android.content.Context;
import android.database.sqlite.SQLiteFullException;
import androidx.annotation.NonNull;

import androidx.work.Worker;
import androidx.work.WorkerParameters;

import com.google.android.gms.common.util.CollectionUtils;

import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Locale;
import java.util.concurrent.TimeUnit;

import ai.accurat.sdk.Accurat;
import ai.accurat.sdk.constants.StorageKeys;
import ai.accurat.sdk.data.RealmHelper;
import ai.accurat.sdk.data.models.Setting;
import ai.accurat.sdk.managers.RealmManager;

/**
 * @author Kenneth Saey
 * @Both
 * @since 04-06-2018 10:46.
 */
public class DispatchWorker extends Worker {

    private static final String TAG = DispatchWorker.class.getSimpleName();
    private static final long EXPIRATION_PERIOD = TimeUnit.DAYS.toMillis(10);

    private List<OpenLocateBasedEndpoint> mEndpoints;

    public DispatchWorker(@NonNull Context context, @NonNull WorkerParameters workerParams) {
        super(context, workerParams);
    }

    @NonNull
    @Override
    public Result doWork() {
        AccuratLogger.init(getApplicationContext());
        AccuratLogger.log(AccuratLogger.WORKMANAGER, TAG + ".doWork()");
        RealmManager.init(getApplicationContext());
        AccuratSettingsManager.init(this.getApplicationContext());

        AccuratSettings settings = AccuratSettingsManager.getSettings();
        if (settings == null) {
            AccuratLogger.log(AccuratLogger.ERROR, "Could not get endpoints, settings are NULL");
            AccuratLogger.log(AccuratLogger.WORKMANAGER, TAG + " - Work done, returning Result.retry()");
            checkRestartTracker();

            return Result.retry();
        }

        mEndpoints = settings.getEndpoints();
        if (mEndpoints == null) {
            AccuratLogger.log(AccuratLogger.ERROR, "Could not parse endpoints");
            AccuratLogger.log(AccuratLogger.WORKMANAGER, TAG + " - Work done, returning Result.failure()");
            checkRestartTracker();

            return Result.failure();
        }
        if (mEndpoints.size() <= 0) {
            AccuratLogger.log(AccuratLogger.NONE, "No endpoints to dispatch to");
            AccuratLogger.log(AccuratLogger.WORKMANAGER, TAG + " - Work done, returning Result.success()");
            checkRestartTracker();

            return Result.success();
        }
        checkRestartTracker();

        return sendLocations();
    }

    // <editor-fold desc="Sending data">
    private Result sendLocations() {
        if (mEndpoints == null) {
            AccuratLogger.log(AccuratLogger.WORKMANAGER, TAG + " - Work done, returning Result.failure()");

            return Result.failure();
        }

        // Init resources
        LocationDataSource dataSource = new AccuratDatabase(DatabaseHelper.getInstance(getApplicationContext()));
        HttpClient httpClient = new HttpClientImpl();
        LocationDispatcher locationDispatcher = new LocationDispatcher();

        List<Long> timestamps = new ArrayList<>(mEndpoints.size());
        MultiProcessStorage storage = MultiProcessStorage.getStorage(this.getApplicationContext(), StorageKeys.ACCURAT_MULTI_PROCESS_STORAGE);

        // Send the locations
        AccuratLogger.log(AccuratLogger.DISPATCH, "Dispatching to " + mEndpoints.size() + " endpoints");

        for (OpenLocateBasedEndpoint endpoint : mEndpoints) {
            AccuratLogger.log(AccuratLogger.DISPATCH, "Dispatching to " + endpoint.getUrl());

            String endpointKey = generateKey(endpoint);
            try {
                long lastSentTimestamp = getLastSentTimestamp(endpointKey);

                List<LocationInterface> sentLocations = locationDispatcher.postLocations(httpClient, endpoint, lastSentTimestamp, dataSource, storage);
                if (!CollectionUtils.isEmpty(sentLocations)) {
                    long mostRecentLocationTimestamp = getMostRecentLocationTimestamp(sentLocations);
                    AccuratLogger.log(AccuratLogger.DISPATCH, "Dispatched locations form " + lastSentTimestamp + " to " + mostRecentLocationTimestamp + " to " + endpoint.getUrl());

                    SharedPreferenceUtils.getInstance(getApplicationContext()).setValue(endpointKey, mostRecentLocationTimestamp);
                    for (LocationInterface locationInterface : sentLocations) {
                        AccuratLogger.log(AccuratLogger.DISPATCH, "Dispatched ["
                                + locationInterface.getInformationFields().getLocationContext() + "] [ "
                                + locationInterface.getLocationInfo().getLatitude() + " | "
                                + locationInterface.getLocationInfo().getLongitude() + " ] from "
                                + new SimpleDateFormat("dd-MM-yyyy HH:mm:ss", Locale.getDefault()).format(new Date(locationInterface.getLocationInfo().getTimeStampSecs() * 1000)));
                    }
                } else {
                    AccuratLogger.log(AccuratLogger.DISPATCH, "No locations dispatched to " + endpoint.getUrl());
                }
            } catch (Exception e) {
                AccuratLogger.log(AccuratLogger.ERROR, e.getMessage());
                e.printStackTrace();
            }
            timestamps.add(getLastSentTimestamp(endpointKey));
        }

        // Clean database
        try {
            if (!CollectionUtils.isEmpty(timestamps)) {
                Long oldestTimestamp = Collections.min(timestamps);
                Long newestTimestamp = Collections.max(timestamps);

                if (oldestTimestamp != null) {
                    long expired = newestTimestamp - EXPIRATION_PERIOD;
                    if (expired > oldestTimestamp) {
                        oldestTimestamp = expired;
                    }

                    dataSource.deleteBefore(oldestTimestamp);
                }
            }
        } catch (SQLiteFullException e) {
            AccuratLogger.log(AccuratLogger.ERROR, "Database is full, cannot purge data");
            e.printStackTrace();
        }

        AccuratLogger.log(AccuratLogger.WORKMANAGER, TAG + " - Work done, returning Result.success()");
        return Result.success();// Todo - Edit to detect when it should be retried
    }
    // </editor-fold>

    // <editor-fold desc="Helpers">
    private String generateKey(OpenLocateBasedEndpoint endpoint) {
        return md5(endpoint.getUrl().toLowerCase());
    }

    public static String md5(String in) {
        MessageDigest digest;
        try {
            digest = MessageDigest.getInstance("MD5");
            digest.reset();
            digest.update(in.getBytes());
            byte[] digestedMessage = digest.digest();
            int len = digestedMessage.length;
            StringBuilder sb = new StringBuilder(len << 1);
            for (byte byteI : digestedMessage) {
                sb.append(Character.forDigit((byteI & 0xf0) >> 4, 16));
                sb.append(Character.forDigit(byteI & 0x0f, 16));
            }
            return sb.toString();
        } catch (NoSuchAlgorithmException e) {
            e.printStackTrace();
        }

        return in;
    }

    private long getLastSentTimestamp(String endpointKey) {
        return SharedPreferenceUtils.getInstance(getApplicationContext()).getLongValue(endpointKey, 0);
    }

    private long getMostRecentLocationTimestamp(List<LocationInterface> locations) {
        if (locations == null || locations.isEmpty()) {
            return -1;
        }

        return locations.get(locations.size() - 1).getCreated().getTime();
    }
    // </editor-fold>

    // <editor-fold desc="Restart location tracker">
    private boolean checkRestartTracker() {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".checkRestartTracker()");
        if (Accurat.isTrackingEnabled(getApplicationContext())) {
            AccuratLogger.log(AccuratLogger.NONE, "Tracking already enabled, nothing to do");
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".checkRestartTracker()");

            return true;
        }

        AccuratLogger.log(AccuratLogger.WARNING, "Accurat is no longer tracking the location");

        MultiProcessStorage storage = MultiProcessStorage.getAccuratStorage(getApplicationContext());
        if (storage == null) {
            AccuratLogger.log(AccuratLogger.WARNING, "Storage not available, can't now if we should restart tracking");
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".checkRestartTracker()");

            return false;
        }

        boolean trackingWasStarted = RealmHelper.loadBooleanSetting(Setting.Keys.State.START_REQUESTED, false);
        if (!trackingWasStarted) {
            AccuratLogger.log(AccuratLogger.NONE, "Tracking was never started, so we shouldn't restart it");
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".checkRestartTracker()");

            return true;
        }

        AccuratLogger.log(AccuratLogger.NONE, "Restarting location tracking...");
        try {
            Accurat.startAfterReboot(getApplicationContext());
        } catch (IllegalStateException e) {
            AccuratLogger.log(AccuratLogger.WARNING, "Accurat has not yet been initialised");
            AccuratLogger.log(AccuratLogger.NONE, "Initialising Accurat...");
            try {
                Accurat.initialize(getApplicationContext());
                AccuratLogger.log(AccuratLogger.NONE, "Restarting location tracking...");
                Accurat.startAfterReboot(getApplicationContext());
            } catch (IllegalStateException ise) {
                AccuratLogger.log(AccuratLogger.ERROR, ise.getMessage());
                AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".checkRestartTracker()");

                return false;
            }
        }
        AccuratLogger.log(AccuratLogger.NONE, "Tracking has been restarted");
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".checkRestartTracker()");

        return true;
    }
    // </editor-fold>
}
