package ai.accurat.sdk.core;

import android.content.Context;
import android.content.Intent;
import android.location.Location;
import androidx.annotation.NonNull;
import androidx.core.app.JobIntentService;

import com.google.android.gms.location.Geofence;
import com.google.android.gms.location.GeofenceStatusCodes;
import com.google.android.gms.location.GeofencingEvent;

import java.util.List;

import ai.accurat.sdk.Accurat;
import ai.accurat.sdk.config.Configuration;


public class GeofenceTransitionsJobIntentService extends JobIntentService {

    private static final int JOB_ID = 7894;
    private static final String TAG = GeofenceTransitionsJobIntentService.class.getSimpleName();

    /**
     * Convenience method for enqueuing work in to this service.
     */
    public static void enqueueWork(Context context, Intent intent) {
        enqueueWork(context, GeofenceTransitionsJobIntentService.class, JOB_ID, intent);
    }

    @Override
    protected void onHandleWork(@NonNull Intent intent) {
        AccuratLogger.init(getApplicationContext());
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".onHandleWork()");
        GeofencingEvent geofencingEvent = GeofencingEvent.fromIntent(intent);
        if (geofencingEvent.hasError()) {
            String errorMessage = "Geofencing event has an error: " + getErrorMessage(geofencingEvent.getErrorCode());
            AccuratLogger.log(AccuratLogger.ERROR, errorMessage);
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".onHandleWork()");
            return;
        }

        // Get the transition type.
        int geofenceTransition = geofencingEvent.getGeofenceTransition();

        // Test that the reported transition was of interest.
        if (geofenceTransition == Geofence.GEOFENCE_TRANSITION_ENTER ||
                geofenceTransition == Geofence.GEOFENCE_TRANSITION_EXIT ||
                geofenceTransition == Geofence.GEOFENCE_TRANSITION_DWELL) {

            // Get the geofences that were triggered. A single event can trigger
            // multiple geofences.
            List<Geofence> triggeringGeofences = geofencingEvent.getTriggeringGeofences();

            // Ensure all system and user variables are initialized (like ad ID and user preferences)
            try {
                Accurat.initialize(getApplicationContext());
            } catch (IllegalStateException e) {
                AccuratLogger.log(AccuratLogger.ERROR, e.getMessage());
                AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".onHandleWork()");

                return;
            }
            GeofencesManager.init(getApplicationContext());
            // Send location update
            AccuratLogger.log(AccuratLogger.SDK_FLOW, "Posting location update from triggered geofence");
            boolean processed = postLocationUpdate(getApplicationContext(), geofencingEvent.getTriggeringLocation(), geofenceTransition, GeofencesManager.shouldUpdateMonitoredGeofences(triggeringGeofences));

            // Get notifications for geofences
            final boolean isExit = geofenceTransition == Geofence.GEOFENCE_TRANSITION_EXIT;
            final boolean isDwelling = geofenceTransition == Geofence.GEOFENCE_TRANSITION_DWELL;
            String transitionType = "exit";
            if (!isExit) {
                transitionType = isDwelling ? "dwell" : "enter";
            }
            for (Geofence triggeringGeofence : triggeringGeofences) {
                if (processed) {
                    // Dwell trigger location is recent enough
                    AccuratLogger.log(AccuratLogger.GEOFENCE, "Triggered geofence with ID " + triggeringGeofence.getRequestId() + " and transition type " + transitionType);
                    GeofencesManager.triggerGeofence(getApplicationContext(), triggeringGeofence, isExit, isDwelling);
                } else if (isDwelling) {
                    if (calculateDwellDistance(geofencingEvent.getTriggeringLocation()) > Configuration.MAX_GEOFENCE_DWELL_DISTANCE_METERS) {
                        // Dwell trigger location is too far from the last known location
                        AccuratLogger.log(AccuratLogger.GEOFENCE, "Ignored geofence with ID " + triggeringGeofence.getRequestId() + " and transition type " + transitionType + " because the location is too far away.");
                    } else {
                        // Dwell trigger location is not recent enough and to far from the last known location
                        AccuratLogger.log(AccuratLogger.GEOFENCE, "Triggered older geofence with ID " + triggeringGeofence.getRequestId() + " and transition type " + transitionType);
                        GeofencesManager.triggerGeofence(getApplicationContext(), triggeringGeofence, isExit, isDwelling);
                    }
                }
            }
        } else {
            // Log the error.
            AccuratLogger.log(AccuratLogger.ERROR, "Invalid transition type: " + geofenceTransition);
        }
    }

    /**
     * Translate Geofence status codes  for errors to human-readable language.
     */
    private String getErrorMessage(int errorCode) {
        switch (errorCode) {
            case GeofenceStatusCodes.GEOFENCE_NOT_AVAILABLE:
                return "Geofence service is not available now";
            case GeofenceStatusCodes.GEOFENCE_TOO_MANY_GEOFENCES:
                return "Your app has registered too many geofences";
            case GeofenceStatusCodes.GEOFENCE_TOO_MANY_PENDING_INTENTS:
                return "You have provided too many PendingIntents to the addGeofences() call";
            default:
                return "Unknown error: the Geofence service is not available now";
        }
    }

    /**
     * Publishes the location where the geofence event has been triggered as a location update.
     *
     * @param context               Context used to save location updates
     * @param triggeringLocation    Location where the geofence was triggered.
     * @param geofenceTransition    Transition of the geofence event
     * @param shouldRedrawGeofences Indicates whether geofences should be redrawn
     */
    private boolean postLocationUpdate(Context context, Location triggeringLocation, int geofenceTransition, boolean shouldRedrawGeofences) {
        LocationContext trackingContext = LocationContext.UNKOWN;
        switch (geofenceTransition) {
            case Geofence.GEOFENCE_TRANSITION_ENTER:
                trackingContext = LocationContext.GEOFENCE_ENTER;
                break;
            case Geofence.GEOFENCE_TRANSITION_EXIT:
                trackingContext = LocationContext.GEOFENCE_EXIT;
                break;
            case Geofence.GEOFENCE_TRANSITION_DWELL:
                trackingContext = LocationContext.GEOFENCE_DWELL;
                break;
        }
        return AccuratLocationManager.getInstance().onLocationChanged(context, triggeringLocation, trackingContext, shouldRedrawGeofences);
    }

    private double calculateDwellDistance(Location dwellTriggerLocation) {
        if (dwellTriggerLocation == null) {
            return Double.MAX_VALUE;
        }
        LocationDataSource locationDataSource = new AccuratDatabase(DatabaseHelper.getInstance(getApplicationContext()));
        LocationInterface lastKnownLocation = locationDataSource.getLastKnownLocation();

        if (lastKnownLocation == null || lastKnownLocation.getLocationInfo() == null) {
            return Double.MAX_VALUE;
        }

        return LocationUtils.distance(
                dwellTriggerLocation.getLongitude(),
                dwellTriggerLocation.getLatitude(),
                lastKnownLocation.getLocationInfo().getLongitude(),
                lastKnownLocation.getLocationInfo().getLatitude()
        );
    }
}
