package ai.accurat.sdk.core;

import java.util.concurrent.TimeUnit;

import ai.accurat.sdk.config.Configuration;
import androidx.work.BackoffPolicy;
import androidx.work.Constraints;
import androidx.work.PeriodicWorkRequest;
import androidx.work.WorkManager;

public class TrackingRestarter {

    private static final String TAG = TrackingRestarter.class.getSimpleName();

    public static boolean startTrackingRestarter(AccuratSettings settings) {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".startTrackingRestarter()");

        long repeatInterval = findRepeatInterval(settings);
        AccuratLogger.log(AccuratLogger.SDK_FLOW, "repeatInterval = " + repeatInterval + " seconds");
        boolean started = startWorkManager(repeatInterval);

        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".startTrackingRestarter()");
        return started;
    }

    private static long findRepeatInterval(AccuratSettings settings) {
        if (settings == null) {
            return Configuration.DEFAULT_LOCATION_RESTART_TIMEOUT;// 6 hours
        }

        long locationRestartTimeout = settings.getLocationRestartTimeout();
        long locationInterval = settings.getLocationInterval();
        long maxWaitTime = settings.getMaxWaitTime() / 1000;// milliseconds => seconds

        long settingsInterval = Math.max(locationInterval, maxWaitTime);

        if (locationRestartTimeout > 0 && locationRestartTimeout >= settingsInterval) {
            // Use the location_restart_timeout server setting
            return clampRepeatInterval(locationRestartTimeout);
        }

        if (settingsInterval > 0) {
            // Use the max_wait_time or location_interval server setting, whichever is longest
            return clampRepeatInterval(settingsInterval * 3);
        }

        return Configuration.DEFAULT_LOCATION_RESTART_TIMEOUT;// 6 hours
    }

    private static long clampRepeatInterval(long repeatInterval) {
        return Math.max(repeatInterval, Configuration.MINIMUM_LOCATION_RESTART_TIMEOUT);// 2 hours
    }

    private static boolean startWorkManager(long repeatInterval) {
        WorkManager workManager = WorkManager.getInstance();

        //noinspection ConstantConditions
        if (workManager == null) {
            AccuratLogger.log(AccuratLogger.WARNING, "WorkManager.getInstance() is null, can't start WorkManager");

            return false;
        }

        // Cancel any previously running tracking restart work
        workManager.cancelAllWorkByTag(Constants.RESTART_TRACKING_WORK_TAG);

        // Create a new WorkRequest
        PeriodicWorkRequest restartTrackingWork = new PeriodicWorkRequest.Builder(
                TrackingRestartWorker.class,
                repeatInterval,
                TimeUnit.SECONDS
        )
                .setBackoffCriteria(
                        BackoffPolicy.EXPONENTIAL,
                        clampRepeatInterval(repeatInterval / 2),
                        TimeUnit.SECONDS
                )
                .setConstraints(getConstraints())
                .addTag(Constants.RESTART_TRACKING_WORK_TAG)
                .build();

        // Schedule the worker
        workManager.enqueue(restartTrackingWork);
        AccuratLogger.log(AccuratLogger.SDK_FLOW, "TrackingRestartWorker started");

        return true;
    }

    private static Constraints getConstraints() {
        return new Constraints.Builder()
                .setRequiresBatteryNotLow(true)
                .build();
    }
}
