package ai.accurat.sdk.viewholders;

import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.graphics.Color;
import android.os.Build;
import android.text.Html;
import android.text.SpannableString;
import android.text.Spanned;
import android.text.TextUtils;
import android.text.method.LinkMovementMethod;
import android.util.Base64;
import android.util.Log;
import android.view.View;

import androidx.appcompat.widget.AppCompatButton;
import androidx.appcompat.widget.AppCompatImageView;
import androidx.appcompat.widget.AppCompatTextView;

import java.util.ArrayList;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import ai.accurat.sdk.R;
import ai.accurat.sdk.core.Consent;
import ai.accurat.sdk.data.enums.ConsentType;
import ai.accurat.sdk.data.models.ConsentModel;
import ai.accurat.sdk.managers.VersionManager;

/**
 * @author Kenneth Saey
 * @since 01-08-2018 14:15.
 */
public class ConsentDialogViewHolder extends ViewHolder {

    public AppCompatTextView title;

    public AppCompatTextView message1;
    public AppCompatImageView image1;
    public AppCompatTextView message2;

    public View containerButtonNegative;
    public View containerNoButtonNegative;

    public AppCompatTextView buttonNegative;
    public AppCompatButton buttonPositive;

    public ConsentDialogViewHolder(View rootView) {
        super(rootView);
    }

    @Override
    public void inflateView() {
        title = i(R.id.title);

        message1 = i(R.id.message_1);
        image1 = i(R.id.image_1);
        message2 = i(R.id.message_2);

        containerButtonNegative = i(R.id.container_button_negative);
        containerNoButtonNegative = i(R.id.container_no_button_negative);

        buttonNegative = i(R.id.button_negative);
        buttonPositive = i(R.id.button_positive);
    }

    // <editor-fold desc="Layout Configuration">
    private void setHasImage(boolean hasImage) {
        image1.setVisibility(hasImage ? View.VISIBLE : View.GONE);
    }

    private void setHasMessage2(boolean hasMessage2) {
        message2.setVisibility(hasMessage2 ? View.VISIBLE : View.GONE);
    }

    private void setHasNegativeButton(boolean hasNegativeButton) {
        containerButtonNegative.setVisibility(hasNegativeButton ? View.VISIBLE : View.GONE);
        containerNoButtonNegative.setVisibility(hasNegativeButton ? View.GONE : View.VISIBLE);
    }

    private void setConsentStyle() {
        setHasImage(false);
        setHasMessage2(false);
        setHasNegativeButton(true);
    }

    private void setPermissionStyle() {
        setHasImage(true);
        setHasMessage2(false);
        setHasNegativeButton(false);
    }

    /**
     * Applies the brand color to the appropriate UI elements.
     *
     * @param brandColor
     */
    private void setBrandColor(int brandColor) {
        title.setTextColor(brandColor);
        title.setLinkTextColor(brandColor);
        message1.setLinkTextColor(brandColor);
        message2.setLinkTextColor(brandColor);
        buttonPositive.setBackgroundColor(brandColor);
    }
    // </editor-fold>

    // <editor-fold desc="Populating the view">
    public ConsentDialogViewHolder setConsent(Consent consent) {
        setConsentStyle();
        setBrandColor(getBrandColor(consent));

        title.setText(consent.getTitle());
        message1.setText(getStyledMessage(consent.getDescription()));

        buttonNegative.setText(consent.getButtonRefuse());
        buttonPositive.setText(consent.getButtonAccept());

        return this;
    }

    public ConsentDialogViewHolder setConsentModel(ConsentModel consent) {
        if (consent.isSystemPermission()) {
            return setPermissionModel(consent);
        }

        setConsentStyle();
        setBrandColor(getBrandColor(consent));

        title.setText(consent.getTitle());
        String message = consent.getMessage();
        Log.d("KENNETH", "consentType = " + consent.getType());
        Log.d("KENNETH", "Build.VERSION.SDK_INT >= Build.VERSION_CODES.R ? " + (Build.VERSION.SDK_INT >= Build.VERSION_CODES.R));
        if (consent.getType() == ConsentType.ANDROID_11_ALWAYS_LOCATION && Build.VERSION.SDK_INT >= Build.VERSION_CODES.R) {
            Log.d("KENNETH", "-- Replacing $permissionoptionlabel$ in " + message);
            Log.d("KENNETH", "with " + rootView.getContext().getPackageManager().getBackgroundPermissionOptionLabel());
            message = message.replace("$permissionoptionlabel$", rootView.getContext().getPackageManager().getBackgroundPermissionOptionLabel());
        }
        message1.setText(getStyledMessage(message));

        buttonNegative.setText(consent.getRefute());
        buttonPositive.setText(consent.getAccept());

        return this;
    }

    public ConsentDialogViewHolder setPermission(Consent consent) {
        setPermissionStyle();
        setBrandColor(getBrandColor(consent));

        title.setText(consent.getSystemPermissionTitle());
        message1.setText(getStyledMessage(extractDescription(consent.getSystemPermissionDescription())));
        if (VersionManager.isBelowAndroid10()) {
            String[] images = extractImages(consent.getSystemPermissionDescription());
            if (images.length > 0) {
                byte[] decodedBytes = Base64.decode(images[0], Base64.DEFAULT);
                Bitmap bitmap = BitmapFactory.decodeByteArray(decodedBytes, 0, decodedBytes.length);
                image1.setImageBitmap(bitmap);
            }
        } else {
            image1.setImageResource(R.drawable.android_10_permission_dialog);
        }

        buttonPositive.setText(consent.getSystemPermissionButtonNext());

        return this;
    }

    private ConsentDialogViewHolder setPermissionModel(ConsentModel consent) {
        setPermissionStyle();
        setBrandColor(getBrandColor(consent));

        title.setText(consent.getTitle());
        String message = consent.getMessage();
        Log.d("KENNETH", "consentType = " + consent.getType());
        Log.d("KENNETH", "Build.VERSION.SDK_INT >= Build.VERSION_CODES.R ? " + (Build.VERSION.SDK_INT >= Build.VERSION_CODES.R));
        if (consent.getType() == ConsentType.ANDROID_11_ALWAYS_LOCATION && VersionManager.is11OrHigher()) {
            Log.d("KENNETH", "-- Replacing $permissionoptionlabel$ in " + message);
            Log.d("KENNETH", "with " + rootView.getContext().getPackageManager().getBackgroundPermissionOptionLabel());
            message = message.replace("$permissionoptionlabel$", rootView.getContext().getPackageManager().getBackgroundPermissionOptionLabel());
        }
        message1.setText(getStyledMessage(extractDescription(message)));

        if (VersionManager.isBelowAndroid10()) {
            String[] images = extractImages(consent.getMessage());
            if (images.length > 0) {
                byte[] decodedBytes = Base64.decode(images[0], Base64.DEFAULT);
                Bitmap bitmap = BitmapFactory.decodeByteArray(decodedBytes, 0, decodedBytes.length);
                image1.setImageBitmap(bitmap);
            }
        } else if (VersionManager.is11OrHigher()) {
            if (consent.getType() == ConsentType.ANDROID_11_ALWAYS_LOCATION) {
                image1.setImageResource(R.drawable.android_11_permission_settings);
            } else {
                image1.setImageResource(R.drawable.android_11_permission_dialog);
            }
        } else {
            image1.setImageResource(R.drawable.android_10_permission_dialog);
        }

        buttonPositive.setText(consent.getNext());
        if (shouldShowNegativeButton(consent)) {
            setHasNegativeButton(true);
            buttonNegative.setText(consent.getRefute());
        }

        return this;
    }

    private boolean shouldShowNegativeButton(ConsentModel consent) {
        return (consent.getType() == ConsentType.ANDROID_11_ALWAYS_LOCATION && VersionManager.is11OrHigher())
                || consent.getType() == ConsentType.IN_APP_LOCATION;
    }
    // </editor-fold>

    // <editor-fold desc="Action handlers">
    public ConsentDialogViewHolder setNegativeButtonHandler(View.OnClickListener onClickListener) {
        buttonNegative.setOnClickListener(onClickListener);

        return this;
    }

    public ConsentDialogViewHolder setPositiveButtonHandler(View.OnClickListener onClickListener) {
        buttonPositive.setOnClickListener(onClickListener);

        return this;
    }
    // </editor-fold>

    // <editor-fold desc="Helpers">
    public void enableDialogLinks() {
        message1.setMovementMethod(LinkMovementMethod.getInstance());
        message2.setMovementMethod(LinkMovementMethod.getInstance());
    }

    private SpannableString getStyledMessage(String message) {
        Spanned htmlMessage;
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) {
            htmlMessage = Html.fromHtml(message != null ? message : "", Html.FROM_HTML_MODE_LEGACY);
        } else {
            htmlMessage = Html.fromHtml(message != null ? message : "");
        }

        SpannableString styledMessage = new SpannableString(htmlMessage);

        return styledMessage;
    }

    private int getBrandColor(Consent consent) {
        String brandColor = consent.getBrandColor();
        if (brandColor != null) {
            return Color.parseColor(brandColor);
        }

        return getColor(R.color.accurat_accent);
    }

    private int getBrandColor(ConsentModel consentModel) {
        String brandColor = consentModel.getBrandColour().trim();
        if (!TextUtils.isEmpty(brandColor)) {
            return Color.parseColor(brandColor);
        }

        return getColor(R.color.accurat_accent);
    }

    private String extractDescription(String message) {
        if (message == null) {
            return "";
        }

        String regex = "<img src=\"[^>]*>";

        return message.replaceAll(regex, "");
    }

    private String[] extractImages(String message) {
        if (message == null) {
            return new String[]{};
        }

        String regex = "<img src=\"([^\"]+)\"[^>]*>";
        Pattern pattern = Pattern.compile(regex);
        Matcher matcher = pattern.matcher(message);

        ArrayList<String> matches = new ArrayList<>();
        while (matcher.find()) {
            matches.add(matcher.group(1));
        }

        return matches.toArray(new String[0]);
    }
    // </editor-fold>
}
