package ai.accurat.sdk.viewholders;

import androidx.annotation.ColorInt;
import androidx.annotation.ColorRes;
import androidx.annotation.IdRes;
import androidx.annotation.NonNull;
import androidx.annotation.StringRes;
import androidx.core.content.ContextCompat;
import android.view.View;

/**
 * @author Kenneth Saey
 * @since 01-08-2018 14:14.
 */
public abstract class ViewHolder {

    public View rootView;

    public ViewHolder(View rootView) {
        this.rootView = rootView;
        inflateView();
        fillView();
    }

    /**
     * Inflate the View for this ViewHolder.
     */
    public abstract void inflateView();

    /**
     * Fill the View for this ViewHolder.
     */
    public void fillView() {
    }

    /**
     * Returns a localized string from the application's package's
     * default string table.
     *
     * @param resId Resource id for the string
     * @return The string data associated with the resource, stripped of styled
     * text information.
     */
    @NonNull
    protected String getString(@StringRes int resId) {
        if (rootView != null) {
            return rootView.getContext().getString(resId);
        }

        return "";
    }

    /**
     * Returns a localized formatted string from the application's package's
     * default string table, substituting the format arguments as defined in
     * {@link java.util.Formatter} and {@link String#format}.
     *
     * @param resId      Resource id for the format string
     * @param formatArgs The format arguments that will be used for
     *                   substitution.
     * @return The string data associated with the resource, formatted and
     * stripped of styled text information.
     */
    @NonNull
    protected String getString(@StringRes int resId, Object... formatArgs) {
        if (rootView != null) {
            return rootView.getContext().getString(resId, formatArgs);
        }

        return "";
    }

    /**
     * Returns a color associated with a particular resource ID and styled for
     * the current theme.
     *
     * @param id The desired resource identifier, as generated by the aapt
     *           tool. This integer encodes the package, type, and resource
     *           entry. The value 0 is an invalid identifier.
     * @return A single color value in the form 0xAARRGGBB.
     * @throws android.content.res.Resources.NotFoundException if the given ID
     *                                                         does not exist.
     */
    @ColorInt
    public final int getColor(@ColorRes int id) {
        if (rootView != null) {
            return ContextCompat.getColor(rootView.getContext(), id);
        }

        return 0;
    }

    protected <T extends View> T i(@IdRes int id) {
        return rootView.findViewById(id);
    }
}
