/*
 * Copyright (c) 2017 OpenLocate
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
package ai.accurat.sdk.core;

import android.database.sqlite.SQLiteOpenHelper;

import java.util.List;

/**
 * Source: OpenLocate.LocationDatabase (locations only)
 * Expanded upon to support Geofences since 2.0.0
 *
 * @Both
 */
public final class AccuratDatabase implements LocationDataSource, GeofenceDataSource {

    private SQLiteOpenHelper helper;

    public AccuratDatabase(SQLiteOpenHelper helper) {
        this.helper = helper;
    }

    //<editor-fold desc="LocationDataSource">
    @Override
    public void addAll(List<LocationInterface> locations) {
        LocationTable.addAll(helper.getWritableDatabase(), locations);
    }

    @Override
    public void add(LocationInterface location) {
        LocationTable.add(helper.getWritableDatabase(), location);
    }

    @Override
    public List<LocationInterface> getSince(long millisecondsSince1970) {
        return LocationTable.getSince(helper.getWritableDatabase(), millisecondsSince1970);
    }

    @Override
    public void deleteBefore(long millisecondsSince1970) {
        LocationTable.deleteBefore(helper.getWritableDatabase(), millisecondsSince1970);
    }

    @Override
    public long getNumberOfLocations() {
        return LocationTable.size(helper.getReadableDatabase());
    }

    @Override
    public LocationInterface getLastKnownLocation() {
        return LocationTable.getLastKnownLocation(helper.getReadableDatabase());
    }

    //</editor-fold>

    //<editor-fold desc="GeofenceDataSource">
    @Override
    public void addGeofences(List<AccuratGeofence> geofences) {
        GeofenceContract.addAll(helper.getWritableDatabase(), geofences);
    }

    @Override
    public long getNumberOfGeofences() {
        return GeofenceContract.count(helper.getReadableDatabase());
    }

    @Override
    public List<AccuratGeofence> getGeofences() {
        return GeofenceContract.getAll(helper.getReadableDatabase());
    }

    @Override
    public List<AccuratGeofence> getGeofencesWithinBox(LocationInterface location, AccuratGeofenceRange box) {
        return GeofenceContract.getAllWithinBox(helper.getReadableDatabase(), location, box);
    }

    @Override
    public List<AccuratGeofence> getGeofencesWithinBox(SearchBox searchBox) {
        return GeofenceContract.getAllWithinBox(helper.getReadableDatabase(), searchBox);
    }

    @Override
    public List<AccuratGeofence> getRecentGeofences(int limit) {
        return GeofenceContract.getRecent(helper.getReadableDatabase(), limit);
    }

    @Override
    public AccuratGeofence getGeofenceById(String serverId) {
        return GeofenceContract.findById(helper.getReadableDatabase(), serverId);
    }

    @Override
    public long removeGeofences() {
        return GeofenceContract.removeGeofences(helper.getWritableDatabase());
    }

    //</editor-fold>
}
