package ai.accurat.sdk.core;

import android.content.Context;
import android.database.sqlite.SQLiteFullException;
import android.location.Location;

import androidx.annotation.NonNull;

import com.google.android.gms.common.ConnectionResult;

import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Locale;

import ai.accurat.sdk.callbacks.AccuratActionableCallback;
import ai.accurat.sdk.callbacks.AccuratProcessCallback;
import ai.accurat.sdk.data.models.ConsentManagerState;
import ai.accurat.sdk.managers.AccuratManager;
import ai.accurat.sdk.managers.RealmManager;

/**
 * Manager class used to track the location of a user.
 */
public final class AccuratLocationManager {

    //<editor-fold desc="Fields">
    private static final int PLAY_SERVICES_RESOLUTION_REQUEST = 9000;
    private static final String TAG = AccuratLocationManager.class.getSimpleName();
    private static AccuratLocationManager instance;

    private SimpleDateFormat timeFormat;
    private boolean running;
    private DispatcherInterface dispatcher;
    private LocationTrackerInterface locationTracker;

    private LocationInterface lastKnownLocation;

    //</editor-fold>

    private AccuratLocationManager() {
        this.timeFormat = new SimpleDateFormat("dd-MM-yyyy HH:mm:ss", Locale.getDefault());
    }

    //<editor-fold desc="Properties">

    /**
     * This class is implemented following SDP.
     *
     * @return The instance of {@link AccuratLocationManager}.
     */
    public static AccuratLocationManager getInstance() {
        if (instance == null) {
            instance = new AccuratLocationManager();
        }
        return instance;
    }

    /**
     * @return true when the location is being tracked, false when the manager is not active.
     */
    public boolean isRunning() {
        return this.running;
    }
    //</editor-fold>

    //<editor-fold desc="Public API">

    /**
     * Initialize the AccuratLocationManager.
     *
     * @param context Context object used to initialise the location tracker and dispatcher.
     *                Will also be used to access local storage.
     */
    public void init(@NonNull Context context) {
        AccuratLogger.init(context);
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".init()");
        AccuratSettingsManager.init(context);

        this.locationTracker = new LocationTrackerImpl();
        this.locationTracker.initLocationTracking(context);

        // Init geofences manager
        LocationDataSource locationDataSource = new AccuratDatabase(DatabaseHelper.getInstance(context));
        this.lastKnownLocation = locationDataSource.getLastKnownLocation();
//        GeofencesManager.init(context);
        CustomGeofenceManager.init(context);
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".init()");
    }

    /**
     * Will be called when the location is updated.
     */
    public boolean onLocationChanged(Context context, Location location, LocationContext trackingContext, boolean shouldRedrawGeofences) {
        AccuratLogger.init(context);
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".onLocationChanged()");
        RealmManager.init(context);

        long lastKnownLocationTimestamp = SharedPreferenceUtils.getInstance(context).getLongValue(Constants.LAST_LOCATION_TIMESTAMP, 0L);
        if (location.getTime() < lastKnownLocationTimestamp) {
            AccuratLogger.log(AccuratLogger.NONE, "Ignoring location update because it's too old (" + timeFormat.format(location.getTime()) + " < " + timeFormat.format(lastKnownLocationTimestamp) + ")");
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".onLocationChanged()");

            return false;
        }

        Calendar timestamp = Calendar.getInstance();
        timestamp.setTimeInMillis(location.getTime());
        String logString = "Location [ " + location.getLatitude() + " | " + location.getLongitude()
                + " ] (speed = " + location.getSpeed()
                + ") received at " + timeFormat.format(timestamp.getTime())
                + "(" + timestamp.getTimeZone().getDisplayName()
                + ") from provider " + location.getProvider()
                + " and context " + trackingContext.getValue()
                + " [Timestamp " + timestamp.getTime().getTime()
                + "]";
        AccuratLogger.log(AccuratLogger.TRACKER, logString);

        // Verify whether all details are available
        if (this.locationTracker == null) {
            AccuratLogger.log(AccuratLogger.WARNING, "locationTracker is NULL, initialising it...");
            init(context);
        }

        if (this.locationTracker.getConfiguration() == null) {
            AccuratLogger.log(AccuratLogger.WARNING, "configuration is NULL, initialising it...");
            this.locationTracker.initLocationTracking(context);
        }

//        AdvertisingManager.init(context);// Should be initialized to be sure to have an ad_id
        AccuratLogger.log(AccuratLogger.ADVERTISING, "Transforming location, ad ID is " + ConsentManagerState.Companion.getAdId());


        // Transform location and keep it in a list of locations together with ad ID and more contextual info
        LocationInterface loc = this.locationTracker.transformLocation(context, location, ConsentManagerState.Companion.getAdInfo(), null);
        loc.getInformationFields().setLocationContext(trackingContext);

        // Save location in database
        saveLocation(context, loc);

//        AccuratLogger.log(AccuratLogger.SDK_FLOW, "Going to check for a hard coded geofence");
//        GeofencesManager.checkHardcodedGeofence(context, location);

        // Notify the geofences manager that the user's location has changed
        if (shouldRedrawGeofences && location.getTime() > lastKnownLocationTimestamp) {
            AccuratLogger.log(AccuratLogger.SDK_FLOW, "Passing location update to CustomGeofenceManager");
            CustomGeofenceManager.onLocationChanged(context, location);
            this.lastKnownLocation = loc;
        } else {
            AccuratLogger.log(AccuratLogger.WARNING, "Not passing location update to CustomGeofenceManager, because:");
            if (!shouldRedrawGeofences) {
                AccuratLogger.log(AccuratLogger.NONE, "shouldRedrawGeofences = false");
            }
            if (location.getTime() <= lastKnownLocationTimestamp) {
                AccuratLogger.log(AccuratLogger.NONE, "location.getTime() <= lastKnownLocationTimestamp");
                AccuratLogger.log(AccuratLogger.NONE, location.getTime() + " <= " + lastKnownLocationTimestamp);
            }
        }

        // Save the timestamp of this location, for service diagnostics
        SharedPreferenceUtils.getInstance(context).setValue(Constants.LAST_LOCATION_TIMESTAMP, location.getTime());
        AccuratLogger.log(AccuratLogger.SDK_FLOW, "Setting lastKnownLocationTimestamp to " + location.getTime());

        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".onLocationChanged()");
        return true;
    }

    /**
     * Starts tracking the location.
     *
     * @param context  Context used to access the network and local storage
     * @param callback Callback which indicates whether the location tracking is now active
     */
    public void start(final Context context, @NonNull final AccuratProcessCallback callback) {
        AccuratLogger.init(context);
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".start()");
        // Do not track the location if you're missing any consents
//        if (ConsentManager.hasMissingConsents() || ConsentManager.hasUnapprovedConsents()) {
        if (!AccuratManager.INSTANCE.canStartTracking()) {
            if (callback != null) {
                callback.onProcessed(false);
            }
            this.running = false;
            AccuratLogger.log(AccuratLogger.NONE, "Setting AccuratLocationManager.running = false");
            AccuratLogger.log(AccuratLogger.WARNING, "Can't start tracking, because the user hasn't given the required consents");
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".start()");
            return;
        }

        // Check Google Play Services
        int resultCode = PlayServicesUtils.isGooglePlayServicesAvailable(context);
        if (resultCode != ConnectionResult.SUCCESS) {
//            GoogleApiAvailability.getInstance().getErrorDialog(context, resultCode, PLAY_SERVICES_RESOLUTION_REQUEST).show();
            AccuratLogger.log(AccuratLogger.ERROR, "Google Play Services is not available");
            if (callback != null) {
                callback.onProcessed(false);
            }
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".start()");
            return;
        }

        AccuratLogger.log(AccuratLogger.SDK_FLOW, "Starting AdvertisingManager");
        // Update advertising info
//        AdvertisingManager.start(context, success -> {
//            if (success) {
                AccuratLogger.log(AccuratLogger.SDK_FLOW, "Starting LocationTracker");
                locationTracker.startLocationTracking(context, started -> {
                    AccuratLogger.log(AccuratLogger.NONE, "Setting AccuratLocationManager.running = " + started);
                    running = started;
                    if (started) {
                        dispatcher = new Dispatcher();
                        dispatcher.initDispatcher(context);

                        AccuratLogger.log(AccuratLogger.SDK_FLOW, "Starting Dispatcher");
                        dispatcher.startDispatcher(dispatched -> {
                            AccuratLogger.log(AccuratLogger.SDK_FLOW, "Dispatcher started");
                        });

                        AccuratLogger.log(AccuratLogger.SDK_FLOW, "Starting TrackingRestarter");
                        TrackingRestarter.startTrackingRestarter(AccuratSettingsManager.getSettings());

                        AccuratLogger.log(AccuratLogger.SDK_FLOW, "Scheduling SettingsManager");
                        AccuratSettingsManager.schedule();
                        AccuratLogger.log(AccuratLogger.SDK_FLOW, "Scheduling GeofenceNotificationManager");
                        GeofenceNotificationManager.schedule();

                        // Start and plan sync of geofences
//                        syncAndTrackGeofences(context);
                        CustomGeofenceSyncManager.sync(context, null);
                    }

                    if (callback != null) {
                        callback.onProcessed(started);
                    }
                });
//            } else {
//                AccuratLogger.log(AccuratLogger.WARNING, "Could not fetch advertising info, or the user didn't give their consent");
//                if (callback != null) {
//                    callback.onProcessed(false);
//                }
//            }
//        });
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".start()");
    }

    /**
     * Stop tracking the location.
     *
     * @param context Used to stop long-running background tasks
     */
    public void stop(Context context) {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".stop()");
        AccuratLogger.log(AccuratLogger.SDK_FLOW, "Stopping LocationTracker");
        this.locationTracker.stopLocationTracking(context);

        AccuratLogger.log(AccuratLogger.SDK_FLOW, "Stopping Dispatcher");
        if (dispatcher == null) {
            dispatcher = new Dispatcher();
            dispatcher.initDispatcher(context);
        }
        this.dispatcher.stopDispatcher(context);

        AccuratLogger.log(AccuratLogger.SDK_FLOW, "Stopping SettingsManager");
        AccuratSettingsManager.stop();

        AccuratLogger.log(AccuratLogger.NONE, "Setting AccuratLocationManager.running = false");
        this.running = false;

        AccuratLogger.log(AccuratLogger.SDK_FLOW, "Stopping CustomGeofencesManager");
        CustomGeofenceManager.stop();
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".stop()");
    }
    //</editor-fold>

    //<editor-fold desc="Helpers">

    /**
     * Saves a location in the database.
     *
     * @param context  Context used to access the database
     * @param location Recently tracked location to remember.
     */
    private void saveLocation(Context context, LocationInterface location) {
        try {
            // Instantiate the location database
            LocationDataSource locations = new AccuratDatabase(DatabaseHelper.getInstance(context));

            // Add the location to the database
            locations.add(location);
        } catch (SQLiteFullException exception) {
            AccuratLogger.log(AccuratLogger.ERROR, "Could not save location [ "
                    + location.getLocationInfo().getLatitude() + " " + location.getLocationInfo().getLongitude()
                    + " ], database is full");
        }
    }

    /**
     * Synchronize the geofences. Is supposed to called when location tracking is starting.
     * Will also start monitoring the geofences.
     *
     * @param context Used to access the network & local storage
     */
    private void syncAndTrackGeofences(final Context context) {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".syncAndTrackGeofences()");
        // Todo - CustomGeofenceManager - Implement
        GeofencesManager.sync(context, new AccuratActionableCallback() {
            @Override
            public void onCompleted(boolean success) {
                if (success) {
                    AccuratLogger.log(AccuratLogger.SDK_FLOW, "Sync succeeded");
                    // Determine last known location to determine closest geofences
                    if (lastKnownLocation == null) {
                        LocationDataSource locationSource = new AccuratDatabase(DatabaseHelper.getInstance(context));
                        lastKnownLocation = locationSource.getLastKnownLocation();
                    }

                    AccuratLogger.log(AccuratLogger.SDK_FLOW, "Going to track geofences");
                    GeofencesManager.startTracking(context, lastKnownLocation, started -> {
                        AccuratLogger.log(AccuratLogger.SDK_FLOW, "Geofence tracking " + (started ? "" : "not ") + "started");
                    });
                } else {
                    AccuratLogger.log(AccuratLogger.SDK_FLOW, "Sync failed");
                    GeofencesSyncWorker.planGeofencesSync();
                }
            }

            @Override
            public void onActionPotentiallyRequired(boolean isActionRequired) {
                // TODO refresh R*Tree
            }
        });
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".syncAndTrackGeofences()");
    }

    /**
     * For debugging purposes.
     *
     * @return The last known location
     */
    public LocationInterface getLastLocation() {
        return lastKnownLocation;
    }
    //</editor-fold>

}
