package ai.accurat.sdk.core;

import android.content.Context;
import android.text.TextUtils;
import android.util.LongSparseArray;

import androidx.work.Constraints;
import androidx.work.NetworkType;
import androidx.work.OneTimeWorkRequest;
import androidx.work.PeriodicWorkRequest;
import androidx.work.WorkManager;

import com.android.volley.AuthFailureError;
import com.android.volley.Request;
import com.android.volley.RequestQueue;
import com.android.volley.toolbox.JsonObjectRequest;
import com.android.volley.toolbox.Volley;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.TimeUnit;

import ai.accurat.sdk.callbacks.AccuratCompletionCallback;
import ai.accurat.sdk.config.Configuration;
import ai.accurat.sdk.constants.AccuratEndpoints;
import ai.accurat.sdk.constants.ApiKeys;
import ai.accurat.sdk.constants.HttpMethod;
import ai.accurat.sdk.constants.StorageKeys;
import ai.accurat.sdk.data.RealmHelper;
import ai.accurat.sdk.data.models.Setting;
import io.reactivex.annotations.NonNull;

public class GeofenceNotificationManager {

    // <editor-fold desc="Private constants">
    private static final String TAG = GeofenceNotificationManager.class.getSimpleName();
    private static final GeofenceNotificationManager INSTANCE = new GeofenceNotificationManager();
    // </editor-fold>

    // <editor-fold desc="Fields">
    private static MultiProcessStorage storage;
    private static RequestQueue requestQueue;
    private static LongSparseArray<ArrayList<GeofenceNotification>> notifications;
    private static GeofenceNotificationLog notificationLog;
    // </editor-fold>

    // <editor-fold desc="Initialisation">
    public static void init(@NonNull Context context) {
        if (!isInitialized()) {
            AccuratLogger.init(context);
            AccuratLogger.log(AccuratLogger.SDK_FLOW, "Initialising " + TAG);
            AdvertisingManager.init(context);
            AccuratUserManager.init(context);
            PeriodicSync.init(context);
            storage = MultiProcessStorage.getStorage(context, StorageKeys.ACCURAT_MULTI_PROCESS_STORAGE);
            requestQueue = Volley.newRequestQueue(context);
        } else if (requestQueue == null) {
            requestQueue = Volley.newRequestQueue(context);
        }
    }

    private static boolean isInitialized() {
        return storage != null;
    }

    private static void checkInitialized() {
        if (!isInitialized()) {
            AccuratLogger.log(AccuratLogger.ERROR, "GeofenceNotificationManager has not yet been initialised.");
            throw new IllegalStateException("GeofenceNotificationManager has not yet been initialised.");
        }
    }

    static GeofenceNotificationManager getInstance() {
        return INSTANCE;
    }
    // </editor-fold>

    // <editor-fold desc="Public interface">
    public static void start() {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".start()");
        AccuratLogger.log(AccuratLogger.SDK_FLOW, "Loading offline geofence notifications");
        loadNotifications();

        if (notifications == null || notifications.size() == 0) {
            AccuratLogger.log(AccuratLogger.STORAGE, "No notifications in storage");
            AccuratLogger.log(AccuratLogger.SDK_FLOW, "Fetching offline geofence notifications from server");
            INSTANCE.fetchNotifications(success -> {
                AccuratLogger.log(AccuratLogger.SDK_FLOW, "Fetching " + (success ? "succeeded" : "failed"));
                AccuratLogger.log(AccuratLogger.STORAGE, "Notifications = ");
                logNotifications();
            });
        }

        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".start()");
    }

    /**
     * Get the first available {@link GeofenceNotification} for each notification ID.
     *
     * @param notificationIds The list of notification IDs.
     * @return The list of {@link GeofenceNotification}s
     */
    public static LongSparseArray<GeofenceNotification> getNotifications(Long[] notificationIds) {
        if (notificationIds == null || notificationIds.length == 0) {
            return new LongSparseArray<>();
        }

        // Make sure the notification are loaded
        loadNotifications();

        if (notifications == null || notifications.size() == 0) {
            return new LongSparseArray<>();
        }

        LongSparseArray<GeofenceNotification> filteredNotifications = new LongSparseArray<>();
        for (Long notificationId : notificationIds) {
            ArrayList<GeofenceNotification> list = notifications.get(notificationId);
            if (list == null || list.isEmpty()) {
                continue;
            }

            filteredNotifications.put(notificationId, list.get(0));
        }

        return filteredNotifications;
    }

    public static GeofenceNotificationLog getNotificationLog() {
        if (notificationLog == null) {
            loadNotificationLog();

            if (notificationLog == null) {
                notificationLog = new GeofenceNotificationLog();
            }
        }

        return notificationLog;
    }
    // </editor-fold>

    // <editor-fold desc="Storage">
    void fetchNotifications(final AccuratCompletionCallback onCompleted) {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".fetchNotifications()");
        if (!AccuratSettingsManager.isGeofencingEnabled()) {
            AccuratLogger.log(AccuratLogger.SDK_FLOW, "Not fetching notifications: Geofencing is disabled");
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".fetchNotifications()");
            if (onCompleted != null) {
                onCompleted.onCompleted(true);
            }

            return;
        }

        if (!AdvertisingManager.hasValidAdId()) {
            AccuratLogger.log(AccuratLogger.WARNING, "No valid ad ID, can't fetch");
            if (onCompleted != null) {
                onCompleted.onCompleted(false);
            }
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".fetchNotifications()");

            return;
        }

        if (requestQueue == null) {
            AccuratLogger.log(AccuratLogger.ERROR, "Failed to make API call, requestQueue is null");
            complete(onCompleted, false);
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".fetchNotifications()");

            return;
        }

        JsonObjectRequest notificationsRequest = new JsonObjectRequest(
                Request.Method.GET,
                AccuratEndpoints.GET_GEOFENCES_NOTIFICATIONS.getUrl(getUrlParameters()),
                null,
                response -> {
                    AccuratLogger.logNetworkResponse(HttpMethod.GET, Configuration.ENDPOINT_GET_GEOFENCES_NOTIFICATIONS, response, false);
                    if (response == null || !response.has("data")) {
                        complete(onCompleted, false);

                        return;
                    }

                    try {
                        JSONArray data = response.getJSONArray("data");
                        ArrayList<GeofenceNotification> notifications = new ArrayList<>();
                        for (int i = 0; i < data.length(); i++) {
                            notifications.add(GeofenceNotification.fromServerJson(data.getJSONObject(i)));
                        }

                        storeNotifications(notifications);
                        loadNotifications(notifications);

                        AccuratLogger.log(AccuratLogger.SDK_FLOW, "Processed offline geofence notifications from server");
                        complete(onCompleted, true);
                    } catch (JSONException e) {
                        AccuratLogger.log(AccuratLogger.JSON_ERROR, "Failed to parse offline geofence notifications: " + e.getMessage());
                        complete(onCompleted, false);
                    }
                },
                error -> {
                    AccuratLogger.logNetworkError(HttpMethod.GET, Configuration.ENDPOINT_GET_GEOFENCES_NOTIFICATIONS, error);
                    complete(onCompleted, false);
                }
        ) {
            @Override
            public Map<String, String> getHeaders() throws AuthFailureError {
                return AccuratApi.getHeaders(
                        storage,
                        "GET",
                        "",
                        AccuratApi.getEncodedRequestBody(""),
                        AccuratEndpoints.GET_GEOFENCES_NOTIFICATIONS.getPath(getUrlParameters())
                );
            }
        };

        notificationsRequest.setTag(TAG)
                .setRetryPolicy(AccuratApi.defaultRetryPolicy)
                .setShouldCache(false);

        AccuratLogger.logNetworkRequest(HttpMethod.GET, Configuration.ENDPOINT_GET_GEOFENCES_NOTIFICATIONS);
        requestQueue.add(notificationsRequest);
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".fetchNotifications()");
    }

    private static void storeNotifications(ArrayList<GeofenceNotification> notifications) {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".storeNotifications()");
        checkInitialized();

        storage.setValue(StorageKeys.ACCURAT_GEOFENCE_NOTIFICATIONS, JSON.toJsonObjectArray(notifications).toString())
                .commit();
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".storeNotifications()");
    }

    private static void loadNotifications() {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".loadNotifications()");
        checkInitialized();

        String jsonString = storage.getString(StorageKeys.ACCURAT_GEOFENCE_NOTIFICATIONS, "");
        if (jsonString == null || jsonString.isEmpty()) {
            AccuratLogger.log(AccuratLogger.STORAGE, "No offline geofence notifications in storage");
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".loadNotifications()");
            return;
        }

        try {
            JSONArray jsonArray = new JSONArray(jsonString);
            ArrayList<GeofenceNotification> notifications = JSON.loadObjectList(jsonArray, new ArrayList<>(), GeofenceNotification.class);
            loadNotifications(notifications);
        } catch (JSONException e) {
            AccuratLogger.log(AccuratLogger.JSON_ERROR, "Failed to parse GeofenceNotifications from storage: " + e.getMessage());
            e.printStackTrace();
        }
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".loadNotifications()");
    }

    private static void loadNotifications(ArrayList<GeofenceNotification> notificationList) {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".loadNotifications(ArrayList<GeofenceNotification>)");
        notifications = new LongSparseArray<>();
        for (GeofenceNotification notification : notificationList) {
            long nid = notification.getNotificationId();
            if (notifications.indexOfKey(nid) < 0) {
                notifications.put(nid, new ArrayList<>());
            }
            notifications.get(nid).add(notification);// Todo - Are nid's unique?
        }
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".loadNotifications(ArrayList<GeofenceNotification>)");
    }

    private static void loadNotificationLog() {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".loadNotificationLog()");
        AccuratLogger.log(AccuratLogger.SDK_FLOW, "Checking if GeofenceNotificationManager is initialised");
        checkInitialized();
        AccuratLogger.log(AccuratLogger.SDK_FLOW, "-- Yes, it is");

        AccuratLogger.log(AccuratLogger.SDK_FLOW, "Loading notification log from storage");
        String jsonString = RealmHelper.loadStringSetting(Setting.Keys.State.GEOFENCE_NOTIFICATION_LOG, "");;
        AccuratLogger.log(AccuratLogger.SDK_FLOW, "jsonString = " + jsonString);
        AccuratLogger.log(AccuratLogger.SDK_FLOW, "Checking if jsonString is empty");
        if (TextUtils.isEmpty(jsonString)) {
            AccuratLogger.log(AccuratLogger.STORAGE, "No notification log in storage");
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".loadNotificationLog()");

            return;
        }
        AccuratLogger.log(AccuratLogger.SDK_FLOW, "jsonString is not empty");
        AccuratLogger.log(AccuratLogger.SDK_FLOW, "Trying to parse the jsonString");

        try {
            JSONObject jsonObject = new JSONObject(jsonString);
            AccuratLogger.log(AccuratLogger.SDK_FLOW, "Successfully parsed it as a JSONObject");
            AccuratLogger.log(AccuratLogger.SDK_FLOW, "Trying to parse it as a GeofenceNotificationLog");
            notificationLog = new GeofenceNotificationLog(jsonObject);
            AccuratLogger.log(AccuratLogger.SDK_FLOW, "Successfully parsed it as a GeofenceNotificationLog");
        } catch (JSONException e) {
            AccuratLogger.log(AccuratLogger.JSON_ERROR, "Failed to parse notification log from storage: " + e.getMessage());
            e.printStackTrace();
        }

        AccuratLogger.log(AccuratLogger.STORAGE_DATA, notificationLog == null ? "null" : notificationLog.toString());
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".loadNotificationLog()");
    }

    public static void storeNotificationLog() {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".storeNotificationLog()");
        checkInitialized();

        AccuratLogger.log(AccuratLogger.NONE, "Notification log = " + notificationLog.toString());
        RealmHelper.storeSetting(Setting.Keys.State.GEOFENCE_NOTIFICATION_LOG, JSON.toJson(notificationLog).toString());
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".storeNotificationLog()");
    }
    // </editor-fold>

    // <editor-fold desc="Scheduling">
    static void schedule() {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".schedule()");
        checkInitialized();

        WorkManager workManager = WorkManager.getInstance();

        // Cancel all previously running work
        workManager.cancelAllWorkByTag(Constants.SYNC_GEOFENCES_NOTIFICATIONS_WORK_TAG);

        // Needs to have an initial delay since a periodic task does not have such a constraint.
        long syncTime = PeriodicSync.getSyncTime(StorageKeys.ACCURAT_GEOFENCES_NOTIFICATIONS_SYNC_MILLIS);
        long delayDuration = syncTime - System.currentTimeMillis();
        // The delay should be positive (in the future). If not, delay by a day (in ms)
        long checkedDelayDuration = delayDuration > 0 ? delayDuration : 86400000;
        AccuratLogger.log(AccuratLogger.NONE, TAG + " should trigger around " + Configuration.LOG_TIME_FORMAT.format(new Date(System.currentTimeMillis() + checkedDelayDuration)));

        OneTimeWorkRequest.Builder builder = new OneTimeWorkRequest.Builder(AccuratScheduleGeofenceNotificationsWorker.class)
                .setInitialDelay(checkedDelayDuration, TimeUnit.MILLISECONDS)
                .addTag(Constants.SYNC_SETTINGS_WORK_TAG);

        // Schedule the work
        workManager.enqueue(builder.build());
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".schedule()");
    }

    static void scheduleFetch() {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".scheduleFetch()");
        checkInitialized();

        WorkManager workManager = WorkManager.getInstance();

        // Cancel any previously running work
        workManager.cancelAllWorkByTag(Constants.SYNC_GEOFENCES_NOTIFICATIONS_WORK_TAG);

        // Create a new WorkRequest
        PeriodicWorkRequest.Builder builder = new PeriodicWorkRequest.Builder(
                AccuratGeofenceNotificationsWorker.class,
                1,
                TimeUnit.DAYS
        ).setConstraints(getWorkerConstraints())
                .addTag(Constants.SYNC_GEOFENCES_NOTIFICATIONS_WORK_TAG);

        // Schedule the work
        workManager.enqueue(builder.build());
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".scheduleFetch()");
    }

    static void stop() {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".stop()");
        WorkManager workManager = WorkManager.getInstance();

        // Cancel any previously running work
        workManager.cancelAllWorkByTag(Constants.SYNC_GEOFENCES_NOTIFICATIONS_WORK_TAG);
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".stop()");
    }

    static void cancelNetworkRequests() {
        if (requestQueue != null) {
            requestQueue.cancelAll(TAG);
        }
    }

    private static Constraints getWorkerConstraints() {
        return new Constraints.Builder()
                .setRequiresBatteryNotLow(true)
                .setRequiredNetworkType(NetworkType.CONNECTED)
                .build();
    }
    // </editor-fold>

    // <editor-fold desc="Private helpers">
    private HashMap<String, Object> getUrlParameters() {
        HashMap<String, Object> urlParameters = new HashMap<>();

        urlParameters.put(ApiKeys.Url.AD_ID, AdvertisingManager.getAdId());
        urlParameters.put(ApiKeys.Url.LANGUAGE_KEY, AccuratUserManager.getUserLanguage());

        return urlParameters;
    }

    private void complete(AccuratCompletionCallback onCompleted, boolean result) {
        if (onCompleted != null) {
            onCompleted.onCompleted(result);
        }
    }

    private static void logNotifications() {
        if (notifications == null) {
            AccuratLogger.log(AccuratLogger.STORAGE_DATA, "[]");
            return;
        }

        StringBuilder log = new StringBuilder("[");
        for (int i = 0; i < notifications.size(); i++) {
            String notificationLog = notifications.get(notifications.keyAt(i)).toString();
            if (log.length() + notificationLog.length() + 3 >= AccuratLogger.MAX_LOG_LENGTH) {
                break;
            }

            log.append(", ")
                    .append(notificationLog);
        }

        log.append("]");
        AccuratLogger.log(AccuratLogger.STORAGE_DATA, log.toString());
    }
    // </editor-fold>
}
