/*
 * Copyright (c) 2017 OpenLocate
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
package ai.accurat.sdk.core;

import android.content.ContentValues;
import android.database.Cursor;
import android.database.DatabaseUtils;
import android.database.sqlite.SQLiteDatabase;
import android.database.sqlite.SQLiteStatement;

import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Locale;

/**
 * @OpenLocate
 */
final class LocationTable {

    public static final SimpleDateFormat TIME_FORMAT = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss.SSS", Locale.getDefault());

    private static final String TAG = "LocationTable";
    private static final String TABLE_NAME = "location";

    private static final String COLUMN_ID = "_id";
    private static final String COLUMN_LOCATION = "location";
    private static final String QUERY_LIMIT = "1500";

    private static final int COLUMN_LOCATION_INDEX = 2;

    public static final String COLUMN_CREATED_AT = "created_at";

    private static final String CREATE_TABLE_SQL = "CREATE TABLE IF NOT EXISTS "
            + TABLE_NAME
            + " ("
            + COLUMN_ID + " INTEGER PRIMARY KEY AUTOINCREMENT, "
            + COLUMN_CREATED_AT + " INTEGER NOT NULL, "
            + COLUMN_LOCATION + " TEXT NOT NULL"
            + ");";

    private static final String CREATE_INDEX_SQL = "CREATE INDEX IF NOT EXISTS `"
            + COLUMN_CREATED_AT + "_index`" + "ON `" + TABLE_NAME
            + "` (`" + COLUMN_CREATED_AT + "` ASC);";

    private static final String DROP_TABLE_SQL = "DROP TABLE IF EXISTS ";
    private static final String BULK_INSERT_LOCATION = "INSERT INTO "
            + TABLE_NAME
            + " (" + COLUMN_LOCATION + ", " + COLUMN_CREATED_AT + ")"
            + " VALUES (?, ?);";

    static void onOpen(SQLiteDatabase db) {
        db.enableWriteAheadLogging();
        db.execSQL(CREATE_TABLE_SQL);
        db.execSQL(CREATE_INDEX_SQL);
    }

    static void createIfRequired(SQLiteDatabase db) {
        db.execSQL(CREATE_TABLE_SQL);
        db.execSQL(CREATE_INDEX_SQL);
    }

    static void upgrade(SQLiteDatabase db, int oldVersion, int newVersion) {
        AccuratLogger.log(AccuratLogger.DATABASE, "Upgrading database from version " + oldVersion + " to " + newVersion);
        db.execSQL(DROP_TABLE_SQL + TABLE_NAME);
        createIfRequired(db);
    }

    synchronized static void add(SQLiteDatabase database, LocationInterface location) {
        if (location == null) {
            AccuratLogger.log(AccuratLogger.DATABASE, "Adding null location to database");

            return;
        }
        AccuratLogger.log(AccuratLogger.DATABASE, "Adding a single location to database");

        if (database == null) {
            AccuratLogger.log(AccuratLogger.WARNING, "Can't add location to database, database is null");

            return;
        }

        ContentValues values = new ContentValues();
        values.put(COLUMN_CREATED_AT, location.getCreated().getTime());
        values.put(COLUMN_LOCATION, location.getJson().toString());
        long insertedRows = database.insert(TABLE_NAME, null, values);
        if (insertedRows > 0) {
            AccuratLogger.log(AccuratLogger.DATABASE, "Added location to database: " + location.getLocationInfo() + "[Creation at " + TIME_FORMAT.format(location.getCreated()) + " (" + location.getCreated().getTime() + ")]");
        } else {
            AccuratLogger.log(AccuratLogger.ERROR, "Failed to add location to database: " + location.getLocationInfo() + "[Creation at " + TIME_FORMAT.format(location.getCreated()) + " (" + location.getCreated().getTime() + ")]");
        }
    }

    synchronized static void addAll(SQLiteDatabase database, List<LocationInterface> locations) {
        if (locations == null || locations.isEmpty()) {
            AccuratLogger.log(AccuratLogger.DATABASE, "Adding 0 locations to database");

            return;
        }
        AccuratLogger.log(AccuratLogger.DATABASE, "Adding " + locations.size() + " locations to database");

        if (database == null) {
            AccuratLogger.log(AccuratLogger.WARNING, "Can't add locations to database, database is NULL");
            return;
        }

        try (SQLiteStatement statement = database.compileStatement(BULK_INSERT_LOCATION)) {
            database.beginTransaction();
            for (LocationInterface location : locations) {
                statement.clearBindings();
                statement.bindString(COLUMN_LOCATION_INDEX, location.getJson().toString());
                statement.bindLong(2, location.getCreated().getTime());
                statement.execute();
            }

            database.setTransactionSuccessful();
            AccuratLogger.log(AccuratLogger.DATABASE, "Successfully added locations to database");
        } finally {
            database.endTransaction();
            AccuratLogger.log(AccuratLogger.WARNING, "Failed to add locations to database");
        }
    }

    synchronized static long size(SQLiteDatabase database) {
        if (database == null) {
            return 0;
        }

        return DatabaseUtils.queryNumEntries(database, TABLE_NAME);
    }

    synchronized static List<LocationInterface> getSince(SQLiteDatabase database, long millisecondsSince1970) {
        AccuratLogger.log(AccuratLogger.DATABASE, "Getting locations since " + TIME_FORMAT.format(new Date(millisecondsSince1970)) + " (" + millisecondsSince1970 + ")");
        if (database == null) {
            AccuratLogger.log(AccuratLogger.WARNING, "Can't get locations, database is NULL");
            return null;
        }

        Cursor cursor = database.query(TABLE_NAME, null, LocationTable.COLUMN_CREATED_AT + " > " + millisecondsSince1970,
                null, null, null, LocationTable.COLUMN_CREATED_AT, QUERY_LIMIT);

        if (cursor == null || cursor.isClosed()) {
            AccuratLogger.log(AccuratLogger.ERROR, "Could not get locations, invalid cursor");

            return null;
        }


        AccuratLogger.log(AccuratLogger.DATABASE, logCount(database));
        AccuratLogger.log(AccuratLogger.DATABASE, cursor.getCount() + " locations since " + TIME_FORMAT.format(new Date(millisecondsSince1970)) + " (" + millisecondsSince1970 + ")");

        return getLocations(cursor);
    }

    synchronized static String logCount(SQLiteDatabase database) {
        if (database == null) {
            return "Could not count rows, database = null";
        }

        long rows = DatabaseUtils.queryNumEntries(database, TABLE_NAME);

        return "Database contains " + rows + " locations";
    }

    synchronized static void deleteBefore(SQLiteDatabase database, long millisecondsSince1970) {
        AccuratLogger.log(AccuratLogger.DATABASE, "Deleting locations before " + TIME_FORMAT.format(new Date(millisecondsSince1970)) + " (" + millisecondsSince1970 + ")");

        long numDeletions = database.delete(TABLE_NAME, LocationTable.COLUMN_CREATED_AT + " <= " + millisecondsSince1970, null);

        AccuratLogger.log(AccuratLogger.DATABASE, "Deleted " + numDeletions + " locations");
        AccuratLogger.log(AccuratLogger.DATABASE, logCount(database));
    }

    private static List<LocationInterface> getLocations(Cursor cursor) {
        List<LocationInterface> locations = null;

        if (cursor.moveToFirst()) {
            locations = new ArrayList<>();
            do {
                if (cursor.isClosed()) {
                    break;
                }

                locations.add(getLocationFromCursor(cursor));
            } while (cursor.moveToNext());
        }

        cursor.close();
        if (locations == null) {
            AccuratLogger.log(AccuratLogger.DATABASE, TAG + ".getLocations() received an empty database cursor");
        } else {
            AccuratLogger.log(AccuratLogger.DATABASE, TAG + ".getLocations() retrieved " + locations.size() + " locations");
        }

        return locations;
    }

    /**
     * Get location from cursor
     *
     * @param cursor Cursor with {@link #COLUMN_CREATED_AT} and {@link #COLUMN_LOCATION} as columns
     * @return The location parsed from the cursor.
     */
    private static LocationInterface getLocationFromCursor(Cursor cursor) {
        Date date = new Date(cursor.getLong(cursor.getColumnIndex(COLUMN_CREATED_AT)));
        String json = cursor.getString(cursor.getColumnIndex(COLUMN_LOCATION));
        return new AccuratLocationBuilder()
                .setDate(date)
                .setJsonRepresentation(json)
                .build();
    }

    /**
     * Get the most recent location in the database.
     *
     * @param db Readable database to get the latest recorded location from
     * @return Last known location.
     */
    synchronized static LocationInterface getLastKnownLocation(SQLiteDatabase db) {
        AccuratLogger.log(AccuratLogger.DATABASE, "Getting last known location from database");
        LocationInterface location = null;

        try (Cursor cursor = db.query(TABLE_NAME, new String[]{COLUMN_ID, COLUMN_LOCATION, COLUMN_CREATED_AT}, null, null, null, null, COLUMN_CREATED_AT + " DESC", "1")) {
            if (cursor.moveToFirst()) {
                location = getLocationFromCursor(cursor);
            }

            return location;
        }
    }
}
