package ai.accurat.sdk.core;

import android.annotation.TargetApi;
import android.os.Build;
import android.util.Range;

/**
 * Defines a search area.
 * This can be used for searching matching {@link AccuratGeofence} objects.
 */
@TargetApi(Build.VERSION_CODES.LOLLIPOP)
class SearchBox implements SearchBoxInterface {

    private Range<Double> latitudeRange;
    private Range<Double> longitudeRange;

    SearchBox(Double minLatitude, Double maxLatitude, Double minLongitude, Double maxLongitude) {
        this(new Range<>(minLatitude, maxLatitude), new Range<>(minLongitude, maxLongitude));
    }

    SearchBox(Range<Double> latitudeRange, Range<Double> longitudeRange) {
        this.latitudeRange = latitudeRange;
        this.longitudeRange = longitudeRange;
    }

    // <editor-fold desc="Checkers">
    /**
     * Determines whether a geofence is within this search box.
     *
     * @param geofence geofence to verify the location of.
     * @return True when within the search box, false if not.
     */
    @Override
    public boolean isWithinRange(AccuratGeofence geofence) {
        return geofence != null
                && this.latitudeRange.contains(geofence.getLatitude())
                && this.longitudeRange.contains(geofence.getLongitude());

    }

    public void halve() {
        double latitudeHeightChange = (latitudeRange.getUpper() - latitudeRange.getLower()) / 4;
        latitudeRange = new Range<>(latitudeRange.getLower() + latitudeHeightChange, latitudeRange.getUpper() - latitudeHeightChange);

        double longitudeWidthChange = (longitudeRange.getUpper() - longitudeRange.getLower()) / 4;
        longitudeRange = new Range<>(longitudeRange.getLower() + longitudeWidthChange, longitudeRange.getUpper() - longitudeWidthChange);
    }

    public boolean isSmallerThan(double smallestWidth, double smallestHeight) {
        double width = LocationUtils.distance(getMinLongitude(), getMinLatitude(), getMaxLongitude(), getMinLatitude());
        if(width <= smallestWidth) {
            return true;
        }

        double height = LocationUtils.distance(getMinLongitude(), getMinLatitude(), getMinLongitude(), getMaxLatitude());

        return height <= smallestHeight;
    }
    // </editor-fold>

    // <editor-fold desc="Getters">
    public double getMinLatitude() {
        return latitudeRange.getLower();
    }

    public double getMaxLatitude() {
        return latitudeRange.getUpper();
    }

    public double getMinLongitude() {
        return longitudeRange.getLower();
    }

    public double getMaxLongitude() {
        return longitudeRange.getUpper();
    }
    // </editor-fold>

    @Override
    public String toString() {
        return "[ " + latitudeRange.getLower() + ", " + longitudeRange.getLower() + " ] -> [ " + latitudeRange.getUpper() + ", " + longitudeRange.getUpper() + " ]";
    }
}
