package ai.accurat.sdk.data.models

import ai.accurat.sdk.config.Configuration
import ai.accurat.sdk.constants.ConsentState
import ai.accurat.sdk.core.AccuratLogger
import ai.accurat.sdk.data.enums.ConsentFlowState
import ai.accurat.sdk.data.enums.ConsentType
import ai.accurat.sdk.data.firstOr
import ai.accurat.sdk.data.save
import ai.accurat.sdk.data.stringify
import ai.accurat.sdk.data.toConsentTypeMap
import ai.accurat.sdk.managers.AccuratConfigurationManager
import ai.accurat.sdk.managers.RealmManager
import com.google.android.gms.ads.identifier.AdvertisingIdClient
import io.realm.RealmList
import io.realm.RealmObject
import io.realm.annotations.PrimaryKey

/**
 * @author Kenneth
 * @since 2020-11-10
 */
open class ConsentManagerState : RealmObject() {

    // <editor-fold desc="Fields">
    @PrimaryKey
    var id: Int = 1
    var forceAskConsents: Boolean = false
    var startRequested: Boolean = false
    var adId: String? = null
    var adIdTrackingAllowed: Boolean = false
    var consentModels: RealmList<ConsentModel> = RealmList()
    var shouldUploadUserConsents: Boolean = false
    var userConsents: RealmList<UserConsent> = RealmList()
    var lastConsentRequestTimestamp: Long = 0
    var configuration: AccuratConfiguration? = AccuratConfigurationManager.load()
    private var appConsentsString: String = ""
    private var consentFlowStateString: String = ConsentFlowState.NONE.name
    var consentFlowState: ConsentFlowState
        get() = ConsentFlowState.from(consentFlowStateString)
        set(value) {
            consentFlowStateString = value.name
        }
    // </editor-fold>

    // <editor-fold desc="Getters">
    fun isValidAskConsentsTiming(): Boolean = lastConsentRequestTimestamp < (System.currentTimeMillis() - Configuration.MIN_CONSENT_REQUEST_HIATUS_IN_MILLISECONDS)

    fun hasAppConsent(consentType: ConsentType): Boolean = appConsentsString.toConsentTypeMap()[consentType] == true

    fun getConsentState(consentType: ConsentType): ConsentState {
        val consent = userConsents.find(consentType)

        return when {
            consent == null -> ConsentState.UNKNOWN
            consent.accepted -> ConsentState.ACCEPTED
            else -> ConsentState.REFUSED
        }
    }
    // </editor-fold>

    fun setConsentModels(serverConsentModels: List<ConsentModel>) {
        this.consentModels = RealmList(*serverConsentModels.toTypedArray())
    }

    fun setUserConsents(userConsents: List<UserConsent>) {
        this.userConsents = RealmList(*userConsents.toTypedArray())
    }

    infix fun isAt(state: ConsentFlowState) {
        AccuratLogger.log(AccuratLogger.SDK_FLOW, "Changing consent flow state from $consentFlowState -> $state")
        consentFlowState = state
    }

    fun setAppConsent(consentType: ConsentType, value: Boolean) {
        AccuratLogger.log(AccuratLogger.METHOD_START, "ConsentManagerState.setAppConsent($consentType, $value)")
        val map = appConsentsString.toConsentTypeMap().toMutableMap()
        map[consentType] = value

        appConsentsString = map.stringify()
        AccuratLogger.log(AccuratLogger.STORAGE_DATA, "appConsentsString = $appConsentsString")
        save()
        AccuratLogger.log(AccuratLogger.METHOD_END, "ConsentManagerState.setAppConsent($consentType, $value)")
    }

    fun reset() {
        load()
        forceAskConsents = false
        startRequested = false
        userConsents = RealmList()
        lastConsentRequestTimestamp = 0
        appConsentsString = ""
        consentFlowState = ConsentFlowState.NONE
        save()
    }
    // </editor-fold>

    // <editor-fold desc="Helpers">
    fun check(type: ConsentType): Boolean {
        return userConsents.find(type)?.accepted ?: false
    }

    fun accept(vararg types: ConsentType) {
        types.forEach { type ->
            userConsents.find(type)?.let {
                it.accepted = true
                it.refuseCount = 0
                AccuratLogger.log(AccuratLogger.NONE, "Accepting $type: accepted => true, refuseCount => 0")
            } ?: userConsents.add(
                    UserConsent.create(type, true)
            )
        }
        save()
    }

    fun autoRefuse(vararg types: ConsentType) {
        types.forEach { type ->
            userConsents.find(type)?.let {
                it.accepted = false
                AccuratLogger.log(AccuratLogger.NONE, "Auto-refusing $type: accepted => false")
            } ?: userConsents.add(
                    UserConsent.create(type)
            )
        }
        save()
    }

    fun refuse(vararg types: ConsentType) {
        types.forEach { type ->
            userConsents.find(type)?.let {
                it.accepted = false
                it.refuseCount = (it.refuseCount + 1).coerceAtMost(maxRefuseCount(type))
                AccuratLogger.log(AccuratLogger.NONE, "Refusing $type: accepted => false, refuseCount => ${it.refuseCount}")
            } ?: userConsents.add(
                    UserConsent.create(type).apply {
                        refuseCount = 1
                    }
            )
        }
        save()
    }

    private fun refuseCount(type: ConsentType): Int {
        return userConsents.find(type)?.refuseCount ?: 0
    }

    private fun maxRefuseCount(type: ConsentType): Int {
        return consentModels.find(type)?.maxRefuseCount ?: if (type.isPermission) {
            Configuration.MAX_PERMISSION_REFUSE_COUNT
        } else {
            Configuration.MAX_CONSENT_REFUSE_COUNT
        }
    }

    fun isAccepted(type: ConsentType): Boolean {
        val accepted = userConsents.find(type)?.let {
            return it.accepted
        } ?: false
        AccuratLogger.log(AccuratLogger.NONE, "Is $type accepted? ${if (accepted) "yes" else "no"}")

        return accepted
    }

    fun mayAsk(type: ConsentType): Boolean {
        val mayAsk = forceAskConsents || refuseCount(type) < maxRefuseCount(type)
        val reason = when {
            forceAskConsents -> "forceAskConsents is true"
            refuseCount(type) < maxRefuseCount(type) -> "refuseCount < maxRefuseCount"
            else -> "forceAskConsents is false and refuseCount >= maxRefuseCount"
        }
        AccuratLogger.log(AccuratLogger.NONE, "mayAsk $type? $mayAsk because $reason")

        return mayAsk
    }
    // </editor-fold>

    override fun toString(): String {
        return "id = $id[br]" +
                "state = $consentFlowState[br]" +
                "forceAskConsent = $forceAskConsents[br]" +
                "startRequested = $startRequested[br]" +
                "adId = $adId (Tracking ${if (adIdTrackingAllowed) "allowed" else "disallowed"})[br]" +
                "lastConsentRequestTimestamp = $lastConsentRequestTimestamp[br]" +
                "configuration = $configuration[br]" +
                "consentModels = ${consentModels.joinToString(transform = ConsentModel::toString)}[br]" +
                "shouldUploadUserConsents = $shouldUploadUserConsents[br]" +
                "userConsents = ${userConsents.joinToString(transform = UserConsent::toString)}[br]" +
                "appConsents = $appConsentsString"
    }

    // <editor-fold desc="Objects">
    object RealmColumns {
        @JvmField
        val _TABLE: String = "ConsentManagerState"

        const val ID: String = "id"
        const val FORCE_ASK_CONSENTS: String = "forceAskConsents"
        const val START_REQUESTED: String = "startRequested"
        const val AD_ID: String = "adId"
        const val AD_ID_TRACKING_ALLOWED: String = "adIdTrackingAllowed"
        const val CONSENT_MODELS: String = "consentModels"
        const val USER_CONSENTS: String = "userConsents"
        const val LAST_CONSENT_REQUEST_TIMESTAMP: String = "lastConsentRequestTimestamp"
        const val CONFIGURATION: String = "configuration"
        const val APP_CONSENTS_STRING: String = "appConsentsString"
        const val CONSENT_FLOW_STATE_STRING: String = "consentFlowStateString"
        const val SHOULD_UPLOAD_USER_CONSENTS: String = "shouldUploadUserConsents"
    }

    companion object {
        private const val DEFAULT_ID: Int = 1

        fun load(): ConsentManagerState = RealmManager.getAccuratInstance()
                .firstOr(ConsentManagerState()) {
                    AccuratLogger.log(AccuratLogger.SDK_FLOW, "Fetching ConsentManagerState from Database")
                    it.equalTo(RealmColumns.ID, DEFAULT_ID)
                }

        private fun updateState(handler: (ConsentManagerState) -> Unit) {
            val state = load()
            handler(state)
            state.save()
        }

        fun getAdId(): String? = load().adId

        fun hasValidAdId(): Boolean {
            val state = load()

            return !state.adId.isNullOrBlank() && state.adIdTrackingAllowed
        }

        fun getAdInfo(): AdvertisingIdClient.Info {
            val state = load()

            return AdvertisingIdClient.Info(state.adId ?: "", !state.adIdTrackingAllowed)
        }

        fun isStartRequested(): Boolean = load().startRequested

        fun stopTracking() {
            val state = load()
            state.startRequested = false
            state.save()
        }

        fun setAppConsent(consentType: ConsentType, value: Boolean) {
            load().setAppConsent(consentType, value)
        }

        fun reset() {
            load().reset()
        }

        fun getConsentState(consentType: ConsentType): ConsentState {
            return load().getConsentState(consentType)
        }

        fun isConsentAccepted(consentType: ConsentType): Boolean {
            return ConsentState.ACCEPTED == getConsentState(consentType)
        }

        fun updateConsentRequestTimestamp() {
            updateState { state ->
                state.lastConsentRequestTimestamp = System.currentTimeMillis()
            }
        }

        fun isValidAskConsentsTiming(): Boolean {
            return load().isValidAskConsentsTiming()
        }

        fun canTrack(): Boolean {
            AccuratLogger.log(AccuratLogger.METHOD_START, "ConsentManagerState.canTrack()")
            val state = load()

            if (state.adId.isNullOrBlank() && !state.adIdTrackingAllowed || !state.startRequested) {
                AccuratLogger.log(AccuratLogger.NONE, "adId.isNullOrBlank() = ${state.adId.isNullOrBlank()}")
                AccuratLogger.log(AccuratLogger.NONE, "adIdTrackingAllowed = ${state.adIdTrackingAllowed}")
                AccuratLogger.log(AccuratLogger.NONE, "startRequested = ${state.startRequested}")
                AccuratLogger.log(AccuratLogger.METHOD_END, "ConsentManagerState.canTrack() = false")
                return false
            }

            val gdprAccepted = state.getConsentState(ConsentType.GDPR) == ConsentState.ACCEPTED
            val locationAccepted = state.getConsentState(ConsentType.IN_APP_LOCATION) == ConsentState.ACCEPTED
            AccuratLogger.log(AccuratLogger.NONE, "gdprAccepted = $gdprAccepted")
            AccuratLogger.log(AccuratLogger.NONE, "locationAccepted = $locationAccepted")

            val result = gdprAccepted && locationAccepted

            AccuratLogger.log(AccuratLogger.METHOD_END, "ConsentManagerState.canTrack() = $result")
            return result
        }
    }
    // </editor-fold>
}