package ai.accurat.sdk.data.models

import ai.accurat.sdk.core.AccuratGeofence
import ai.accurat.sdk.core.AccuratLogger
import ai.accurat.sdk.data.all
import ai.accurat.sdk.data.count
import ai.accurat.sdk.data.enums.GeofenceType
import ai.accurat.sdk.data.save
import ai.accurat.sdk.managers.RealmManager
import android.annotation.SuppressLint
import io.realm.RealmObject
import io.realm.annotations.PrimaryKey
import io.realm.kotlin.where
import java.text.SimpleDateFormat
import java.util.Date
import java.util.Locale
import java.util.TimeZone

/**
 * @author Kenneth
 * @since 2025-05-21
 */
open class TriggeredGeofence : RealmObject() {

    // <editor-fold desc="Fields">
    @PrimaryKey
    var id: String = ""
    var triggerType: String = ""
    var timestamp: String = ""
    var data: String = "{}"
    // </editor-fold>

    companion object {
        private const val MAX_BUFFERED_GEOFENCES: Long = 1500L

        @SuppressLint("ConstantLocale")
        private val timestampFormatter = SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss.SSS'Z'", Locale.getDefault()).apply {
            timeZone = TimeZone.getTimeZone("UTC")
        }

        fun fromAccuratGeofence(
            geofence: AccuratGeofence,
            trigger: GeofenceType,
            triggeredAt: Long = System.currentTimeMillis(),
        ): TriggeredGeofence = TriggeredGeofence().apply {
            id = geofence.id
            triggerType = trigger.name
            timestamp = timestampFormatter.format(Date(triggeredAt))
            data = geofence.data ?: "{}"
        }

        fun storeTriggeredGeofence(
            geofence: AccuratGeofence,
            trigger: GeofenceType,
            triggeredAt: Long = System.currentTimeMillis(),
        ) {
            val triggeredGeofence = fromAccuratGeofence(
                geofence,
                trigger,
                triggeredAt
            )
            AccuratLogger.log(
                AccuratLogger.GEOFENCE,
                "Saving triggered geofence in database"
            )
            AccuratLogger.log(
                AccuratLogger.DATABASE,
                triggeredGeofence.toString(),
            )
            // Save the triggered geofence
            triggeredGeofence.save()

            // Check if the buffer is full
            val count = TriggeredGeofence().count<TriggeredGeofence>()
            AccuratLogger.log(
                AccuratLogger.DATABASE,
                "There are now $count triggered geofences in the database (Max allowed: $MAX_BUFFERED_GEOFENCES)",
            )
            if (count <= MAX_BUFFERED_GEOFENCES) {
                return
            }

            // Remove the oldest geofences from the buffer
            AccuratLogger.log(
                AccuratLogger.DATABASE,
                "$count > $MAX_BUFFERED_GEOFENCES, removing oldest ${count - MAX_BUFFERED_GEOFENCES} geofences from the database",
            )
            val realm = RealmManager.getAccuratInstance()
            realm.executeTransactionAsync {
                val isDeleted = realm.where(TriggeredGeofence::class.java)
                    .sort(RealmColumns.TIMESTAMP)
                    .limit(count - MAX_BUFFERED_GEOFENCES)
                    .findAll()
                    .deleteAllFromRealm()
                if (isDeleted) {
                    AccuratLogger.log(
                        AccuratLogger.DATABASE,
                        "Deleted ${count - MAX_BUFFERED_GEOFENCES} triggered geofences from the database",
                    )
                } else {
                    AccuratLogger.log(
                        AccuratLogger.WARNING,
                        "Failed to deleted ${count - MAX_BUFFERED_GEOFENCES} triggered geofences from the database",
                    )
                }
            }
            realm.close()
        }

        fun getAll(deleteReturnedItems: Boolean = false): List<TriggeredGeofence> {
            val geofences = TriggeredGeofence().all<TriggeredGeofence>()
            if (deleteReturnedItems) {
                val realm = RealmManager.getAccuratInstance()
                realm.executeTransaction {
                    realm.where<TriggeredGeofence>()
                        .`in`(
                            RealmColumns.ID,
                            geofences
                                .map(TriggeredGeofence::id)
                                .toTypedArray()
                        )
                        .findAll()
                        .deleteAllFromRealm()
                }
                realm.close()
            }

            return geofences
        }
    }

    override fun toString(): String {
        return "{\"id\":\"$id\",\"triggerType\":\"$triggerType\",\"timestamp\":\"$timestamp\",\"data\":\"$data\"}"
    }

    object RealmColumns {
        const val TABLE: String = "TriggeredGeofence"

        const val ID: String = "id"
        const val TRIGGER_TYPE: String = "triggerType"
        const val TIMESTAMP: String = "timestamp"
        const val DATA: String = "data"
    }
}